#!/usr/bin/env python3
"""
Reads and plots active (i.e. non-zero) signals from SIEMENS advanced physiological log / DICOM files

This function expects to find either a combination of individual logfiles (*_ECG.log, *_RESP.log,
*_PULS.log, *_EXT.log, *_Info.log) generated by >=R013 sequences, or a single encoded "_PHYSIO" DICOM
file generated by >=R015 sequences.
"""

import physio as ph
import logging, coloredlogs

# Set-up logging
LOGGER = logging.getLogger(__name__)
if not LOGGER.handlers:
    coloredlogs.install(fmt='%(asctime)s - %(levelname)s %(message)s', datefmt='%Y-%m-%d %H:%M:%S')


def main():
    """Console script usage"""

    # Parse the input arguments and run physio2tsv(args)
    import argparse

    parser = argparse.ArgumentParser(formatter_class=argparse.RawDescriptionHelpFormatter,
                                     description=__doc__,
                                     epilog='examples:\n'
                                            '  physio2tsv /project/3022026.01/sub-001/MR000000.dcm /project/3022026.01/bids/sub-001/func/sub-001_physio\n'
                                            '  physio2tsv /project/3022026.01/sub-001/Physio_20200428_142451_007e910e-02d9-4d7a-8fdb-8e3568be8322 /project/3022026.01/bids/sub-001/func/sub-001_physio\n\n'
                                            '@author: Marcel Zwiers\n ')
    parser.add_argument('physiofile', help="Either the fullpath of the DICOM file or the basename of the PHYSIO logfiles (fullpath without suffix and file extension, e.g. 'foo/bar/Physio_DATE_TIME_UUID'")
    parser.add_argument('tsvfile',    help="The fullpath of the BIDS filenames, e.g. 'foo/bids/sub-001/func/sub-001_physio'")
    args = parser.parse_args()

    physio = ph.readphysio(args.physiofile)
    ph.physio2tsv(physio, args.tsvfile)


if __name__ == "__main__":
    main()
