import collections
import copy
import datetime
import functools
import json
import logging
import inspect
import itertools
import operator
import os.path
import pickle
import re
import sys
import types
import typing
from urllib import parse as url_parse

import frozendict  # type: ignore
import jsonschema  # type: ignore
import numpy  # type: ignore
import pandas  # type: ignore
from pytypes import type_util  # type: ignore

import d3m
from . import hyperparams as hyperparams_module, primitive_names
from d3m import deprecate, exceptions, utils

# See: https://gitlab.com/datadrivendiscovery/d3m/issues/66
try:
    from pyarrow import lib as pyarrow_lib  # type: ignore
except ModuleNotFoundError:
    pyarrow_lib = None

__all__ = (
    'ALL_ELEMENTS', 'NO_VALUE', 'DataMetadata', 'PrimitiveMetadata', 'CONTAINER_SCHEMA_VERSION',
    'DATA_SCHEMA_VERSION', 'PRIMITIVE_SCHEMA_VERSION', 'PrimitiveMethodKind',
    'PrimitiveArgumentKind', 'PrimitiveInstallationType', 'PrimitiveAlgorithmType',
    'PrimitiveFamily', 'PrimitivePrecondition', 'PrimitiveEffect', 'ForeignKeyType', 'Context',
    'PipelineRunPhase', 'PipelineStepType', 'PipelineRunStatusState', 'ArgumentType',
)

logger = logging.getLogger(__name__)


def _return_all_elements() -> 'ALL_ELEMENTS_TYPE':
    return ALL_ELEMENTS


@functools.total_ordering
class ALL_ELEMENTS_TYPE:
    __slots__ = ()

    def __repr__(self) -> str:
        return '__ALL_ELEMENTS__'

    def __lt__(self, other: typing.Any) -> bool:
        # "ALL_ELEMENTS" is smaller than anything else, and equal to itself.
        # "ALL_ELEMENTS" is a singleton, so is equal only if referentially equal
        # (which is a default implementation of "__eq__").
        return self != other

    def __deepcopy__(self, memo: typing.Dict) -> 'ALL_ELEMENTS_TYPE':
        return ALL_ELEMENTS

    def __copy__(self) -> 'ALL_ELEMENTS_TYPE':
        return ALL_ELEMENTS

    def __reduce__(self) -> typing.Tuple[typing.Callable, typing.Tuple]:
        return _return_all_elements, ()


def _return_no_value() -> 'NO_VALUE_TYPE':
    return NO_VALUE


class NO_VALUE_TYPE:
    __slots__ = ()

    def __repr__(self) -> str:
        return '__NO_VALUE__'

    def __deepcopy__(self, memo: typing.Dict) -> 'NO_VALUE_TYPE':
        return NO_VALUE

    def __copy__(self) -> 'NO_VALUE_TYPE':
        return NO_VALUE

    def __reduce__(self) -> typing.Tuple[typing.Callable, typing.Tuple]:
        return _return_no_value, ()


ALL_ELEMENTS = ALL_ELEMENTS_TYPE()
NO_VALUE = NO_VALUE_TYPE()

COMMIT_HASH_REGEX = re.compile(r'^[0-9a-f]{40}$')

ARGUMENT_NAME_REGEX = re.compile(r'^[A-Za-z][A-Za-z_0-9]*$')

CONTAINER_SCHEMA_VERSION = 'https://metadata.datadrivendiscovery.org/schemas/v0/container.json'
DATA_SCHEMA_VERSION = 'https://metadata.datadrivendiscovery.org/schemas/v0/data.json'
PRIMITIVE_SCHEMA_VERSION = 'https://metadata.datadrivendiscovery.org/schemas/v0/primitive.json'

SCHEMAS_PATH = os.path.join(os.path.dirname(__file__), 'schemas', 'v0')

# A map of all known schemas from their URIs to loaded JSONs. Not validated.
SCHEMAS = {}
for schema_uri in [
    CONTAINER_SCHEMA_VERSION,
    DATA_SCHEMA_VERSION,
    'https://metadata.datadrivendiscovery.org/schemas/v0/definitions.json',
    'https://metadata.datadrivendiscovery.org/schemas/v0/pipeline.json',
    'https://metadata.datadrivendiscovery.org/schemas/v0/pipeline_run.json',
    PRIMITIVE_SCHEMA_VERSION,
    'https://metadata.datadrivendiscovery.org/schemas/v0/problem.json',
]:
    schema_filename = os.path.basename(schema_uri)
    with open(os.path.join(SCHEMAS_PATH, schema_filename), 'r', encoding='utf8') as schema_file:
        SCHEMAS[schema_uri] = json.load(schema_file)

# We validate schemas using unmodified validator.
for schema_json in SCHEMAS.values():
    jsonschema.Draft7Validator.check_schema(schema_json)

DEFINITIONS_JSON = SCHEMAS['https://metadata.datadrivendiscovery.org/schemas/v0/definitions.json']

CONTAINER_SCHEMA_VALIDATOR, DATA_SCHEMA_VALIDATOR, PRIMITIVE_SCHEMA_VALIDATOR = utils.load_schema_validators(SCHEMAS, ('container.json', 'data.json', 'primitive.json'))

HYPERPARAMETER_REQUIRED_SEMANTIC_TYPES = {
    'https://metadata.datadrivendiscovery.org/types/TuningParameter',
    'https://metadata.datadrivendiscovery.org/types/ControlParameter',
    'https://metadata.datadrivendiscovery.org/types/ResourcesUseParameter',
    'https://metadata.datadrivendiscovery.org/types/MetafeatureParameter',
}

TABULAR_SEMANTIC_TYPES = {
    'https://metadata.datadrivendiscovery.org/types/Table',
    'https://metadata.datadrivendiscovery.org/types/TabularRow',
    'https://metadata.datadrivendiscovery.org/types/TabularColumn',
}

ALL_SEMANTIC_TYPES = set(utils._get_names(DEFINITIONS_JSON, 'definitions.semantic_types.items.anyOf[*].enum[*]'))

# A list of all fields which is being generated by "_generate_metadata" method.
ALL_GENERATED_FIELDS = [
    'schema',
    'structural_type',
    'semantic_types',
    'dimension',
    'name',
]

PrimitiveMethodKind = utils.create_enum_from_json_schema_enum(
    'PrimitiveMethodKind', DEFINITIONS_JSON,
    'definitions.primitive_code.properties.instance_methods.additionalProperties.properties.kind.oneOf[*].enum[*]',
    module=__name__,
)
PrimitiveArgumentKind = utils.create_enum_from_json_schema_enum(
    'PrimitiveArgumentKind', DEFINITIONS_JSON,
    'definitions.primitive_code.properties.arguments.additionalProperties.properties.kind.oneOf[*].enum[*]',
    module=__name__,
)
PrimitiveInstallationType = utils.create_enum_from_json_schema_enum(
    'PrimitiveInstallationType', DEFINITIONS_JSON,
    [
        'definitions.installation.items.oneOf[*].properties.type.enum[*]',
        'definitions.installation.items.oneOf[*].allOf[*].properties.type.enum[*]'
    ],
    module=__name__,
)
PrimitiveAlgorithmType = utils.create_enum_from_json_schema_enum(
    'PrimitiveAlgorithmType', DEFINITIONS_JSON,
    'definitions.algorithm_types.items.oneOf[*].enum[*]',
    module=__name__,
)
PrimitiveFamily = utils.create_enum_from_json_schema_enum(
    'PrimitiveFamily', DEFINITIONS_JSON,
    'definitions.primitive_family.oneOf[*].enum[*]',
    module=__name__,
)
PrimitivePrecondition = utils.create_enum_from_json_schema_enum(
    'PrimitivePrecondition', DEFINITIONS_JSON,
    'definitions.preconditions.items.oneOf[*].enum[*]',
    module=__name__,
)
PrimitiveEffect = utils.create_enum_from_json_schema_enum(
    'PrimitiveEffect', DEFINITIONS_JSON,
    'definitions.effects.items.oneOf[*].enum[*]',
    module=__name__,
)
ForeignKeyType = utils.create_enum_from_json_schema_enum(
    'ForeignKeyType', DEFINITIONS_JSON,
    'definitions.foreign_key.oneOf[*].properties.type.enum[*]',
    module=__name__,
)
Context = utils.create_enum_from_json_schema_enum(
    'Context', DEFINITIONS_JSON,
    'definitions.context.oneOf[*].enum[*]',
    module=__name__,
)
PipelineRunPhase = utils.create_enum_from_json_schema_enum(
    'PipelineRunPhase', DEFINITIONS_JSON,
    'definitions.pipeline_run.properties.phase.anyOf[*].enum[*]',
    module=__name__,
)
PipelineStepType = utils.create_enum_from_json_schema_enum(
    'PipelineStepType', DEFINITIONS_JSON,
    'definitions.pipeline_steps.items.oneOf[*].properties.type.enum[*]',
    module=__name__,
)
PipelineRunStatusState = utils.create_enum_from_json_schema_enum(
    'StatusState', DEFINITIONS_JSON,
    'definitions.status.properties.state.enum[*]',
    module=__name__,
)
# Enumeration of argument and hyper-parameter types to a primitive in a step.
ArgumentType = utils.create_enum_from_json_schema_enum(
    'ArgumentType', DEFINITIONS_JSON,
    'definitions[container_argument,container_arguments,primitive_argument,primitive_arguments,data_argument,data_arguments,value_argument].properties.type.enum[*]',
    module=__name__,
)

M = typing.TypeVar('M', bound='MetadataEntry')
T = typing.TypeVar('T', bound='Metadata')
D = typing.TypeVar('D', bound='DataMetadata')
P = typing.TypeVar('P', bound='PrimitiveMetadata')
SimpleSelectorSegment = typing.Union[int, str]
SelectorSegment = typing.Union[SimpleSelectorSegment, ALL_ELEMENTS_TYPE]
ListSelector = typing.List[SelectorSegment]
TupleSelector = typing.Tuple[SelectorSegment, ...]
# A list or tuple of integers, strings, or ALL_ELEMENTS.
Selector = typing.Union[ListSelector, TupleSelector]

# We register additional immutable values. We are doing it this way to overcome issues with import cycles.
if ALL_ELEMENTS not in utils.additional_immutable_values:
    utils.additional_immutable_values += (ALL_ELEMENTS,)
if NO_VALUE not in utils.additional_immutable_values:
    utils.additional_immutable_values += (NO_VALUE,)


class ColumnReference(typing.NamedTuple):
    resource_id: str
    column_index: int


class MetadataEntry:
    __slots__ = ('elements', 'all_elements', 'metadata', 'is_empty', 'is_elements_empty')

    def __init__(
        self, elements: utils.PMap = utils.EMPTY_PMAP, all_elements: 'MetadataEntry' = None,
        metadata: frozendict.FrozenOrderedDict = frozendict.FrozenOrderedDict(), is_empty: bool = True,
        is_elements_empty: bool = True,
    ) -> None:
        self.elements = elements
        self.all_elements = all_elements
        self.metadata = metadata
        self.is_empty = is_empty
        self.is_elements_empty = is_elements_empty

    def copy(self: M) -> M:
        return type(self)(self.elements, self.all_elements, self.metadata, self.is_empty, self.is_elements_empty)

    def __copy__(self: M) -> M:
        return self.copy()

    def update_is_empty(self) -> None:
        self.is_empty = not self.metadata and self.is_elements_empty and self.all_elements is None


class Metadata:
    """
    A basic class to be used as a value for `metadata` attribute
    on values passed between primitives.

    Instances are immutable.

    Parameters
    ----------
    metadata:
        Optional initial metadata for the top-level of the value.
    source:
        DEPRECATED: argument ignored.
    timestamp:
        DEPRECATED: argument ignored.
    """

    @deprecate.arguments('source', 'timestamp', message="argument ignored")
    def __init__(self, metadata: typing.Dict[str, typing.Any] = None, *, source: typing.Any = None, timestamp: datetime.datetime = None) -> None:
        self._current_metadata = MetadataEntry()

        self._hash: int = None

        if metadata is not None:
            self._update_in_place((), metadata, self._current_metadata)

    @deprecate.arguments('source', 'timestamp', message="argument ignored")
    def update(self: T, selector: Selector, metadata: typing.Dict[str, typing.Any], *, source: typing.Any = None, timestamp: datetime.datetime = None) -> T:
        """
        Updates metadata with new ``metadata`` for data pointed to with ``selector``.

        If value of any field is ``NO_VALUE``, that field is deleted.

        It returns a copy of this metadata object with new metadata applied.

        Parameters
        ----------
        selector:
            A selector pointing to data.
        metadata:
            A map of fields and values with metadata.
        source:
            DEPRECATED: argument ignored.
        timestamp:
            DEPRECATED: argument ignored.

        Returns
        -------
        Updated metadata.
        """

        cls = type(self)

        new_metadata = cls()

        new_metadata._update_in_place(selector, metadata, self._current_metadata)

        return new_metadata

    @deprecate.arguments('source', 'timestamp', message="argument ignored")
    def remove(self: T, selector: Selector, *, recursive: bool = False, strict_all_elements: bool = False,
               source: typing.Any = None, timestamp: datetime.datetime = None) -> T:
        """
        Removes all metadata at ``selector``.

        Parameters
        ----------
        selector:
            A selector to remove metadata at.
        recursive:
            Should remove also all metadata under the ``selector``?
        strict_all_elements:
            If ``True``, then when removing ``ALL_ELEMENTS`` entry, do not remove also metadata for all elements it matches.
        source:
            DEPRECATED: argument ignored.
        timestamp:
            DEPRECATED: argument ignored.

        Returns
        -------
        Updated metadata.
        """

        cls = type(self)

        new_metadata = cls()

        new_metadata._remove_in_place(selector, recursive, strict_all_elements, self._current_metadata)

        return new_metadata

    @deprecate.function(message="create a DataMetadata instance explicitly instead")
    @deprecate.arguments('source', 'timestamp', message="argument ignored")
    def clear(self: T, metadata: typing.Dict[str, typing.Any] = None, *, source: typing.Any = None, timestamp: datetime.datetime = None) -> T:
        """
        DEPRECATED: create a Metadata instance explicitly instead.

        Creates and returns a new (clear) metadata object.

        Parameters
        ----------
        metadata:
            Optional new initial metadata for the top-level of the value.
        source:
            DEPRECATED: argument ignored.
        timestamp:
            DEPRECATED: argument ignored.

        Returns
        -------
        New metadata object.
        """

        return type(self)(metadata)

    def _update_in_place(self, selector: Selector, metadata: typing.Dict[str, typing.Any],
                         parent_current_metadata: MetadataEntry) -> None:
        """
        This method exist only for internal purposes and you should never ever call this to update metadata from outside.
        """

        self.check_selector(selector)

        # If metadata is already an instance of frozen dict, we just check that it is immutable.
        if isinstance(metadata, frozendict.FrozenOrderedDict):
            utils.check_immutable(metadata)
        else:
            metadata = utils.make_immutable_copy(metadata)

            if not isinstance(metadata, frozendict.FrozenOrderedDict):
                raise exceptions.InvalidArgumentTypeError("Metadata should be a dict.")

        self._current_metadata = self._update(selector, parent_current_metadata, metadata)

    def _remove_in_place(self, selector: Selector, recursive: bool, strict_all_elements: bool,
                         parent_current_metadata: MetadataEntry) -> None:
        """
        This method exist only for internal purposes and you should never ever call this to remove metadata from outside.
        """

        self.check_selector(selector)

        self._current_metadata = self._remove(selector, recursive, strict_all_elements, parent_current_metadata)

    # TODO: Allow querying only a subset of metadata (not the whole dict).
    # TODO: Maybe cache results? LRU?
    def query(self, selector: Selector, *, ignore_all_elements: bool = False, remove_no_value: bool = True) -> frozendict.FrozenOrderedDict:
        """
        Returns metadata for data pointed to with ``selector``.

        When querying using ``ALL_ELEMENTS`` means only metadata which has been set using ALL_ELEMENTS
        is returned.

        Parameters
        ----------
        selector:
            A selector to query metadata for.
        ignore_all_elements:
            By default, metadata from ALL_ELEMENTS is merged with metadata for an element itself.
            By setting this argument to ``True``, this is disabled and just metadata from an element is returned.
        remove_no_value:
            By default all ``NO_VALUE`` values are removed. If set to ``False``, they are not removed.

        Returns
        -------
        Metadata at a given selector.
        """

        self.check_selector(selector)

        metadata = self._query(selector, self._current_metadata, 0 if ignore_all_elements else None)

        if remove_no_value:
            return self._remove_no_value(metadata)
        else:
            return metadata

    def query_with_exceptions(self, selector: Selector, *, remove_no_value: bool = True) -> typing.Tuple[frozendict.FrozenOrderedDict, typing.Dict[TupleSelector, frozendict.FrozenOrderedDict]]:
        """
        In addition to returning metadata for data pointed to with ``selector``, this method for every ``ALL_ELEMENTS``
        selector segment also returns a map between selectors and metadata for all elements which have metadata
        which differs from that of ``ALL_ELEMENTS``.

        Parameters
        ----------
        selector:
            A selector to query metadata for.
        remove_no_value:
            By default all ``NO_VALUE`` values are removed. If set to ``False``, they are not removed.

        Returns
        -------
        A tuple of metadata at a given selector and a dict of exceptions.
        """

        self.check_selector(selector)

        metadata = self._query(selector, self._current_metadata, None)
        if remove_no_value:
            metadata = self._remove_no_value(metadata)

        exceptions = self._query_exceptions(selector, self._current_metadata)

        exceptions_with_selectors = {}
        for exception_selector in exceptions:
            exception_metadata = self._query(exception_selector, self._current_metadata, None)
            if remove_no_value:
                exception_metadata = self._remove_no_value(exception_metadata)

            if exception_metadata and exception_metadata != metadata:
                exceptions_with_selectors[exception_selector] = exception_metadata

        return metadata, exceptions_with_selectors

    def query_field(self, selector: Selector, field: str, *, strict_all_elements: bool = True) -> typing.Any:
        """
        Queries metadata for data pointed to with ``selector`` and returns only the
        ``field`` of that metadata. Raises `KeyError` exception if metadata or field
        is not set.

        ``field`` represents only top-level fields in metadata.

        Parameters
        ----------
        selector:
            A selector to query metadata for.
        field:
            A field name to query.
        strict_all_elements:
            If set, the method does not just return ``field`` value of the metadata
            under ``selector``, but checks that the value really holds for all
            elements matching the ``selector``, without exception. This is helpful
            also if metadata is not compacted and ``field`` value is the same
            across all elements, but ``ALL_ELEMENTS`` metadata does not contain
            that field.

        Returns
        -------
        A value of ``field`` of metadata at ``selector``.
        """

        if not strict_all_elements:
            return self.query(selector)[field]

        metadata, exceptions_with_selectors = self.query_with_exceptions(selector)

        # We have a candidate which potentially holds for all elements.
        if field in metadata:
            value = metadata[field]

            for exception_metadata in exceptions_with_selectors.values():
                # Is there an exception for this field? We care only if field exists,
                # then it has to match in the value. But if field does not exist,
                # value from "metadata" will be used anyway, so that is OK.
                if field in exception_metadata and exception_metadata[field] != value:
                    raise KeyError("Field '{field}' is not the same across all elements.".format(field=field))

            return value

        # If selector is without "ALL_ELEMENTS" then field is simply not set.
        if ALL_ELEMENTS not in selector:
            assert not exceptions_with_selectors
            raise KeyError("Field '{field}' is not set.".format(field=field))

        # Field might be set on all elements, but metadata is no compacted,
        # check if field is the same across all metadata exceptions.
        # TODO: Check that metadata exceptions cover whole dimension.
        #       When field is not set for ALL_ELEMENTS, we have to traverse all potential elements,
        #       not just those which have metadata set, but any which could have it set. We can do
        #       that if dimension length is set, we can enumerate all elements and check that they
        #       contain equal field value. But easier it is to just check that dimension length
        #       matches the number of metadata exceptions. Then we know we have checked all elements
        #       which can exist on data. And if any element is missing (does not have metadata set),
        #       it does not have field set anyway, which means it does not match field value of other
        #       elements. This dimension length comparison can work even in the case when dimension
        #       is not enumerable (e.g., a dict). Checking dimension lengths becomes tricky when
        #       multiple ALL_ELEMENTS are present in the selector though, and especially if data
        #       is jagged (does not have same size sub-dimensions for all elements). An issue is
        #       also that dimensions defined for DataMetadata and not Metadata.

        # Can raise KeyError.
        first_exception_selector, first_exception_metadata = exceptions_with_selectors.popitem()

        # Can raise KeyError.
        value = first_exception_metadata[field]

        for exception_metadata in exceptions_with_selectors.values():
            # We require that "field" both exist in all exception metadata and has the same value
            # as all other fields (which we check by checking against the first exception metadata).
            if field not in exception_metadata or exception_metadata[field] != value:
                raise KeyError("Field '{field}' is not the same across all elements.".format(field=field))

        return value

    def query_field_with_exceptions(self, selector: Selector, field: str) -> typing.Tuple[typing.Any, typing.Dict[TupleSelector, typing.Any]]:
        """
        In addition to returning ``field`` of metadata for data pointed to with ``selector``,
        this method for every ``ALL_ELEMENTS`` selector segment also returns a map between
        selectors and field values for all elements which have field which differs from that
        of ``ALL_ELEMENTS``.

        If ``field`` does not exist under ``selector``, ``NO_VALUE`` is returned instead,
        and all exceptions are required to contain ``field``.

        ``field`` represents only top-level fields in metadata.

        Parameters
        ----------
        selector:
            A selector to query metadata for.
        field:
            A field name to query.

        Returns
        -------
        A tuple of value at a given selector and field and a dict of exceptions.
        """

        metadata, exceptions_with_selectors = self.query_with_exceptions(selector)

        if field in metadata:
            # If "field" exist in "metadata", we return only those exceptions which contain "field" which
            # differs from that in "metadata". Only they are real "exceptions" for this "selector" and "field".
            return metadata[field], {
                exception_selector: exception_metadata[field] for exception_selector, exception_metadata in exceptions_with_selectors.items()
                if field in exception_metadata and exception_metadata[field] != metadata[field]
            }

        # If selector is without "ALL_ELEMENTS" then field is simply not set.
        if ALL_ELEMENTS not in selector:
            assert not exceptions_with_selectors
            raise KeyError("Field '{field}' is not set.".format(field=field))

        field_exceptions = {}

        for exception_selector, exception_metadata in exceptions_with_selectors.items():
            if field not in exception_metadata:
                raise KeyError("Field '{field}' is not set.".format(field=field))

            field_exceptions[exception_selector] = exception_metadata[field]

        return NO_VALUE, field_exceptions

    def _query(self, selector: Selector, metadata_entry: typing.Optional[MetadataEntry], ignore_all_elements: typing.Optional[int]) -> frozendict.FrozenOrderedDict:
        if metadata_entry is None:
            return frozendict.FrozenOrderedDict()
        if len(selector) == 0:
            return metadata_entry.metadata

        segment, selector_rest = selector[0], selector[1:]

        if ignore_all_elements is not None:
            new_ignore_all_elements = ignore_all_elements - 1
        else:
            new_ignore_all_elements = None

        all_elements_metadata = self._query(selector_rest, metadata_entry.all_elements, new_ignore_all_elements)
        if segment is ALL_ELEMENTS:
            metadata = all_elements_metadata
        elif segment in metadata_entry.elements:
            segment = typing.cast(SimpleSelectorSegment, segment)
            metadata = self._query(selector_rest, metadata_entry.elements[segment], new_ignore_all_elements)
            if ignore_all_elements is None or ignore_all_elements > 0:
                metadata = self._merge_metadata(all_elements_metadata, metadata)
        elif ignore_all_elements is not None and ignore_all_elements <= 0:
            metadata = frozendict.FrozenOrderedDict()
        else:
            metadata = all_elements_metadata

        return metadata

    def _query_exceptions(self, selector: Selector, metadata_entry: typing.Optional[MetadataEntry]) -> typing.Sequence[TupleSelector]:
        if metadata_entry is None:
            return []
        if len(selector) == 0:
            return []

        segment, selector_rest = selector[0], selector[1:]

        exceptions: typing.List[TupleSelector] = []
        if segment is ALL_ELEMENTS:
            if selector_rest:
                for exception_selector in self._query_exceptions(selector_rest, metadata_entry.all_elements):
                    exceptions.append((segment,) + exception_selector)

            for element_segment, element_metadata_entry in metadata_entry.elements.items():
                if selector_rest:
                    for exception_selector in self._query_exceptions(selector_rest, element_metadata_entry):
                        exceptions.append((typing.cast(SelectorSegment, element_segment),) + exception_selector)
                else:
                    if element_metadata_entry.metadata:
                        exceptions.append((element_segment,))
        elif segment in metadata_entry.elements:
            element_metadata_entry = metadata_entry.elements[typing.cast(SimpleSelectorSegment, segment)]
            if selector_rest:
                for exception_selector in self._query_exceptions(selector_rest, element_metadata_entry):
                    exceptions.append((segment,) + exception_selector)
            elif element_metadata_entry.metadata:
                exceptions.append((segment,))

        return exceptions

    def _remove(self, selector: Selector, recursive: bool, strict_all_elements: bool,
                metadata_entry: typing.Optional[MetadataEntry]) -> MetadataEntry:
        if metadata_entry is None:
            new_metadata_entry = MetadataEntry()
        else:
            new_metadata_entry = metadata_entry.copy()

        if len(selector) == 0:
            new_metadata_entry.metadata = frozendict.FrozenOrderedDict()
            if recursive:
                new_metadata_entry.all_elements = None
                new_metadata_entry.elements = utils.EMPTY_PMAP
                new_metadata_entry.is_elements_empty = True
                new_metadata_entry.is_empty = True
            else:
                new_metadata_entry.update_is_empty()
            return new_metadata_entry

        segment, selector_rest = selector[0], selector[1:]

        if segment is ALL_ELEMENTS:
            new_metadata_entry.all_elements = self._remove(selector_rest, recursive, strict_all_elements, new_metadata_entry.all_elements)
            if new_metadata_entry.all_elements.is_empty:
                new_metadata_entry.all_elements = None
            new_metadata_entry.update_is_empty()

            if not strict_all_elements and new_metadata_entry.elements:
                new_elements_evolver = new_metadata_entry.elements.evolver()
                for element_segment, element_metadata_entry in new_metadata_entry.elements.items():
                    new_element_metadata_entry = self._remove(selector_rest, recursive, strict_all_elements, element_metadata_entry)
                    if new_element_metadata_entry.is_empty:
                        new_elements_evolver.remove(element_segment)
                    else:
                        new_elements_evolver.set(element_segment, new_element_metadata_entry)
                new_metadata_entry.elements = new_elements_evolver.persistent()
                new_metadata_entry.is_elements_empty = not new_metadata_entry.elements
                new_metadata_entry.update_is_empty()

        else:
            segment = typing.cast(SimpleSelectorSegment, segment)
            if segment in new_metadata_entry.elements:
                new_element_metadata_entry = self._remove(selector_rest, recursive, strict_all_elements, new_metadata_entry.elements[segment])
                if new_element_metadata_entry.is_empty:
                    new_metadata_entry.elements = new_metadata_entry.elements.remove(segment)
                else:
                    new_metadata_entry.elements = new_metadata_entry.elements.set(segment, new_element_metadata_entry)
                new_metadata_entry.is_elements_empty = not new_metadata_entry.elements
                new_metadata_entry.update_is_empty()

        return new_metadata_entry

    def _update(self, selector: Selector, metadata_entry: typing.Optional[MetadataEntry],
                metadata: frozendict.FrozenOrderedDict) -> MetadataEntry:
        if metadata_entry is None:
            new_metadata_entry = MetadataEntry()
        else:
            new_metadata_entry = metadata_entry.copy()

        if len(selector) == 0:
            # One would think that we could remove "NO_VALUE" values during merging, but we have to
            # keep them to know which values we have to remove when merging with all elements metadata.
            new_metadata_entry.metadata = self._merge_metadata(new_metadata_entry.metadata, metadata)
            new_metadata_entry.update_is_empty()
            return new_metadata_entry

        segment, selector_rest = selector[0], selector[1:]

        if segment is ALL_ELEMENTS:
            new_metadata_entry.all_elements = self._update(selector_rest, new_metadata_entry.all_elements, metadata)
            if new_metadata_entry.all_elements.is_empty:
                new_metadata_entry.all_elements = None
            new_metadata_entry.update_is_empty()

            if new_metadata_entry.elements:
                # Fields on direct elements have precedence over fields on ALL_ELEMENTS, but we want the last
                # call to update to take precedence. So all fields found in metadata just set on ALL_ELEMENTS
                # are removed from all metadata on direct elements.
                new_elements_evolver = new_metadata_entry.elements.evolver()
                for element_segment, element_metadata_entry in new_metadata_entry.elements.items():
                    new_element_metadata_entry = self._prune(selector_rest, element_metadata_entry, metadata)
                    if new_element_metadata_entry is None or new_element_metadata_entry.is_empty:
                        new_elements_evolver.remove(element_segment)
                    else:
                        new_elements_evolver.set(element_segment, new_element_metadata_entry)
                new_metadata_entry.elements = new_elements_evolver.persistent()
                new_metadata_entry.is_elements_empty = not new_metadata_entry.elements
                new_metadata_entry.update_is_empty()

        else:
            segment = typing.cast(SimpleSelectorSegment, segment)
            new_element_metadata_entry = self._update(selector_rest, new_metadata_entry.elements.get(segment, None), metadata)
            if new_element_metadata_entry.is_empty:
                new_metadata_entry.elements = new_metadata_entry.elements.discard(segment)
            else:
                new_metadata_entry.elements = new_metadata_entry.elements.set(segment, new_element_metadata_entry)
            new_metadata_entry.is_elements_empty = not new_metadata_entry.elements
            new_metadata_entry.update_is_empty()

        return new_metadata_entry

    def _merge_metadata(self, metadata1: frozendict.FrozenOrderedDict, metadata2: frozendict.FrozenOrderedDict) -> frozendict.FrozenOrderedDict:
        """
        Merges all fields from ``metadata2`` on top of ``metadata1``, recursively.

        Only dicts are merged recursively, arrays are not.
        """

        # Copy so that we can mutate.
        metadata = collections.OrderedDict(metadata1)

        for name, value in metadata2.items():
            if name in metadata:
                if isinstance(metadata[name], frozendict.FrozenOrderedDict) and isinstance(value, frozendict.FrozenOrderedDict):
                    merged_value = self._merge_metadata(metadata[name], value)
                    # If value is an empty dict, but before merging it was not, we just remove the whole field.
                    if metadata[name] and not merged_value:
                        del metadata[name]
                    else:
                        metadata[name] = merged_value
                else:
                    metadata[name] = value
            else:
                metadata[name] = value

        return frozendict.FrozenOrderedDict(metadata)

    def _merge_metadata_entries(self, metadata_entry1: MetadataEntry, metadata_entry2: MetadataEntry) -> MetadataEntry:
        """
        Merges ``metadata_entry2`` on top of ``metadata_entry1``, recursively, and
        returns a new metadata entry.
        """

        output_metadata_entry = MetadataEntry()

        # Merging elements.
        new_elements_evolver = metadata_entry1.elements.evolver()
        for element_segment, element_metadata_entry in metadata_entry2.elements.items():
            if element_segment not in new_elements_evolver:
                new_elements_evolver.set(element_segment, element_metadata_entry)
            else:
                new_elements_evolver.set(
                    element_segment, self._merge_metadata_entries(new_elements_evolver[element_segment], element_metadata_entry),
                )
        output_metadata_entry.elements = new_elements_evolver.persistent()
        output_metadata_entry.is_elements_empty = not output_metadata_entry.elements

        # Merging "ALL_ELEMENTS".
        if metadata_entry1.all_elements is not None and metadata_entry2.all_elements is not None:
            output_metadata_entry.all_elements = self._merge_metadata_entries(metadata_entry1.all_elements, metadata_entry2.all_elements)
        elif metadata_entry1.all_elements is not None:
            output_metadata_entry.all_elements = metadata_entry1.all_elements
        elif metadata_entry2.all_elements is not None:
            output_metadata_entry.all_elements = metadata_entry2.all_elements

        # Merging metadata:
        output_metadata_entry.metadata = self._merge_metadata(metadata_entry1.metadata, metadata_entry2.metadata)

        output_metadata_entry.update_is_empty()

        return output_metadata_entry

    def _remove_no_value(self, metadata: frozendict.FrozenOrderedDict) -> frozendict.FrozenOrderedDict:
        # Copy so that we can mutate.
        metadata = collections.OrderedDict(metadata)

        # We iterate over a list so that we can change dict while iterating.
        for name, value in list(metadata.items()):
            if value is NO_VALUE:
                del metadata[name]
            elif isinstance(value, frozendict.FrozenOrderedDict):
                new_value = self._remove_no_value(value)
                # If value is an empty dict, but before removing "NO_VALUE" it was not, we just remove the whole field.
                if metadata[name] and not new_value:
                    del metadata[name]
                else:
                    metadata[name] = new_value

        return frozendict.FrozenOrderedDict(metadata)

    def _prune(self, selector: Selector, metadata_entry: typing.Optional[MetadataEntry], metadata: frozendict.FrozenOrderedDict) -> typing.Optional[MetadataEntry]:
        if metadata_entry is None:
            return metadata_entry

        new_metadata_entry = metadata_entry.copy()

        if len(selector) == 0:
            new_metadata_entry.metadata = self._prune_metadata(new_metadata_entry.metadata, metadata)
            new_metadata_entry.update_is_empty()
            return new_metadata_entry

        segment, selector_rest = selector[0], selector[1:]

        if segment is ALL_ELEMENTS:
            new_metadata_entry.all_elements = self._prune(selector_rest, new_metadata_entry.all_elements, metadata)
            if new_metadata_entry.all_elements is not None and new_metadata_entry.all_elements.is_empty:
                new_metadata_entry.all_elements = None
            new_metadata_entry.update_is_empty()

            if new_metadata_entry.elements:
                new_elements_evolver = new_metadata_entry.elements.evolver()
                for element_segment, element_metadata_entry in new_metadata_entry.elements.items():
                    new_element_metadata_entry = self._prune(selector_rest, element_metadata_entry, metadata)
                    if new_element_metadata_entry is None or new_element_metadata_entry.is_empty:
                        new_elements_evolver.remove(element_segment)
                    else:
                        new_elements_evolver.set(element_segment, new_element_metadata_entry)
                new_metadata_entry.elements = new_elements_evolver.persistent()
                new_metadata_entry.is_elements_empty = not new_metadata_entry.elements
                new_metadata_entry.update_is_empty()

        elif segment in new_metadata_entry.elements:
            segment = typing.cast(SimpleSelectorSegment, segment)
            new_element_metadata_entry = self._prune(selector_rest, new_metadata_entry.elements[segment], metadata)
            if new_element_metadata_entry is None or new_element_metadata_entry.is_empty:
                new_metadata_entry.elements = new_metadata_entry.elements.remove(segment)
            else:
                new_metadata_entry.elements = new_metadata_entry.elements.set(segment, new_element_metadata_entry)
            new_metadata_entry.is_elements_empty = not new_metadata_entry.elements
            new_metadata_entry.update_is_empty()

        return new_metadata_entry

    def _prune_metadata(self, metadata1: frozendict.FrozenOrderedDict, metadata2: frozendict.FrozenOrderedDict) -> frozendict.FrozenOrderedDict:
        """
        Removes all fields which are found in ``metadata2`` from ``metadata1``, recursively.

        Values of ``metadata2`` do not matter, except if they are a dict, in which case
        removal is done recursively.
        """

        # Copy so that we can mutate.
        metadata = collections.OrderedDict(metadata1)

        for name, value in metadata2.items():
            if name not in metadata:
                continue

            if isinstance(metadata[name], frozendict.FrozenOrderedDict) and isinstance(value, frozendict.FrozenOrderedDict):
                pruned_value = self._prune_metadata(metadata[name], value)
                # If value is an empty dict, but before pruning it was not, we just remove the whole field.
                if metadata[name] and not pruned_value:
                    del metadata[name]
                else:
                    metadata[name] = pruned_value
            else:
                del metadata[name]

        return frozendict.FrozenOrderedDict(metadata)

    def compact(self: T, fields_to_compact: typing.Sequence[str]) -> T:
        """
        Compact metadata and return it. Produces equivalent but compact
        metadata where equal metadata for all elements in a dimension are compacted
        into ``ALL_ELEMENTS`` selector segment.

        Parameters
        ----------
        fields_to_compact:
            Which fields to compact in the metadata.

        Returns
        -------
        Compacted metadata.
        """

        metadata_dict: typing.Dict[TupleSelector, typing.Dict] = collections.OrderedDict()

        for metadata_description in self.to_internal_simple_structure():
            metadata_dict[tuple(metadata_description['selector'])] = metadata_description['metadata']

        metadata_dict = self._compact_metadata(metadata_dict, fields_to_compact)

        new_metadata = copy.copy(self)

        for selector, metadata in metadata_dict.items():
            metadata = utils.make_immutable_copy(metadata)

            if not isinstance(metadata, frozendict.FrozenOrderedDict):
                raise exceptions.InvalidArgumentTypeError("Metadata should be a dict.")

            new_metadata._current_metadata = new_metadata._update(selector, new_metadata._current_metadata, metadata)

        return new_metadata

    # TODO: During compacting, we could also create an Union type of all structural types in elements and set it on "ALL_ELEMENTS".
    @classmethod
    def _compact_metadata(cls: typing.Type[T], metadata_dict: typing.Dict[TupleSelector, typing.Dict], fields_to_compact: typing.Sequence[str]) -> typing.Dict[TupleSelector, typing.Dict]:
        """
        Compacts only top-level fields (if their values are all equal) listed in ``fields_to_compact``.

        Only top-level fields listed in ``fields_to_compact`` will be compacted. The reason for ``fields_to_compact``
        is that it is an optimization, so that we do not have to first go over all metadata to detect which all
        fields are there. When used by ``_generate``, ``_generate_metadata`` is producing a fixed set of fields which
        works in our advantage.

        We prefer to compact segments at the beginning of the selector over the segments later on.

        Parameters
        ----------
        metadata_dict:
            A dict where field is selector and value is the metadata dict under this selector.
        fields_to_compact:
            Which fields to compact in the metadata.

        Returns
        -------
        Compacted metadata representation in the form of a dict where fields are selectors.
        """

        # We rely on the fact that dicts preserve order in Python 3.6+ and do not use
        # "OrderedDict" here for simplicity (we do not compare by equality dicts here to care
        # about order of fields in equality check).
        results: typing.Dict[TupleSelector, typing.Dict] = collections.defaultdict(dict)

        # Key is the length of selectors and the value is a list of selectors of the same length.
        selector_lengths: typing.Dict[int, typing.List[TupleSelector]] = collections.defaultdict(list)
        for selector in metadata_dict.keys():
            selector_lengths[len(selector)].append(selector)

        for length, selectors in sorted(selector_lengths.items(), key=operator.itemgetter(0)):
            update_selectors: typing.Dict[TupleSelector, typing.List] = collections.defaultdict(list)

            for field in fields_to_compact:
                values_to_selectors: typing.Dict[typing.Any, typing.List[TupleSelector]] = collections.defaultdict(list)
                for selector in selectors:
                    if field in metadata_dict[selector]:
                        values_to_selectors[metadata_dict[selector][field]].append(selector)

                for value in values_to_selectors.keys():
                    compacted_selectors = cls._get_compacted_selectors(values_to_selectors[value], selectors)

                    for selector in compacted_selectors:
                        update_selectors[selector].append({field: value})

            for selector, items in sorted(update_selectors.items(), key=operator.itemgetter(0)):
                for item in items:
                    results[selector].update(item)

        return collections.OrderedDict(results)

    @classmethod
    def _get_compacted_selectors(cls, selectors_to_compact: typing.List[TupleSelector], total_selectors: typing.List[TupleSelector]) -> typing.List[TupleSelector]:
        """
        This function returns a compacted representation of ``selectors_to_compact``.

        Parameters
        ----------
        selectors_to_compact:
            A list of selectors to be compacted which have the same value under a certain field.
        total_selectors:
            All possible selectors of a certain length.

        Returns
        -------
        A list of compacted selectors.
        """

        input_selectors = copy.copy(selectors_to_compact)
        input_selectors_set = set(input_selectors)
        output_selectors = selectors_to_compact

        length_of_selector = len(input_selectors[0])

        other_selectors_set = set(total_selectors) - input_selectors_set

        for other_selector in sorted(other_selectors_set):
            if cls._selector_overlap(other_selector, input_selectors_set):
                other_selectors_set.remove(other_selector)

        for i in range(length_of_selector):
            all_segments = {selector[i] for selector in total_selectors}
            for index, selector_tuple in enumerate(output_selectors):
                can_collapse = True

                for segment in all_segments:
                    test_selector = list(selector_tuple)
                    test_selector[i] = segment
                    if cls._selector_overlap(test_selector, other_selectors_set):
                        can_collapse = False

                if can_collapse:
                    selector_list = list(selector_tuple)
                    selector_list[i] = ALL_ELEMENTS
                    output_selectors[index] = tuple(selector_list)

            output_selectors = sorted(set(output_selectors))

        output_selectors = cls._greedy_prune_selector(output_selectors, input_selectors)

        return output_selectors

    @classmethod
    def _selector_overlap(cls, test_selector: Selector, selectors_set: typing.Set[TupleSelector]) -> bool:
        """
        This function checks if ``test_selector`` overlaps with selectors ``selectors_set``.

        Parameters
        ----------
        test_selector:
            The input selector.
        selectors_set:
            A set of selectors.

        Returns
        -------
        Whether the selector ``test_selector`` overlaps with any selector in ``selectors_set``.
        """

        for selector in selectors_set:
            assert len(selector) == len(test_selector)

            is_same = True
            for i in range(len(test_selector)):
                if test_selector[i] is ALL_ELEMENTS:
                    continue
                if selector[i] is not ALL_ELEMENTS:
                    if test_selector[i] != selector[i]:
                        is_same = False

            if is_same:
                return True

        return False

    @classmethod
    def _selector_contained(cls, selector_1: Selector, selector_2: Selector) -> bool:
        """
        This function checks if ``selector_1`` is contained in ``selector_2``.

        Returns
        -------
        Whether ``selector_1`` is contained in ``selector_2``.

        Notes
        -----
        This function is different from `_selector_overlap` which checks if two selectors overlap.
        """

        for i in range(len(selector_1)):
            if selector_1[i] is ALL_ELEMENTS:
                if selector_2[i] is not ALL_ELEMENTS:
                    return False
                continue
            if selector_2[i] is not ALL_ELEMENTS:
                if selector_1[i] != selector_2[i]:
                    return False

        return True

    @classmethod
    def _greedy_prune_selector(cls, compacted_selectors: typing.List[TupleSelector], selectors_to_compact: typing.List[TupleSelector]) -> typing.List[TupleSelector]:
        """
        This method implements a greedy algorithm to remove unnecessary selectors from ``compacted_selectors``.

        Parameters
        ----------
        compacted_selectors:
            This is an already compacted list of selectors which we get from ``selectors_to_compact``.
        selectors_to_compact:
            This is the list of original selectors with the same value under a certain field.

        Returns
        -------
        The list of selectors where unnecessary selectors have been removed from ``compacted_selectors``.
        """

        # Maps from each selector in "compacted_selectors" to selectors which it covers in "selectors_to_compact".
        contained_selectors: typing.Dict[TupleSelector, typing.List[TupleSelector]] = collections.defaultdict(list)
        selector_count_mask: typing.Dict[TupleSelector, int] = collections.defaultdict(int)

        # Compute for each selector in "selectors_to_compact" how many selectors in "compacted_selectors" cover them.
        # Also builds the "contained_selectors".
        for compact_selector in compacted_selectors:
            for selector in selectors_to_compact:
                if cls._selector_contained(selector, compact_selector):
                    selector_count_mask[selector] += 1
                    contained_selectors[compact_selector].append(selector)

        continue_flag = True
        while continue_flag:
            continue_flag = False
            for compact_selector in compacted_selectors:
                remove_flag = True
                for selector in contained_selectors[compact_selector]:
                    if selector_count_mask[selector] == 1:
                        remove_flag = False
                if remove_flag:
                    continue_flag = True
                    redundant_selector = compact_selector
            if continue_flag:
                compacted_selectors.remove(redundant_selector)
                for selector in contained_selectors[redundant_selector]:
                    selector_count_mask[selector] -= 1

        return compacted_selectors

    @classmethod
    def check_selector(cls, selector: Selector) -> None:
        """
        Checks that a given ``selector`` is a valid selector. If ``selector`` is invalid it raises an exception.

        It checks that it is a tuple or a list and currently we require that all segments of a selector
        are strings, integers, or a special value ``ALL_ELEMENTS``.

        Parameters
        ----------
        selector:
            Selector to check.
        """

        if not isinstance(selector, (tuple, list)):
            raise exceptions.InvalidArgumentTypeError("Selector is not a tuple or a list.")

        for i, segment in enumerate(selector):
            if not isinstance(segment, (str, int)) and segment is not ALL_ELEMENTS:
                raise exceptions.InvalidArgumentTypeError(
                    "'{segment}' at {path} is not a str, int, or ALL_ELEMENTS.".format(
                        segment=segment,
                        path=list(selector[0:i + 1]),
                    ),
                )

    def __hash__(self) -> int:
        if self._hash is None:
            self._hash = hash(self._current_metadata)

        return self._hash

    def __eq__(self, other):  # type: ignore
        if not isinstance(other, Metadata):
            return NotImplemented

        return self._current_metadata == other._current_metadata

    def get_elements(self, selector: Selector) -> typing.Sequence[SelectorSegment]:
        """
        Returns a list of element names which exists under a selector, if any.

        Parameters
        ----------
        selector:
            A selector to return elements under.

        Returns
        -------
        List of element names.
        """

        self.check_selector(selector)

        return self._get_elements(selector, self._current_metadata)

    def _get_elements(self, selector: Selector, metadata_entry: typing.Optional[MetadataEntry]) -> typing.Sequence[SelectorSegment]:
        if metadata_entry is None:
            return []
        if len(selector) == 0:
            if metadata_entry.all_elements is not None:
                all_elements: ListSelector = [ALL_ELEMENTS]
            else:
                all_elements = []
            return all_elements + list(metadata_entry.elements.keys())

        segment, selector_rest = selector[0], selector[1:]

        all_elements_elements = self._get_elements(selector_rest, metadata_entry.all_elements)
        if segment is ALL_ELEMENTS:
            elements = all_elements_elements
        elif segment in metadata_entry.elements:
            segment = typing.cast(SimpleSelectorSegment, segment)
            elements = self._get_elements(selector_rest, metadata_entry.elements[segment])
            elements = sorted(set(typing.cast(typing.List, all_elements_elements) + typing.cast(typing.List, elements)))
        else:
            elements = all_elements_elements

        return elements

    def to_internal_json_structure(self) -> typing.Sequence[typing.Dict]:
        """
        Converts metadata to a JSON-compatible structure.

        The structure exposes how metadata is stored internally (metadata for ``ALL_ELEMENTS``
        separate from metadata for individual elements) and can change in the future.
        This method exist for debugging purposes and to allow serialization of metadata.
        Use `to_json_structure` method if you want to access semantically valid
        representation of metadata.

        Returns
        -------
        A JSON-compatible list of dicts.
        """

        ALL_ELEMENTS_REPR = repr(ALL_ELEMENTS)

        return [
            {
                'selector': [ALL_ELEMENTS_REPR if segment is ALL_ELEMENTS else segment for segment in entry['selector']],
                'metadata': utils.to_reversible_json_structure(entry['metadata']),
            }
            for entry in self.to_internal_simple_structure()
        ]

    def to_internal_simple_structure(self) -> typing.Sequence[typing.Dict]:
        """
        Converts metadata to a simple structure, similar to JSON, but with values
        left as Python values.

        The structure exposes how metadata is stored internally (metadata for ``ALL_ELEMENTS``
        separate from metadata for individual elements) and can change in the future.
        This method exist for debugging purposes and to allow serialization of metadata.
        Use `to_simple_structure` method if you want to access semantically valid
        representation of metadata.

        Returns
        -------
        A list of dicts.
        """

        return self._to_internal_simple_structure([], self._current_metadata)

    @classmethod
    def from_internal_json_structure(cls: typing.Type[T], json_structure: typing.Iterable[typing.Dict]) -> T:
        """
        Constructs metadata object back from an internal JSON-compatible structure.
        as made by ``to_internal_json_structure``.

        Parameters
        ----------
        json_structure:
            Iterable of the structure.

        Returns
        -------
        Constructed metadata object.
        """

        ALL_ELEMENTS_REPR = repr(ALL_ELEMENTS)

        return cls.from_internal_simple_structure(
            {
                'selector': [ALL_ELEMENTS if segment == ALL_ELEMENTS_REPR else segment for segment in entry['selector']],
                'metadata': utils.from_reversible_json_structure(entry['metadata']),
            } for entry in json_structure
        )

    @classmethod
    def from_internal_simple_structure(cls: typing.Type[T], structure: typing.Iterable[typing.Dict]) -> T:
        """
        Constructs metadata object back from an internal simple structure,
        as made by ``to_internal_simple_structure``.

        Parameters
        ----------
        structure:
            Iterable of the structure.

        Returns
        -------
        Constructed metadata object.
        """

        metadata = cls()

        # TODO: Optimize, see: https://gitlab.com/datadrivendiscovery/d3m/issues/408
        for entry in structure:
            metadata = metadata.update(entry['selector'], entry['metadata'])

        return metadata

    def _to_internal_simple_structure(self, selector: Selector, metadata_entry: typing.Optional[MetadataEntry]) -> typing.List[typing.Dict]:
        output = []

        selector = typing.cast(ListSelector, selector)

        if metadata_entry.metadata:
            output.append({
                'selector': list(selector),
                'metadata': metadata_entry.metadata,
            })

        if metadata_entry.all_elements is not None:
            output += self._to_internal_simple_structure(selector + [ALL_ELEMENTS], metadata_entry.all_elements)

        for element_segment, element_metadata_entry in metadata_entry.elements.items():
            output += self._to_internal_simple_structure(selector + [element_segment], element_metadata_entry)

        return output

    def to_json_structure(self) -> typing.Sequence[typing.Dict]:
        """
        Converts metadata to a JSON-compatible structure.

        The output matches the output one obtain by using `query` method and is a
        semantically valid representation of metadata, but it does not matches
        how metadata is stored internally. To obtain that, you can use
        `to_internal_json_structure` method.

        It does not make a JSON structure which can then be parsed back to
        reconstruct original metadata object. To obtain that, you can use
        `to_internal_json_structure` method.

        Returns
        -------
        A JSON-compatible list of dicts.
        """

        return utils.to_json_structure(self.to_simple_structure())

    def to_simple_structure(self) -> typing.Sequence[typing.Dict]:
        """
        Converts metadata to a simple structure, similar to JSON, but with values
        left as Python values.

        The output matches the output one obtain by using `query` method and is a
        semantically valid representation of metadata, but it does not matches
        how metadata is stored internally. To obtain that, you can use
        `to_internal_simple_structure` method.

        It does not make a structure which can then be converted back to
        reconstruct original metadata object. To obtain that, you can use
        `to_internal_simple_structure` method.

        Returns
        -------
        A list of dicts.
        """

        return self._to_simple_structure([])

    def _to_simple_structure(self, selector: Selector) -> typing.List[typing.Dict]:
        output = []

        selector = typing.cast(ListSelector, selector)

        if 'selector' in inspect.signature(self.query).parameters:
            query = self.query
        else:
            def query(selector: Selector, *, ignore_all_elements: bool = False, remove_no_value: bool = True) -> frozendict.FrozenOrderedDict:
                return self.query()  # type: ignore

        metadata = query(selector=selector)
        if metadata:
            output.append({
                'selector': list(selector),
                'metadata': metadata,
            })

        elements = self.get_elements(selector)

        for element in elements:
            output += self._to_simple_structure(selector + [element])

        return output

    def pretty_print(self, selector: Selector = None, handle: typing.IO[typing.Any] = None, _level: int = 0) -> None:
        """
        Pretty-prints metadata to ``handle``, or `sys.stdout` if not specified.

        The output matches the output one obtain by using `query` method and is a
        semantically valid representation of metadata, but it does not matches
        how metadata is stored internally. To obtain that, you can use
        `to_internal_json_structure` and `to_internal_simple_structure` methods.

        Parameters
        ----------
        selector:
            A selector to start pretty-printing at.
        handle:
            A handle to pretty-print to. Default is `sys.stdout`.
        """

        if selector is None:
            selector = []

        if handle is None:
            handle = sys.stdout

        self.check_selector(selector)

        selector = list(selector)

        if 'selector' in inspect.signature(self.query).parameters:
            query = self.query
        else:
            def query(selector: Selector, *, ignore_all_elements: bool = False, remove_no_value: bool = True) -> frozendict.FrozenOrderedDict:
                return self.query()  # type: ignore

        indent = ' ' * _level

        handle.write('{indent}Selector:\n{indent} {selector}\n'.format(indent=indent, selector=tuple(selector)))

        handle.write('{indent}Metadata:\n'.format(indent=indent))
        for line in json.dumps(utils.to_json_structure(query(selector=selector)), indent=1, allow_nan=False).splitlines():
            handle.write('{indent} {line}\n'.format(indent=indent, line=line))

        elements = self.get_elements(selector)

        if not elements:
            return

        if ALL_ELEMENTS in elements:
            handle.write('{indent}All elements:\n'.format(indent=indent))
            self.pretty_print(selector + [ALL_ELEMENTS], handle=handle, _level=_level + 1)

        first_element = True
        for element in elements:
            if element is ALL_ELEMENTS:
                continue

            if first_element:
                handle.write('{indent}Elements:\n'.format(indent=indent))
                first_element = False

            self.pretty_print(selector + [element], handle=handle, _level=_level + 1)

    def _copy_elements_metadata(self, target_metadata: T, from_selector: ListSelector,
                                to_selector: ListSelector, selector: ListSelector, ignore_all_elements: bool) -> T:
        # "ALL_ELEMENTS" is always first, if it exists, which works in our favor here.
        # We are copying metadata for both "ALL_ELEMENTS" and elements themselves, so
        # we do not have to merge metadata together for elements themselves.
        elements = self.get_elements(from_selector + selector)

        for element in elements:
            new_selector = selector + [element]
            metadata = self._query(from_selector + new_selector, self._current_metadata, 0 if ignore_all_elements else len(from_selector))
            target_metadata = target_metadata.update(to_selector + new_selector, metadata)
            target_metadata = self._copy_elements_metadata(target_metadata, from_selector, to_selector, new_selector, ignore_all_elements)

        return target_metadata

    def copy_to(self, target_metadata: T, from_selector: Selector,
                to_selector: Selector = (), *, ignore_all_elements: bool = False) -> T:
        """
        Recursively copies metadata to ``target_metadata``, starting at the
        ``from_selector`` and to a selector starting at ``to_selector``.
        """

        metadata = self._query(from_selector, self._current_metadata, 0 if ignore_all_elements else len(from_selector))

        # Do not copy top-level "schema" field to a lower level.
        if from_selector == () and to_selector != () and 'schema' in metadata:
            # Copy so that we can mutate.
            metadata_dict = collections.OrderedDict(metadata)
            del metadata_dict['schema']
            metadata = frozendict.FrozenOrderedDict(metadata_dict)

        target_metadata = target_metadata.update(to_selector, metadata)

        return self._copy_elements_metadata(target_metadata, list(from_selector), list(to_selector), [], ignore_all_elements)


class DataMetadata(Metadata):
    """
    A class for metadata for data values.

    It checks all updates against container and data schemas. Note that as such empty (just created) metadata object
    does not validate against schemas. Consider setting required fields manually or use `generate` method as a
    helper to do so.

    It has additional helper methods for operating on metadata of tabular data.

    Parameters
    ----------
    metadata:
        Optional initial metadata for the top-level of the value.
    for_value:
        Optional value to automatically generate metadata for. DEPRECATED: use explicit generate method call instead.
    generate_metadata: bool
        Automatically generate metadata from ``for_value`` and update the metadata accordingly.
        DEPRECATED: use explicit generate method call instead.
    check:
        DEPRECATED: argument ignored.
    source:
        DEPRECATED: argument ignored.
    timestamp:
        DEPRECATED: argument ignored.
    """

    @deprecate.arguments('for_value', 'generate_metadata', message="use explicit generate method call instead")
    @deprecate.arguments('source', 'timestamp', 'check', message="argument ignored")
    def __init__(self, metadata: typing.Dict[str, typing.Any] = None, for_value: typing.Any = None, *,
                 generate_metadata: bool = True, check: bool = True, source: typing.Any = None, timestamp: datetime.datetime = None) -> None:
        super().__init__(metadata=metadata)

        if for_value is not None and generate_metadata:
            self._generate(for_value)

    @deprecate.arguments('source', 'timestamp', 'check', 'for_value', message="argument ignored")
    def update(self: D, selector: Selector, metadata: typing.Dict[str, typing.Any], *, for_value: typing.Any = None,
               check: bool = True, source: typing.Any = None, timestamp: datetime.datetime = None) -> D:
        """
        Updates metadata with new ``metadata`` for data pointed to with ``selector``.

        If value of any field is ``NO_VALUE``, that field is deleted.

        It returns a copy of this metadata object with new metadata applied.

        Parameters
        ----------
        selector:
            A selector pointing to data.
        metadata:
            A map of fields and values with metadata.
        for_value:
            DEPRECATED: argument ignored.
        check:
            DEPRECATED: argument ignored.
        source:
            DEPRECATED: argument ignored.
        timestamp:
            DEPRECATED: argument ignored.

        Returns
        -------
        Updated metadata.
        """

        return super().update(selector=selector, metadata=metadata)

    @deprecate.arguments('source', 'timestamp', 'check', 'for_value', message="argument ignored")
    def remove(self: D, selector: Selector, *, recursive: bool = False, strict_all_elements: bool = False,
               for_value: typing.Any = None, check: bool = True, source: typing.Any = None, timestamp: datetime.datetime = None) -> D:
        """
        Removes all metadata at ``selector``.

        Parameters
        ----------
        selector:
            A selector to remove metadata at.
        recursive:
            Should remove also all metadata under the ``selector``?
        strict_all_elements:
            If ``True``, then when removing ``ALL_ELEMENTS`` entry, do not remove also metadata for all elements it matches.
        for_value:
            DEPRECATED: argument ignored.
        check:
            DEPRECATED: argument ignored.
        source:
            DEPRECATED: argument ignored.
        timestamp:
            DEPRECATED: argument ignored.

        Returns
        -------
        Updated metadata.
        """

        return super().remove(selector=selector, recursive=recursive, strict_all_elements=strict_all_elements)

    @deprecate.function(message="use generate method instead")
    @deprecate.arguments('source', 'timestamp', 'check', message="argument ignored")
    def set_for_value(self: D, for_value: typing.Any = None, *, generate_metadata: bool = True, check: bool = True,
                      source: typing.Any = None, timestamp: datetime.datetime = None) -> D:
        """
        DEPRECATED: use ``generate`` method instead.

        If ``generate_metadata`` is set, generate metadata from ``for_value`` and update the metadata accordingly.

        Parameters
        ----------
        for_value:
            Value to automatically generate metadata for.
        generate_metadata: bool
            Automatically generate metadata from ``for_value`` and update the metadata accordingly.
        check:
            DEPRECATED: argument ignored.
        source:
            DEPRECATED: argument ignored.
        timestamp:
            DEPRECATED: argument ignored.

        Returns
        -------
        Metadata object updated with automatically generated metadata.
        """

        if for_value is not None and generate_metadata:
            return self.generate(for_value)
        else:
            return self

    def generate(self: D, value: typing.Any = None, *, compact: bool = False) -> D:
        """
        Metadata about structure of data (dimensions) and structural types is
        generated for the ``value``, and existing metadata is updated accordingly.

        Parameters
        ----------
        value:
            Value to automatically generate metadata for.
        compact:
            Compact automatically generated metadata. Produces equivalent but compact
            metadata where equal metadata for all elements in a dimension are compacted
            into ``ALL_ELEMENTS`` selector segment.

        Returns
        -------
        Metadata object updated with automatically generated metadata.
        """

        new_metadata = copy.copy(self)

        new_metadata._generate(value, compact)

        return new_metadata

    def _generate(self, value: typing.Any = None, compact: bool = False) -> None:
        # Importing here to prevent import cycle. And to not import it many times inside "_generate_metadata".
        from d3m import container, types as d3m_types

        if value is None:
            raise exceptions.InvalidArgumentValueError("\"value\" argument cannot be None.")

        generated_metadata_dict = self._generate_metadata(container, d3m_types, value, (), True)

        if compact:
            # We make all metadata immutable so that it is hashable, which is required for the "_compact_generated_metadata".
            for selector, metadata in generated_metadata_dict.items():
                generated_metadata_dict[selector] = utils.make_immutable_copy(metadata)

            # Because we generated all metadata we know that we can compact it.
            # If some metadata holds for all elements we know that we can move it to "ALL_ELEMENTS".
            generated_metadata_dict = self._compact_metadata(generated_metadata_dict, ALL_GENERATED_FIELDS)

        self._update_with_generated_metadata(generated_metadata_dict)

        # TODO: Also remove metadata for columns/rows which do not exist anymore.
        #       See: https://gitlab.com/datadrivendiscovery/d3m/issues/336

    # TODO: Should we handle inheritance between semantic types here?
    def has_semantic_type(self, selector: Selector, semantic_type: str) -> bool:
        try:
            return semantic_type in self.query_field(selector, 'semantic_types')
        except KeyError:
            return False

    @deprecate.arguments('source', 'timestamp', message="argument ignored")
    def remove_semantic_type(self: D, selector: Selector, semantic_type: str, *, source: typing.Any = None, timestamp: datetime.datetime = None) -> D:
        try:
            semantic_types = self.query_field(selector, 'semantic_types')
        except KeyError:
            return self
        if not semantic_types:
            return self
        new_semantic_types = tuple(st for st in semantic_types if st != semantic_type)
        if new_semantic_types == semantic_types:
            return self
        return self.update(selector, {'semantic_types': new_semantic_types})

    @deprecate.arguments('source', 'timestamp', message="argument ignored")
    def add_semantic_type(self: D, selector: Selector, semantic_type: str, *, source: typing.Any = None, timestamp: datetime.datetime = None) -> D:
        try:
            semantic_types = self.query_field(selector, 'semantic_types')
        except KeyError:
            semantic_types = ()
        if semantic_type in semantic_types:
            return self
        semantic_types += (semantic_type,)
        return self.update(selector, {'semantic_types': semantic_types})

    # TODO: This does not look too efficient. Optimize?
    def get_elements_with_semantic_type(self, selector: Selector, semantic_type: str) -> typing.Sequence[SelectorSegment]:
        all_elements = self.get_elements(selector)

        return [element for element in all_elements if self.has_semantic_type(list(selector) + [element], semantic_type)]

    def query_column(self, column_index: int, *, at: Selector = (), ignore_all_elements: bool = False) -> frozendict.FrozenOrderedDict:
        """
        Returns column metadata.

        This assumes that column metadata is stored under ``(ALL_ELEMENTS, column_index)``, at
        optionally ``at`` selector, which might not necessary hold if metadata is not compacted.
        Consider using `query_column_field`.

        Parameters
        ----------
        column_index:
            Column index to use.
        at:
            Selector at which to assume tabular metadata.
        ignore_all_elements:
            By default, metadata from ALL_ELEMENTS is merged with metadata for an element itself.
            By setting this argument to ``True``, this is disabled and just metadata from an element is returned.

        Returns
        -------
        Metadata of a given column.
        """

        return self.query(list(at) + [ALL_ELEMENTS, column_index], ignore_all_elements=ignore_all_elements)

    def query_column_field(self, column_index: int, field: str, *, at: Selector = (), strict_all_elements: bool = True) -> typing.Any:
        """
        Returns ``field`` value of column metadata. Raises `KeyError` exception if metadata or field
        is not set.

        ``field`` represents only top-level fields in metadata.

        Parameters
        ----------
        column_index:
            Column index to use.
        field:
            A field name to query.
        at:
            Selector at which to assume tabular metadata.
        strict_all_elements:
            If set, the method does not just return ``field`` value of column metadata,
            but checks that the value really holds for all rows matching the ``selector``,
            without exception. This is helpful also if metadata is not compacted and
            ``field`` value is the same across all rows, but ``ALL_ELEMENTS`` metadata
            does not contain that field.

        Returns
        -------
        A value of ``field`` of a given column.
        """

        return self.query_field(list(at) + [ALL_ELEMENTS, column_index], field, strict_all_elements=strict_all_elements)

    @deprecate.arguments('source', 'timestamp', message="argument ignored")
    def update_column(self: D, column_index: int, metadata: typing.Dict[str, typing.Any], *, at: Selector = (), source: typing.Any = None, timestamp: datetime.datetime = None) -> D:
        """
        Updates column metadata with new ``metadata`` for column identified by  ``column_index``.

        This stores column metadata under ``(ALL_ELEMENTS, column_index)``, at optionally ``at`` selector.

        Parameters
        ----------
        column_index:
            Column index to update.
        metadata:
            A map of fields and values with metadata.
        at:
            Selector at which to assume tabular metadata.
        source:
            DEPRECATED: argument ignored.
        timestamp:
            DEPRECATED: argument ignored.

        Returns
        -------
        Updated column metadata.
        """

        return self.update(list(at) + [ALL_ELEMENTS, column_index], metadata)

    @deprecate.arguments('source', 'timestamp', 'for_value', message="argument ignored")
    def remove_column(self: D, column_index: int, *, at: Selector = (), recursive: bool = False, strict_all_elements: bool = False,
                      for_value: typing.Any = None, source: typing.Any = None, timestamp: datetime.datetime = None) -> D:
        """
        Removes all column metadata for column ``column_index``.

        This removes column metadata under ``(ALL_ELEMENTS, column_index)``, at optionally ``at`` selector.
        It does not move to the left metadata for columns after the removed column.
        If you want that, use ``remove_columns``.

        Parameters
        ----------
        column_index:
            Column index to remove.
        at:
            Selector at which to assume tabular metadata.
        recursive:
            Should remove also all metadata under the ``selector``?
        strict_all_elements:
            If ``True``, then when removing ``ALL_ELEMENTS`` entry, do not remove also metadata for all elements it matches.
        for_value:
            DEPRECATED: argument ignored.
        source:
            DEPRECATED: argument ignored.
        timestamp:
            DEPRECATED: argument ignored.

        Returns
        -------
        Updated metadata.
        """

        return self.remove(
            list(at) + [ALL_ELEMENTS, column_index], recursive=recursive, strict_all_elements=strict_all_elements,
        )

    def get_columns_with_semantic_type(self, semantic_type: str, *, at: Selector = ()) -> typing.Sequence[SelectorSegment]:
        return self.get_elements_with_semantic_type(list(at) + [ALL_ELEMENTS], semantic_type)

    def list_columns_with_semantic_types(self, semantic_types: typing.Sequence[str], *, at: Selector = ()) -> typing.Sequence[int]:
        """
        This is similar to ``get_columns_with_semantic_type``, but it returns all column indices
        for a dimension instead of ``ALL_ELEMENTS`` element.

        Moreover, it operates on a list of semantic types, where a column is returned
        if it matches any semantic type on the list.
        """

        columns = []

        for element in self.get_elements(list(at) + [ALL_ELEMENTS]):
            try:
                metadata_semantic_types = self.query_field(list(at) + [ALL_ELEMENTS, element], 'semantic_types')
            except KeyError:
                metadata_semantic_types = ()

            # TODO: Should we handle inheritance between semantic types here?
            if any(semantic_type in metadata_semantic_types for semantic_type in semantic_types):
                if element is ALL_ELEMENTS:
                    try:
                        dimension = self.query_field(list(at) + [ALL_ELEMENTS], 'dimension')
                    except KeyError:
                        dimension = {}
                    return list(range(dimension.get('length', 0)))
                else:
                    columns.append(typing.cast(int, element))

        return columns

    def list_columns_with_structural_types(
        self, structural_types: typing.Union[typing.Callable, typing.Sequence[typing.Union[str, type]]], *,
        at: Selector = (),
    ) -> typing.Sequence[int]:
        """
        Returns a list of columns matching any of the structural types listed in
        ``structural_types``. Matching allows subclasses of those types. ``structural_types`` can also be
        a function to call to check a structural type.
        """

        columns = []

        if callable(structural_types):
            predicate = structural_types
        else:
            def predicate(typ: type) -> bool:
                return any(utils.matches_structural_type(typ, structural_type) for structural_type in typing.cast(typing.Sequence[typing.Union[str, type]], structural_types))

        for element in self.get_elements(list(at) + [ALL_ELEMENTS]):
            try:
                metadata_structural_type = self.query_field(list(at) + [ALL_ELEMENTS, element], 'structural_type')
            except KeyError:
                continue

            if predicate(metadata_structural_type):
                if element is ALL_ELEMENTS:
                    try:
                        dimension = self.query_field(list(at) + [ALL_ELEMENTS], 'dimension')
                    except KeyError:
                        dimension = {}
                    return list(range(dimension.get('length', 0)))
                else:
                    columns.append(typing.cast(int, element))

        return columns

    def _merge_generated_metadata(self, old_metadata: frozendict.FrozenOrderedDict, metadata: frozendict.FrozenOrderedDict) -> frozendict.FrozenOrderedDict:
        # Copy so that we can mutate.
        new_metadata = collections.OrderedDict(metadata)

        # Use generated "name" only if "name" does not already exist.
        # This holds even if existing "name" is "NO_VALUE".
        if 'name' in new_metadata and 'name' in old_metadata:
            del new_metadata['name']

        if 'name' in new_metadata.get('dimension', {}) and 'name' in old_metadata.get('dimension', {}):
            # Copy so that we can mutate.
            new_metadata['dimension'] = collections.OrderedDict(new_metadata['dimension'])
            del new_metadata['dimension']['name']
            new_metadata['dimension'] = frozendict.FrozenOrderedDict(new_metadata['dimension'])

        if 'semantic_types' in new_metadata:
            semantic_types = list(old_metadata.get('semantic_types', []))
            for semantic_type in new_metadata['semantic_types']:
                if semantic_type not in semantic_types:
                    # Only one tabular semantic type can exist at a time.
                    if semantic_type in TABULAR_SEMANTIC_TYPES:
                        semantic_types = [st for st in semantic_types if st not in TABULAR_SEMANTIC_TYPES]
                    semantic_types.append(semantic_type)
            new_metadata['semantic_types'] = tuple(semantic_types)

        if 'semantic_types' in new_metadata.get('dimension', {}):
            semantic_types = list(old_metadata.get('dimension', {}).get('semantic_types', []))
            for semantic_type in new_metadata['dimension']['semantic_types']:
                if semantic_type not in semantic_types:
                    # Only one tabular semantic type can exist at a time.
                    if semantic_type in TABULAR_SEMANTIC_TYPES:
                        semantic_types = [st for st in semantic_types if st not in TABULAR_SEMANTIC_TYPES]
                    semantic_types.append(semantic_type)
            # Copy so that we can mutate.
            new_metadata['dimension'] = collections.OrderedDict(new_metadata['dimension'])
            new_metadata['dimension']['semantic_types'] = tuple(semantic_types)
            new_metadata['dimension'] = frozendict.FrozenOrderedDict(new_metadata['dimension'])

        # If structural type was not generated now, but it exists before, we have to remove it.
        # Here we just delete it from "old_metadata" so that it is not re-set back, while
        # we really handle it in "_update_with_generated_metadata".
        if 'structural_type' not in new_metadata and 'structural_type' in old_metadata:
            # Copy so that we can mutate.
            old_metadata_dict = collections.OrderedDict(old_metadata)
            del old_metadata_dict['structural_type']
            old_metadata = frozendict.FrozenOrderedDict(old_metadata_dict)

        return self._merge_metadata(old_metadata, frozendict.FrozenOrderedDict(new_metadata))

    def _diff_generated_metadata(self, element_metadata: frozendict.FrozenOrderedDict, metadata: frozendict.FrozenOrderedDict) -> frozendict.FrozenOrderedDict:
        """
        When preparing updates for automatically generated metadata we want to make sure we do not override any metadata
        directly set on elements with metadata on ``ALL_ELEMENTS``. In this method we compute which metadata to update
        after the automatically generated metadata is set for ``ALL_ELEMENTS`` to restore the metadata directly set
        on elements.
        """

        # Copy so that we can mutate.
        new_element_metadata = collections.OrderedDict(element_metadata)

        # No need to set name if it is equal to metadata on "ALL_ELEMENTS".
        if 'name' in new_element_metadata and 'name' in metadata and new_element_metadata['name'] == metadata['name']:
            del new_element_metadata['name']

        # No need to set name if it is equal to metadata on "ALL_ELEMENTS".
        if 'name' in new_element_metadata.get('dimension', {}) and 'name' in metadata.get('dimension', {}) and new_element_metadata['dimension']['name'] == metadata['dimension']['name']:
            # Copy so that we can mutate.
            new_element_metadata['dimension'] = collections.OrderedDict(new_element_metadata['dimension'])
            del new_element_metadata['dimension']['name']
            new_element_metadata['dimension'] = frozendict.FrozenOrderedDict(new_element_metadata['dimension'])

        if 'semantic_types' in new_element_metadata and 'semantic_types' in metadata:
            # No need to merge semantic types if they are equal to metadata on "ALL_ELEMENTS".
            if set(new_element_metadata['semantic_types']) == set(metadata['semantic_types']):
                del new_element_metadata['semantic_types']
            else:
                semantic_types = list(new_element_metadata['semantic_types'])
                for semantic_type in metadata['semantic_types']:
                    if semantic_type not in semantic_types:
                        # Only one tabular semantic type can exist at a time.
                        if semantic_type in TABULAR_SEMANTIC_TYPES:
                            semantic_types = [st for st in semantic_types if st not in TABULAR_SEMANTIC_TYPES]
                        semantic_types.append(semantic_type)
                new_element_metadata['semantic_types'] = tuple(semantic_types)

        if 'semantic_types' in new_element_metadata.get('dimension', {}) and 'semantic_types' in metadata.get('dimension', {}):
            # No need to merge semantic types if they are equal to metadata on "ALL_ELEMENTS".
            if set(new_element_metadata['dimension']['semantic_types']) == set(metadata['dimension']['semantic_types']):
                new_element_metadata['dimension'] = collections.OrderedDict(new_element_metadata['dimension'])
                del new_element_metadata['dimension']['semantic_types']
                new_element_metadata['dimension'] = frozendict.FrozenOrderedDict(new_element_metadata['dimension'])
            else:
                semantic_types = list(new_element_metadata['dimension']['semantic_types'])
                for semantic_type in metadata['dimension']['semantic_types']:
                    if semantic_type not in semantic_types:
                        # Only one tabular semantic type can exist at a time.
                        if semantic_type in TABULAR_SEMANTIC_TYPES:
                            semantic_types = [st for st in semantic_types if st not in TABULAR_SEMANTIC_TYPES]
                        semantic_types.append(semantic_type)
                # Copy so that we can mutate.
                new_element_metadata['dimension'] = collections.OrderedDict(new_element_metadata['dimension'])
                new_element_metadata['dimension']['semantic_types'] = tuple(semantic_types)
                new_element_metadata['dimension'] = frozendict.FrozenOrderedDict(new_element_metadata['dimension'])

        # Structural type is always set or removed by generated metadata, so it should not be directly set on elements.
        if 'structural_type' in new_element_metadata:
            del new_element_metadata['structural_type']

        for generated_field in ALL_GENERATED_FIELDS:
            # We already processed these.
            if generated_field in {'name', 'dimension', 'semantic_types', 'structural_type'}:
                continue

            # No need to set this field if it is equal to metadata on "ALL_ELEMENTS".
            if generated_field in new_element_metadata and generated_field in metadata and new_element_metadata[generated_field] == metadata[generated_field]:
                del new_element_metadata[generated_field]

        # We iterate over a list so that we can change dict while iterating.
        for field in list(new_element_metadata.keys()):
            # We already processed these.
            if field in ALL_GENERATED_FIELDS:
                continue

            # Other fields are never generated, so they are never overridden, so no need to set them again.
            del new_element_metadata[field]

        if 'dimension' in new_element_metadata:
            # Copy so that we can mutate.
            new_element_metadata['dimension'] = collections.OrderedDict(new_element_metadata['dimension'])

            # Length is always set by generated metadata, so it should not be directly set on elements.
            if 'length' in new_element_metadata['dimension']:
                del new_element_metadata['dimension']['length']

            # We iterate over a list so that we can change dict while iterating.
            for field in list(new_element_metadata['dimension'].keys()):
                # We already processed these.
                if field in {'name', 'semantic_types'}:
                    continue

                # Other fields are never generated, so they are never overridden, so no need to set them again.
                del new_element_metadata['dimension'][field]

            new_element_metadata['dimension'] = frozendict.FrozenOrderedDict(new_element_metadata['dimension'])

            # If dimension ended up empty, remove it.
            if not new_element_metadata['dimension']:
                del new_element_metadata['dimension']

        return frozendict.FrozenOrderedDict(new_element_metadata)

    @classmethod
    def _generate_metadata(cls: typing.Type[D], container: types.ModuleType, d3m_types: types.ModuleType, value: typing.Any,
                           selector: TupleSelector, is_root: bool = False) -> typing.Dict[TupleSelector, typing.Dict]:
        """
        Returned metadata should be additionally compacted before use.

        We make sure that the first element of the returned dict is the entry which corresponds to the ``selector``.

        Important: Any top-level field set by this method should be listed in ``ALL_GENERATED_KEYS``.
        """

        generated_metadata: dict = {}

        if is_root:
            generated_metadata['schema'] = CONTAINER_SCHEMA_VERSION

        # We use a simple type here, not "utils.get_type" because it is faster and also because we anyway
        # traverse the data structure ourselves and store nested typing information ourselves into metadata.
        generated_metadata['structural_type'] = type(value)

        # TODO: Traverse structure also for Graph objects.
        # Fast path. We first check if the value is of a simple data type.
        if isinstance(value, d3m_types.simple_data_types):  # type: ignore
            # We just store structural type of the value (already present in "generated_metadata").
            return collections.OrderedDict([(selector, generated_metadata)])

        if isinstance(value, container.List):  # type: ignore
            generated_metadata['dimension'] = {
                'length': len(value),
            }

            metadata_dict = collections.OrderedDict([(selector, generated_metadata)])

            metadata_dict_list: typing.List[typing.Dict[TupleSelector, typing.Dict]] = []
            for v in value:
                # We recurse with selector set to "()"so that it is easier to compare results for equality.
                metadata_dict_list.append(cls._generate_metadata(container, d3m_types, v, ()))

            if metadata_dict_list:
                # Equality of "OrderedDict" also checks for the equality in order of fields.
                if all(element_dict == metadata_dict_list[0] for element_dict in metadata_dict_list):
                    selector_all_elements = selector + (ALL_ELEMENTS,)

                    # All elements are equal, so we use the first element.
                    for element_selector, element_metadata in metadata_dict_list[0].items():
                        # We recursed with selector set to "()" so we have to adapt the real selector now.
                        new_selector = selector_all_elements + element_selector
                        assert new_selector not in metadata_dict
                        metadata_dict[new_selector] = element_metadata

                else:
                    for element_index, element_dict in enumerate(metadata_dict_list):
                        for element_selector, element_metadata in element_dict.items():
                            # We recursed with selector set to "()" so we have to adapt the real selector now.
                            new_selector = selector + (element_index,) + element_selector
                            assert new_selector not in metadata_dict
                            metadata_dict[new_selector] = element_metadata

            return metadata_dict

        if isinstance(value, container.Dataset):  # type: ignore
            generated_metadata['dimension'] = {
                'name': 'resources',
                'semantic_types': ['https://metadata.datadrivendiscovery.org/types/DatasetResource'],
                'length': len(value),
            }

            metadata_dict = collections.OrderedDict([(selector, generated_metadata)])

            for k, v in value.items():
                if not isinstance(k, str):
                    raise TypeError("Dataset resource ID has to be a string, not: {k_type}".format(k_type=type(k)))
                metadata_dict.update(cls._generate_metadata(container, d3m_types, v, selector + (k,)))

            # It is unlikely that metadata is equal across dataset resources, so we do not try to compact metadata here.

            return metadata_dict

        if isinstance(value, container.DataFrame):  # type: ignore
            if len(value.shape) != 2:
                raise ValueError("Only two-dimensional DataFrames are supported, at {selector}.".format(selector=selector))

            generated_metadata['semantic_types'] = ['https://metadata.datadrivendiscovery.org/types/Table']

            generated_metadata['dimension'] = {
                'name': 'rows',
                'semantic_types': ['https://metadata.datadrivendiscovery.org/types/TabularRow'],
                'length': value.shape[0],
            }

            metadata_dict = collections.OrderedDict([(selector, generated_metadata)])

            # Reusing the variable for next dimension.
            generated_metadata = {
                'dimension': {
                    'name': 'columns',
                    'semantic_types': ['https://metadata.datadrivendiscovery.org/types/TabularColumn'],
                    'length': value.shape[1],
                },
            }

            selector_all_rows = selector + (ALL_ELEMENTS,)
            metadata_dict[selector_all_rows] = generated_metadata

            for column_index, dtype in enumerate(value.dtypes):
                column_metadata = {}

                # Only if a column name is a string. DataFrame can have a sequence/numbers for column names
                # but those are generally automatically generated so we do not use them as column names here.
                if isinstance(value.columns[column_index], str):
                    # We set the name first, so that recursive calls to "_generate_metadata" can potentially
                    # override it. "_generate_metadata" does not do it for now, but it could do it in the future.
                    # Generated names to not override names if they already exists in metadata, which is
                    # handled in the "_update_with_generated_metadata" method.
                    column_metadata['name'] = value.columns[column_index]

                selector_all_rows_column = selector_all_rows + (column_index,)

                # Values are objects. This could be something as simple as a Python string, or a whole other container value nested.
                if dtype.kind == 'O':
                    metadata_column_dict_list: typing.List[typing.Dict[TupleSelector, dict]] = []
                    for row_index, cell_value in enumerate(value.iloc[:, column_index]):
                        # We recurse with selector set to "()"so that it is easier to compare results for equality.
                        metadata_column_dict_list.append(cls._generate_metadata(container, d3m_types, cell_value, ()))

                    if metadata_column_dict_list:
                        # Equality of "OrderedDict" also checks for the equality in order of fields.
                        if all(row_dict == metadata_column_dict_list[0] for row_dict in metadata_column_dict_list):
                            # All rows are equal, so we use the first row.
                            for row_selector, row_metadata in metadata_column_dict_list[0].items():
                                # We recursed with selector set to "()" so we have to adapt the real selector now.
                                new_selector = selector_all_rows_column + row_selector
                                if new_selector == selector_all_rows_column:
                                    row_metadata.update(column_metadata)
                                assert new_selector not in metadata_dict
                                metadata_dict[new_selector] = row_metadata

                        else:
                            metadata_dict[selector_all_rows_column] = column_metadata

                            for row_index, row_dict in enumerate(metadata_column_dict_list):
                                for row_selector, row_metadata in row_dict.items():
                                    # We recursed with selector set to "()" so we have to adapt the real selector now.
                                    new_selector = selector + (row_index, column_index) + row_selector
                                    assert new_selector not in metadata_dict
                                    metadata_dict[new_selector] = row_metadata

                    else:
                        metadata_dict[selector_all_rows_column] = column_metadata

                else:
                    # DataFrame is trying to be smart and returns sometimes Python types instead
                    # of numpy types when retrieving values from it. On the other hand, dtypes are
                    # generally numpy types. So there can be discrepancy between recorded structural
                    # type in metadata and what you get for some operations out of a DataFrame.
                    # See: https://github.com/pandas-dev/pandas/issues/20791
                    #      https://github.com/pandas-dev/pandas/issues/13468
                    column_metadata['structural_type'] = dtype.type
                    metadata_dict[selector_all_rows_column] = column_metadata

            return metadata_dict

        if isinstance(value, container.ndarray):  # type: ignore
            if not value.shape:
                raise ValueError("Zero-dimensional arrays are not supported, at {selector}.".format(selector=selector))

            metadata_dict = collections.OrderedDict()

            for dimension_index, dimension_length in enumerate(value.shape):
                generated_metadata['dimension'] = {
                    'length': dimension_length,
                }

                if len(value.shape) == 2:
                    if dimension_index == 0:
                        generated_metadata['semantic_types'] = ['https://metadata.datadrivendiscovery.org/types/Table']
                        generated_metadata['dimension']['name'] = 'rows'
                        generated_metadata['dimension']['semantic_types'] = ['https://metadata.datadrivendiscovery.org/types/TabularRow']
                    elif dimension_index == 1:
                        generated_metadata['dimension']['name'] = 'columns'
                        generated_metadata['dimension']['semantic_types'] = ['https://metadata.datadrivendiscovery.org/types/TabularColumn']

                metadata_dict[selector + (ALL_ELEMENTS,) * dimension_index] = generated_metadata

                # Reusing the variable for next dimension.
                generated_metadata = {}

            if value.dtype.kind == 'O':
                metadata_cell_dict_list: typing.List[typing.Dict[TupleSelector, typing.Dict]] = []
                metadata_cell_indices: typing.List[typing.Tuple] = []

                iterator = numpy.nditer(value, flags=['multi_index', 'refs_ok'])
                while not iterator.finished:
                    # We recurse with selector set to "()"so that it is easier to compare results for equality.
                    metadata_cell_dict_list.append(cls._generate_metadata(container, d3m_types, iterator.value.item(), ()))
                    metadata_cell_indices.append(tuple(iterator.multi_index))
                    iterator.iternext()

                if metadata_cell_dict_list:
                    # Equality of "OrderedDict" also checks for the equality in order of fields.
                    if all(cell_dict == metadata_cell_dict_list[0] for cell_dict in metadata_cell_dict_list):
                        selector_all_cells = selector + (ALL_ELEMENTS,) * len(value.shape)

                        # All cells are equal, so we use the first cell.
                        for cell_selector, cell_metadata in metadata_cell_dict_list[0].items():
                            # We recursed with selector set to "()" so we have to adapt the real selector now.
                            new_selector = selector_all_cells + cell_selector
                            assert new_selector not in metadata_dict
                            metadata_dict[new_selector] = cell_metadata

                    else:
                        for cell_index, cell_dict in zip(metadata_cell_indices, metadata_cell_dict_list):
                            for cell_selector, cell_metadata in cell_dict.items():
                                # We recursed with selector set to "()" so we have to adapt the real selector now.
                                new_selector = selector + cell_index + cell_selector
                                assert new_selector not in metadata_dict
                                metadata_dict[new_selector] = cell_metadata

            else:
                metadata_dict[selector + (ALL_ELEMENTS,) * len(value.shape)] = {'structural_type': value.dtype.type}

            return metadata_dict

        # We went through all container types and none matched.
        if is_root:
            assert not isinstance(value, d3m_types.Container), type(value)  # type: ignore
            raise TypeError("Value is not of a container type, but '{type}'.".format(type=type(value)))

        # A special case for dicts, for which we traverse the structure.
        if isinstance(value, dict):
            generated_metadata['dimension'] = {
                'length': len(value),
            }

            metadata_dict = collections.OrderedDict([(selector, generated_metadata)])

            metadata_dict_list = []
            metadata_indices: typing.List[typing.Tuple] = []
            for k, v in value.items():
                if not isinstance(k, (str, int)):
                    raise TypeError("Dict key has to be a string or an integer, not: {k_type}".format(k_type=type(k)))
                # We recurse with selector set to "()"so that it is easier to compare results for equality.
                metadata_dict_list.append(cls._generate_metadata(container, d3m_types, v, ()))
                metadata_indices.append(k)

            if metadata_dict_list:
                # Equality of "OrderedDict" also checks for the equality in order of fields.
                if all(element_dict == metadata_dict_list[0] for element_dict in metadata_dict_list):
                    selector_all_elements = selector + (ALL_ELEMENTS,)

                    # All elements are equal, so we use the first element.
                    for element_selector, element_metadata in metadata_dict_list[0].items():
                        # We recursed with selector set to "()" so we have to adapt the real selector now.
                        new_selector = selector_all_elements + element_selector
                        assert new_selector not in metadata_dict
                        metadata_dict[new_selector] = element_metadata

                else:
                    for element_index, element_dict in zip(metadata_indices, metadata_dict_list):
                        for element_selector, element_metadata in element_dict.items():
                            # We recursed with selector set to "()" so we have to adapt the real selector now.
                            new_selector = selector + (element_index,) + element_selector
                            assert new_selector not in metadata_dict
                            metadata_dict[new_selector] = element_metadata

            return metadata_dict

        # We checked for all simple data types, container types, and a dict. Nothing else is left.
        assert not isinstance(value, d3m_types.Data)  # type: ignore
        raise TypeError("Value is not of a data type, but '{type}'.".format(type=type(value)))

    def _update_with_generated_metadata(self, generated_metadata_dict: typing.Dict[TupleSelector, dict]) -> None:
        """
        This method works well really just with generated metadata. It has some assumptions what ``generated_metadata_dict``
        contains and how to merge things (merge semantic types, do not override names, clear unset structural types).
        """

        # We first preprocess given updates. We have to specially merge some fields and respect overrides
        # on direct elements.
        updates: typing.List[typing.Tuple[TupleSelector, dict]] = []
        for selector, metadata in generated_metadata_dict.items():
            existing_metadata, metadata_exceptions = self.query_with_exceptions(selector, remove_no_value=False)

            # If structural type was not generated now, but it exists before, we have to remove it. In "_merge_generated_metadata" we make sure
            # it is not re-set back, and here we add an update at the beginning which removes it. The reason why it is at the beginning is that
            # it could be that the reason why there is no "structural_type" in "metadata" is because it was moved to metadata for corresponding
            # "ALL_ELEMENTS". So, the order is then: we remove it through direct selector, then maye "ALL_ELEMENTS" selector re-sets it back,
            # and merged metadata does not re-set it, because we made sure about that in "_merge_generated_metadata".
            if 'structural_type' not in metadata and 'structural_type' in existing_metadata:
                updates.insert(0, (selector, {'structural_type': NO_VALUE}))

            metadata = self._merge_generated_metadata(existing_metadata, metadata)

            updates.append((selector, metadata))

            for exception_selector, exception_metadata in metadata_exceptions.items():
                diff_metadata = self._diff_generated_metadata(exception_metadata, metadata)

                if diff_metadata:
                    updates.append((exception_selector, diff_metadata))

        for selector, metadata in updates:
            metadata = utils.make_immutable_copy(metadata)

            if not isinstance(metadata, frozendict.FrozenOrderedDict):
                raise exceptions.InvalidArgumentTypeError("Metadata should be a dict.")

            self._current_metadata = self._update(selector, self._current_metadata, metadata)

    @deprecate.function(message="create a DataMetadata instance explicitly instead")
    @deprecate.arguments('source', 'timestamp', 'check', message="argument ignored")
    def clear(self: D, metadata: typing.Dict[str, typing.Any] = None, *, for_value: typing.Any = None,
              generate_metadata: bool = True, check: bool = True, source: typing.Any = None, timestamp: datetime.datetime = None) -> D:
        """
        DEPRECATED: create a DataMetadata instance explicitly instead.

        Creates and returns a new (clear) metadata object.

        Parameters
        ----------
        metadata:
            Optional new initial metadata for the top-level of the value.
        for_value:
            Optional value to automatically generate metadata for.
        generate_metadata: bool
            Automatically generate metadata from ``for_value`` and update the metadata accordingly.
        check:
            DEPRECATED: argument ignored.
        source:
            DEPRECATED: argument ignored.
        timestamp:
            DEPRECATED: argument ignored.

        Returns
        -------
        New metadata object.
        """

        # We call wrapped parent method directly so that there are no double warnings.
        new_metadata = super().clear.__wrapped__(self, metadata=metadata)

        if for_value is not None and generate_metadata:
            new_metadata._generate(for_value)

        return new_metadata

    # TODO: Check if structural types match the real type of a value.
    def check(self, value: typing.Any) -> None:
        """
        Checks that all metadata has a corresponding data in ``value`` and that every
        metadata value is valid according to schema. If not it raises an exception.

        Parameters
        ----------
        value:
            Value to check against.
        """

        self._check_value(self._current_metadata, value, [])
        self._check_metadata([])

    @classmethod
    def _check_value(cls, metadata_entry: MetadataEntry, value: typing.Any, path: typing.List[SimpleSelectorSegment]) -> None:
        if metadata_entry.all_elements is not None:
            try:
                # We should be able to at least compute length at this dimension
                # (to signal that it is a sequence or a map).
                len(value)
            except Exception as error:
                raise ValueError("ALL_ELEMENTS set but dimension missing at {path}.".format(path=path)) from error

        if isinstance(value, numpy.matrix):
            # One cannot iterate over a matrix segment by segment. You always get back
            # a matrix (2D structure) and not an array of rows or columns. By converting
            # it to an array such iteration segment by segment works.
            value = numpy.array(value)

        if isinstance(value, pandas.DataFrame):
            for element_segment, element_metadata_entry in metadata_entry.elements.items():
                try:
                    # Fetch a row as a list.
                    element_value = [value.iloc[element_segment, k] for k in range(len(value.columns))]
                except Exception as error:
                    raise ValueError("'{element_segment}' at {path} cannot be resolved.".format(element_segment=element_segment, path=path)) from error

                cls._check_value(element_metadata_entry, element_value, path + [element_segment])

        else:
            for element_segment, element_metadata_entry in metadata_entry.elements.items():
                try:
                    element_value = value[element_segment]
                except Exception as error:
                    raise ValueError("'{element_segment}' at {path} cannot be resolved.".format(element_segment=element_segment, path=path)) from error

                cls._check_value(element_metadata_entry, element_value, path + [element_segment])

    def _check_metadata(self, selector: ListSelector) -> None:
        metadata = self.query(selector)

        if selector:
            DATA_SCHEMA_VALIDATOR.validate(metadata)
        else:
            CONTAINER_SCHEMA_VALIDATOR.validate(metadata)

        for element in self.get_elements(selector):
            self._check_metadata(selector + [element])

    @classmethod
    @deprecate.arguments('for_value', message="argument ignored")
    def check_selector(cls, selector: Selector, for_value: typing.Any = None) -> None:
        """
        Checks that a given ``selector`` is a valid selector. If ``selector`` is invalid it raises an exception.

        It checks that it is a tuple or a list and currently we require that all segments of a selector
        are strings, integers, or a special value ``ALL_ELEMENTS``.

        Parameters
        ----------
        selector:
            Selector to check.
        for_value:
            DEPRECATED: argument ignored.
        """

        super().check_selector(selector=selector)

    def get_column_index_from_column_name(self, column_name: str, *, at: Selector = ()) -> int:
        column_indices = []

        for column_index in range(self.query_field(list(at) + [ALL_ELEMENTS], 'dimension')['length']):
            try:
                if self.query_field(list(at) + [ALL_ELEMENTS, column_index], 'name') == column_name:
                    column_indices.append(column_index)
            except KeyError:
                pass

        if len(column_indices) > 1:
            raise KeyError(
                "Cannot resolve column name '{column_name}' at '{at}' because of duplicate column names".format(
                    column_name=column_name,
                    at=at,
                ),
            )
        elif column_indices:
            return column_indices[0]
        else:
            raise KeyError(
                "Cannot resolve column name '{column_name}' at '{at}' because column could not be found.".format(
                    column_name=column_name,
                    at=at,
                ),
            )

    def select_columns(self: D, columns: typing.Sequence[SimpleSelectorSegment], *, allow_empty_columns: bool = False) -> D:
        """
        Returns a new metadata object with metadata only for given ``columns``.
        Moreover, columns are renumbered based on the position in ``columns`` list.
        Top-level metadata stays unchanged, except for updating the length of the columns dimension to
        the number of columns.

        So if the ``columns`` is ``[3, 6, 5]`` then output metadata will have three columns, ``[0, 1, 2]``,
        mapping metadata for columns ``3`` to ``0``, ``6`` to ``1`` and ``5`` to ``2``.

        This allows also duplication of columns.
        """

        if not columns and not allow_empty_columns:
            raise exceptions.InvalidArgumentValueError("No columns selected.")

        # This makes a copy so that we can modify metadata in-place.
        outputs_metadata = self.update(
            (ALL_ELEMENTS,),
            {
                'dimension': {
                    'length': len(columns),
                },
            },
        )

        for element_metadata_entry in itertools.chain(
            [outputs_metadata._current_metadata.all_elements],
            outputs_metadata._current_metadata.elements.values(),
        ):
            if element_metadata_entry is None:
                continue

            elements = element_metadata_entry.elements
            new_elements_evolver = utils.EMPTY_PMAP.evolver()
            for i, column_index in enumerate(columns):
                if column_index in elements:
                    # If "column_index" is really numeric, we re-enumerate it.
                    if isinstance(column_index, int):
                        new_elements_evolver.set(i, elements[column_index])
                    else:
                        new_elements_evolver.set(column_index, elements[column_index])
            element_metadata_entry.elements = new_elements_evolver.persistent()
            element_metadata_entry.is_elements_empty = not element_metadata_entry.elements
            element_metadata_entry.update_is_empty()

        # TODO: Update boundary columns and "confidence for" references.

        return outputs_metadata

    def remove_columns(self: D, column_indices: typing.Sequence[int]) -> D:
        """
        Removes columns from metadata.

        It moves to the left metadata for columns after removed columns.
        If you do not want that, use ``remove_column``.

        It throws an exception if no columns would be left after removing columns.
        """

        columns = list(range(self.query_field((ALL_ELEMENTS,), 'dimension')['length']))

        if not columns:
            raise ValueError("No columns to remove.")

        for column_index in column_indices:
            columns.remove(column_index)

        if not columns:
            raise ValueError("Removing columns would have removed the last column.")

        # TODO: Update boundary columns and "confidence for" references.

        return self.select_columns(columns)

    def append_columns(self: D, right: D, *, use_right_metadata: bool = False) -> D:
        """
        Appends metadata for all columns from ``right`` to the right of this metadata.

        Top-level metadata of ``right`` is ignored, not merged, except if ``use_right_metadata``
        is set, in which case top-level metadata of this metadata is ignored and one from ``right`` is
        used instead.
        """

        left_length = self.query_field((ALL_ELEMENTS,), 'dimension')['length']
        right_length = right.query_field((ALL_ELEMENTS,), 'dimension')['length']

        if not use_right_metadata:
            outputs_metadata = self

            for column_index in range(right_length):
                # To go over "ALL_ELEMENTS" and all rows.
                for element in right.get_elements(()):
                    outputs_metadata = right.copy_to(outputs_metadata, [element, ALL_ELEMENTS], [element, left_length + column_index], ignore_all_elements=True)
                    outputs_metadata = right.copy_to(outputs_metadata, [element, column_index], [element, left_length + column_index], ignore_all_elements=True)

        else:
            # This makes a copy so that we can modify metadata in-place.
            outputs_metadata = right.update(
                (ALL_ELEMENTS,),
                {},
            )

            # Move columns and make space for left metadata to be prepended.
            # We iterate over a list so that we can change dict while iterating.
            for element_metadata_entry in itertools.chain(
                [outputs_metadata._current_metadata.all_elements],
                outputs_metadata._current_metadata.elements.values(),
            ):
                if element_metadata_entry is None:
                    continue

                new_elements_evolver = element_metadata_entry.elements.evolver()
                for element, metadata in element_metadata_entry.elements.items(reverse=True):
                    new_elements_evolver.remove(element)
                    new_elements_evolver.set(element + left_length, metadata)
                element_metadata_entry.elements = new_elements_evolver.persistent()
                element_metadata_entry.is_elements_empty = not element_metadata_entry.elements
                element_metadata_entry.update_is_empty()

            for column_index in range(left_length):
                # To go over "ALL_ELEMENTS" and all rows.
                for element in right.get_elements(()):
                    outputs_metadata = self.copy_to(outputs_metadata, [element, ALL_ELEMENTS], [element, column_index], ignore_all_elements=True)
                    outputs_metadata = self.copy_to(outputs_metadata, [element, column_index], [element, column_index], ignore_all_elements=True)

        outputs_metadata = outputs_metadata.update((ALL_ELEMENTS,), {'dimension': {'length': left_length + right_length}})

        # TODO: Update boundary columns and "confidence for" references.

        return outputs_metadata

    def insert_columns(self: D, columns: D, at_column_index: int) -> D:
        """
        Inserts metadata for all columns from ``columns`` before ``at_column_index`` column in this metadata,
        pushing all existing columns to the right.

        E.g., ``at_column_index == 0`` means inserting ``columns`` at the beginning of this metadata.

        Top-level metadata of ``columns`` is ignored.
        """

        columns_length = columns.query_field((ALL_ELEMENTS,), 'dimension')['length']

        if at_column_index < 0:
            raise exceptions.InvalidArgumentValueError("\"at_column_index\" is smaller than 0.")
        if at_column_index > columns_length:
            raise exceptions.InvalidArgumentValueError("\"at_column_index\" is larger than the range of existing columns.")

        if at_column_index == 0:
            return columns.append_columns(self, use_right_metadata=True)

        if at_column_index == columns_length:
            return self.append_columns(columns)

        # TODO: This could probably be optimized without all the slicing and joining.

        before = self.select_columns(list(range(0, at_column_index)))
        after = self.select_columns(list(range(at_column_index, columns_length)))

        # TODO: Update boundary columns and "confidence for" references.

        return before.append_columns(columns).append_columns(after)

    def _replace_column(self: D, column_index: int, columns: 'DataMetadata', columns_column_index: int) -> D:
        outputs_metadata = self.remove_column(column_index)

        # To go over "ALL_ELEMENTS" and all rows.
        for element in columns.get_elements(()):
            outputs_metadata = columns.copy_to(outputs_metadata, [element, ALL_ELEMENTS], [element, column_index], ignore_all_elements=True)
            outputs_metadata = columns.copy_to(outputs_metadata, [element, columns_column_index], [element, column_index], ignore_all_elements=True)

        return outputs_metadata

    def replace_columns(self: D, columns: D, column_indices: typing.Sequence[int]) -> D:
        """
        Replaces columns listed in ``column_indices`` with ``columns``, in order, in this metadata.

        ``column_indices`` and ``columns`` do not have to match in number of columns. Columns are first
        replaced in order for matching indices and columns. If then there are more ``column_indices`` than
        ``columns``, additional ``column_indices`` columns are removed. If there are more ``columns`` than
        ``column_indices`` columns, then additional ``columns`` are inserted after the last replaced column.

        If ``column_indices`` is empty, then the behavior is equivalent to calling ``append_columns``.

        Top-level metadata of ``columns`` is ignored.
        """

        # TODO: This could probably be optimized without all the slicing and joining.

        if not column_indices:
            return self.append_columns(columns)

        outputs = self
        columns_length = columns.query_field((ALL_ELEMENTS,), 'dimension')['length']
        columns_to_remove = []
        i = 0

        # This loop will run always at least once, so "column_index" will be set.
        while i < len(column_indices):
            column_index = column_indices[i]

            if i < columns_length:
                outputs = outputs._replace_column(column_index, columns, i)
            else:
                # If there are more column indices than columns in "columns", we
                # select additional columns for removal.
                columns_to_remove.append(column_index)

            i += 1

        # When there are less column indices than columns in "columns", we insert the rest after
        # the last replaced column.
        if i < columns_length:
            columns = columns.select_columns(list(range(i, columns_length)))
            # "column_index" points to the last place we inserted a column, so "+ 1" points after it.
            outputs = outputs.insert_columns(columns, column_index + 1)

        # We remove columns at the end so that we do not break and column index used before.
        # When removing columns, column indices shift.
        if columns_to_remove:
            outputs = outputs.remove_columns(columns_to_remove)

        # TODO: Update boundary columns and "confidence for" references.

        return outputs

    def _check_same_number_of_samples(self, metadata: 'DataMetadata') -> None:
        if self.query_field((), 'dimension')['length'] != metadata.query_field((), 'dimension')['length']:
            raise ValueError("Data does not match in the number of samples.")

    def get_index_columns(self, *, at: Selector = ()) -> typing.Sequence[int]:
        """
        Returns column indices of the primary index columns.

        It makes sure ``d3mIndex`` is always first listed.
        """

        index_columns = self.list_columns_with_semantic_types(('https://metadata.datadrivendiscovery.org/types/PrimaryKey', 'https://metadata.datadrivendiscovery.org/types/PrimaryMultiKey'), at=at)

        def d3m_index_first(index_column: int) -> int:
            try:
                if self.query_field((ALL_ELEMENTS, index_column), 'name') == 'd3mIndex':
                    return -1
            except KeyError:
                pass

            return 0

        return sorted(index_columns, key=d3m_index_first)

    def horizontal_concat(self: D, right: D, *, use_index: bool = True, remove_second_index: bool = True, use_right_metadata: bool = False) -> D:
        """
        Similar to ``append_columns``, but it respects primary index columns, by default.

        It is required that both inputs have the same number of samples.
        """

        self._check_same_number_of_samples(right)

        left_indices = self.get_index_columns()
        right_indices = right.get_index_columns()

        if left_indices and right_indices:
            if use_index:
                # TODO: Reorder metadata rows as well.
                #       We cannot really do this without data?
                pass

            # Removing second primary key column.
            if remove_second_index:
                right = right.remove_columns(right_indices)

        # TODO: Update boundary columns and "confidence for" references.

        return self.append_columns(right, use_right_metadata=use_right_metadata)

    def set_table_metadata(self: D, *, at: Selector = ()) -> D:
        at = list(at)

        outputs_metadata = self

        try:
            dimension = self.query_field(at + [ALL_ELEMENTS], 'dimension')
        except KeyError:
            dimension = None

        # If input is at least 2D, then we set table metadata.
        if dimension is not None:
            metadata = outputs_metadata.query(at)

            semantic_types = list(metadata.get('semantic_types', []))
            if 'https://metadata.datadrivendiscovery.org/types/Table' not in semantic_types:
                semantic_types.append('https://metadata.datadrivendiscovery.org/types/Table')

            dimension_semantic_types = list(metadata.get('dimension', {}).get('semantic_types', []))
            if 'https://metadata.datadrivendiscovery.org/types/TabularRow' not in dimension_semantic_types:
                dimension_semantic_types.append('https://metadata.datadrivendiscovery.org/types/TabularRow')
            dimension_semantic_types = [semantic_type for semantic_type in dimension_semantic_types if semantic_type not in {'https://metadata.datadrivendiscovery.org/types/TabularColumn'}]

            outputs_metadata = outputs_metadata.update(at, {
                'dimension': {
                    'name': 'rows',
                    'semantic_types': dimension_semantic_types,
                },
                'semantic_types': semantic_types,
            })

            metadata = outputs_metadata.query(at + [ALL_ELEMENTS])

            dimension_semantic_types = list(metadata.get('dimension', {}).get('semantic_types', []))
            if 'https://metadata.datadrivendiscovery.org/types/TabularColumn' not in dimension_semantic_types:
                dimension_semantic_types.append('https://metadata.datadrivendiscovery.org/types/TabularColumn')
            dimension_semantic_types = [semantic_type for semantic_type in dimension_semantic_types if semantic_type not in {'https://metadata.datadrivendiscovery.org/types/TabularRow'}]

            new_metadata: typing.Dict = {
                'dimension': {
                    'name': 'columns',
                    'semantic_types': dimension_semantic_types,
                },
            }

            if 'semantic_types' in metadata:
                new_metadata['semantic_types'] = [semantic_type for semantic_type in metadata['semantic_types'] if semantic_type not in {'https://metadata.datadrivendiscovery.org/types/Table'}]
                if not new_metadata['semantic_types']:
                    new_metadata['semantic_types'] = NO_VALUE

            outputs_metadata = outputs_metadata.update(at + [ALL_ELEMENTS], new_metadata)

            selector: ListSelector = at + [ALL_ELEMENTS, ALL_ELEMENTS]
            while True:
                try:
                    dimension = self.query_field(selector, 'dimension')
                except KeyError:
                    break

                metadata = outputs_metadata.query(selector)

                new_metadata = {}

                if 'semantic_types' in metadata:
                    new_metadata['semantic_types'] = [semantic_type for semantic_type in metadata['semantic_types'] if semantic_type not in {'https://metadata.datadrivendiscovery.org/types/Table'}]
                    if not new_metadata['semantic_types']:
                        new_metadata['semantic_types'] = NO_VALUE

                if 'semantic_types' in dimension:
                    new_metadata['dimension'] = {}

                    dimension_semantic_types = list(dimension['semantic_types'])
                    if 'https://metadata.datadrivendiscovery.org/types/TabularColumn' in dimension_semantic_types and dimension.get('name', None) == 'columns':
                        new_metadata['dimension']['name'] = NO_VALUE
                    if 'https://metadata.datadrivendiscovery.org/types/TabularRow' in dimension_semantic_types and dimension.get('name', None) == 'rows':
                        new_metadata['dimension']['name'] = NO_VALUE

                    dimension_semantic_types = [
                        semantic_type for semantic_type in dimension_semantic_types
                        if semantic_type not in {'https://metadata.datadrivendiscovery.org/types/TabularColumn', 'https://metadata.datadrivendiscovery.org/types/TabularRow'}
                    ]
                    new_metadata['dimension']['semantic_types'] = dimension_semantic_types
                    if not new_metadata['dimension']['semantic_types']:
                        new_metadata['dimension']['semantic_types'] = NO_VALUE

                if new_metadata:
                    outputs_metadata = outputs_metadata.update(selector, new_metadata)

                selector.append(ALL_ELEMENTS)

        return outputs_metadata

    def get_column_references_by_column_index(self, current_resource_id: str, *, at: Selector = ()) -> typing.Dict[str, typing.Dict[ColumnReference, typing.List[ColumnReference]]]:
        references: typing.Dict[str, typing.Dict[ColumnReference, typing.List[ColumnReference]]] = {
            'confidence_for': {},
            'rank_for': {},
            'boundary_for': {},
            'foreign_key': {},
        }

        for column_index in range(self.query_field(list(at) + [ALL_ELEMENTS], 'dimension')['length']):
            column_metadata = self.query_column(column_index, at=at)

            column_reference = ColumnReference(current_resource_id, column_index)

            if 'confidence_for' in column_metadata and 'column_indices' in column_metadata['confidence_for']:
                reference_resource_id = column_metadata['confidence_for'].get('resource_id', current_resource_id)

                references['confidence_for'][column_reference] = [
                    ColumnReference(reference_resource_id, reference_column_index)
                    for reference_column_index in column_metadata['confidence_for']['column_indices']
                ]

            if 'rank_for' in column_metadata and 'column_indices' in column_metadata['rank_for']:
                reference_resource_id = column_metadata['rank_for'].get('resource_id', current_resource_id)

                references['rank_for'][column_reference] = [
                    ColumnReference(reference_resource_id, reference_column_index)
                    for reference_column_index in column_metadata['rank_for']['column_indices']
                ]

            if 'boundary_for' in column_metadata and 'column_index' in column_metadata['boundary_for']:
                reference_resource_id = column_metadata['boundary_for'].get('resource_id', current_resource_id)

                references['boundary_for'][column_reference] = [
                    ColumnReference(reference_resource_id, column_metadata['boundary_for']['column_index']),
                ]

            if 'foreign_key' in column_metadata and column_metadata['foreign_key']['type'] == 'COLUMN' and 'column_index' in column_metadata['foreign_key']:
                reference_resource_id = column_metadata['foreign_key']['resource_id']

                references['foreign_key'][column_reference] = [
                    ColumnReference(reference_resource_id, column_metadata['foreign_key']['column_index']),
                ]

        return references


class PrimitiveMetadata(Metadata):
    """
    A class for metadata for primitives.

    It checks all updates against primitive schema. Note that as such empty (just created) metadata object
    does not validate against the schema. If an instance is set on a primitive class, primitive's metaclass
    logic will automatically link metadata object with the primitive class and generate required metadata.
    """

    def __init__(self, metadata: typing.Dict[str, typing.Any] = None) -> None:
        super().__init__(metadata=metadata)

        # We do not do validation here because provided metadata on its own is
        # probably not sufficient for validation to pass. Validation happens
        # inside "contribute_to_class" method instead.

        # Importing here to prevent import cycle.
        from d3m.primitive_interfaces import base

        self.primitive: typing.Type[base.PrimitiveBase] = None

    # Not adhering to Liskov substitution principle: we do not have "selector" argument.
    @deprecate.arguments('source', 'timestamp', message="argument ignored")
    def update(self: P, metadata: typing.Dict[str, typing.Any], *, source: typing.Any = None, timestamp: datetime.datetime = None) -> P:  # type: ignore
        new_metadata = super().update(selector=(), metadata=metadata)

        self._validate(new_metadata.query())

        return new_metadata

    @deprecate.function(message="create a PrimitiveMetadata instance explicitly instead")
    @deprecate.arguments('source', 'timestamp', message="argument ignored")
    def clear(self: P, metadata: typing.Dict[str, typing.Any] = None, *, source: typing.Any = None, timestamp: datetime.datetime = None) -> P:
        return super().clear(metadata=metadata)

    # Not adhering to Liskov substitution principle: we do not have "selector" argument.
    def query(self) -> frozendict.FrozenOrderedDict:  # type: ignore
        return super().query(selector=())

    # "primitive" should be of PrimitiveBase here, but we do not want to introduce a
    # cyclic dependency. We validate the type at runtime in the method.
    def contribute_to_class(self: P, primitive: typing.Any) -> None:
        # Importing here to prevent import cycle.
        from d3m.primitive_interfaces import base

        if self.primitive is not None:
            raise exceptions.InvalidStateError("Primitive is already set to '{primitive}'.".format(primitive=self.primitive))

        if not issubclass(primitive, base.PrimitiveBase):
            raise exceptions.InvalidArgumentTypeError("Primitive argument is not a subclass of 'PrimitiveBase' class.")

        self.primitive = primitive

        self._generate_and_update()

    @classmethod
    def _validate_contact_information(cls, metadata: typing.Dict) -> None:
        # See https://gitlab.com/datadrivendiscovery/d3m/issues/178 for motivation for this check.

        # If it is a locally registered/used primitive, we do not validate contact information.
        if 'installation' not in metadata:
            return

        if 'source' not in metadata:
            logger.warning(
                "%(python_path)s: No \"source\" field in the primitive metadata. Metadata should contain contact information and bug reporting URI.",
                {
                    'python_path': metadata['python_path'],
                },
            )
            return

        if not metadata['source'].get('contact', None):
            logger.warning(
                "%(python_path)s: Contact information such as the email address of the author "
                "(e.g., \"mailto:author@example.com\") should be specified in primitive metadata in its \"source.contact\" field.",
                {
                    'python_path': metadata['python_path'],
                },
            )

        # If the list is empty, it is also false.
        if not metadata['source'].get('uris', None):
            logger.warning(
                "%(python_path)s: A bug reporting URI should be specified in primitive metadata in its \"source.uris\" field.",
                {
                    'python_path': metadata['python_path'],
                },
            )

    # Make sure a primitive provides a description (through docstring). Because we use special metaclass
    # which inherits description from a base class, we have to check the description itself.
    # See: https://gitlab.com/datadrivendiscovery/d3m/issues/167
    @classmethod
    def _validate_description(cls, metadata: typing.Dict) -> None:
        # Importing here to prevent import cycle.
        from d3m.primitive_interfaces import base

        if 'description' not in metadata or not metadata['description'] or metadata['description'].startswith(base.DEFAULT_DESCRIPTION):
            logger.warning(
                "%(python_path)s: Primitive is not providing a description through its docstring.",
                {
                    'python_path': metadata['python_path'],
                },
            )

    # Checks that the primitive's Python path complies with namespace requirements.
    # See: https://gitlab.com/datadrivendiscovery/d3m/issues/3
    @classmethod
    def _validate_namespace_compliance(cls, python_path: str, primitive_family: typing.Union[PrimitiveFamily, str]) -> None:  # type: ignore
        segments = python_path.split('.')

        if len(segments) != 5:
            logger.warning(
                "%(python_path)s: Primitive's Python path does not adhere to d3m.primitives namespace specification. "
                "Reason: must have 5 segments.",
                {
                    'python_path': python_path,
                },
            )
        else:
            if segments[0] != 'd3m' or segments[1] != 'primitives':
                logger.warning(
                    "%(python_path)s: Primitive's Python path does not adhere to d3m.primitives namespace specification. "
                    "Reason: must start with \"d3m.primitives\".",
                    {
                        'python_path': python_path,
                    },
                )

            family = segments[2]
            name = segments[3]
            kind = segments[4]

            # "primitive_family" could also already be a string.
            if isinstance(primitive_family, str):
                primitive_family_name = primitive_family
            else:
                primitive_family_name = primitive_family.name

            if family != primitive_family_name.lower():  # type: ignore
                logger.warning(
                    "%(python_path)s: Primitive's Python path does not adhere to d3m.primitives namespace specification. "
                    "Reason: primitive family segment must match primitive's primitive family.",
                    {
                        'python_path': python_path,
                    },
                )

            if name not in primitive_names.PRIMITIVE_NAMES:
                logger.warning(
                    "%(python_path)s: Primitive's Python path does not adhere to d3m.primitives namespace specification. "
                    "Reason: must have a known primitive name segment.",
                    {
                        'python_path': python_path,
                    },
                )

            if not kind[0].isupper():
                logger.warning(
                    "%(python_path)s: Primitive's Python path does not adhere to d3m.primitives namespace specification. "
                    "Reason: primitive kind segment must start with upper case.",
                    {
                        'python_path': python_path,
                    },
                )

    @classmethod
    def _validate(cls, metadata: typing.Dict) -> None:
        PRIMITIVE_SCHEMA_VALIDATOR.validate(metadata)

        cls._validate_installation(metadata)
        cls._validate_volumes(metadata)
        cls._validate_docker_containers(metadata)
        cls._validate_hyperparams_to_tune(metadata)
        cls._validate_optional_constructor_arguments(metadata)
        #cls._validate_namespace_compliance(metadata['python_path'], metadata['primitive_family'])
        cls._validate_contact_information(metadata)
        cls._validate_description(metadata)

    def _generate_and_update(self) -> None:
        generated_metadata = self._generate_metadata_for_primitive()

        self._update_in_place((), generated_metadata, self._current_metadata)

        self._validate(self.query())

    @classmethod
    def _validate_installation(cls, metadata: typing.Dict) -> None:
        for entry in metadata.get('installation', []):
            # We can check simply equality because metadata enumerations are equal to strings as well,
            # and "entry['type']" can be both a string or an enumeration instance.
            if entry['type'] != PrimitiveInstallationType.PIP:
                continue

            if 'package' in entry:
                if '/' in entry['package']:
                    raise exceptions.InvalidMetadataError("Invalid package name '{package_name}'. If you want to use an URI pointing to a package, use 'package_uri' instead.".format(
                        package_name=entry['package'],
                    ))

                continue

            if 'package_uri' not in entry:
                continue

            if entry['package_uri'].startswith('git+git@'):
                # "git+git@git.myproject.org:MyProject" format cannot be parsed with urlparse.
                raise exceptions.InvalidMetadataError("Only git+http and git+https URI schemes are allowed.")

            parsed_uri = url_parse.urlparse(entry['package_uri'])

            # It is not a git pip URI. For now we then do not validate it.
            if not parsed_uri.scheme.startswith('git'):
                continue

            if parsed_uri.scheme not in ['git+http', 'git+https']:
                raise exceptions.InvalidMetadataError("Only git+http and git+https URI schemes are allowed.")

            if '@' not in parsed_uri.path:
                raise exceptions.InvalidMetadataError("Package URI does not include a commit hash: {package_uri}".format(package_uri=entry['package_uri']))

            path, commit_hash = parsed_uri.path.rsplit('@', 1)

            if not COMMIT_HASH_REGEX.match(commit_hash):
                raise exceptions.InvalidMetadataError("Package URI does not include a commit hash: {package_uri}".format(package_uri=entry['package_uri']))

            if not parsed_uri.fragment:
                raise exceptions.InvalidMetadataError("Package URI does not include a '#egg=package_name' URI suffix.")

            parsed_fragment = url_parse.parse_qs(parsed_uri.fragment, strict_parsing=True)

            if 'egg' not in parsed_fragment:
                raise exceptions.InvalidMetadataError("Package URI does not include a '#egg=package_name' URI suffix.")

    @classmethod
    def _validate_optional_constructor_arguments(cls, metadata: typing.Dict) -> None:
        installation = metadata.get('installation', [])

        containers = [entry for entry in installation if entry.get('type', None) == PrimitiveInstallationType.DOCKER]
        if containers and 'docker_containers' not in metadata['primitive_code'].get('instance_methods', {})['__init__']['arguments']:
            raise exceptions.InvalidPrimitiveCodeError("Primitive defines a Docker container dependency but does not accept 'docker_containers' argument to the constructor.")

        volumes = cls._get_volumes(metadata)
        if volumes and 'volumes' not in metadata['primitive_code'].get('instance_methods', {})['__init__']['arguments']:
            raise exceptions.InvalidPrimitiveCodeError("Primitive defines a volume dependency but does not accept 'volumes' argument to the constructor.")

    @classmethod
    def _validate_hyperparams_to_tune(cls, metadata: typing.Dict) -> None:
        hyperparams = metadata['primitive_code'].get('hyperparams', {})

        for name in metadata.get('hyperparams_to_tune', []):
            if name not in hyperparams:
                raise exceptions.InvalidMetadataError("Hyper-parameter in 'hyperparams_to_tune' metadata does not exist: {name}".format(name=name))

    def _generate_metadata_for_primitive(self) -> typing.Dict[str, typing.Any]:
        # Importing here to prevent import cycle.
        from d3m.primitive_interfaces import base

        type_arguments = self._get_type_arguments()
        class_attributes = self._get_class_attributes()
        hyperparams_class = typing.cast(typing.Type[hyperparams_module.Hyperparams], type_arguments[base.Hyperparams])
        arguments, instance_methods = self._get_arguments_and_methods(hyperparams_class, type_arguments)
        self._validate_constructor(instance_methods)
        self._validate_multi_produce(instance_methods)
        self._validate_fit_multi_produce(instance_methods)
        hyperparams = self._get_hyperparams(hyperparams_class)
        class_methods = self._get_class_methods(type_arguments)
        instance_attributes = self._get_instance_attributes()
        params = self._get_params(type_arguments)

        # Sanity check.
        hyperparams_keys = set(hyperparams.keys())
        # We can check simply equality because metadata enumerations are equal to strings as well,
        # and "argument['kind']" can be both a string or an enumeration instance.
        non_hyperparameter_arguments_keys = {name for name, argument in arguments.items() if argument['kind'] != PrimitiveArgumentKind.HYPERPARAMETER}
        overlapping_keys = hyperparams_keys & non_hyperparameter_arguments_keys
        if len(overlapping_keys):
            raise exceptions.InvalidPrimitiveCodeError("Hyper-paramater names are overlapping with non-hyperparameter argument names: {overlapping_keys}".format(overlapping_keys=overlapping_keys))

        primitive_code = {
            # We have to convert parameters to their names because JSON schema supports only strings for keys.
            'class_type_arguments': {parameter.__name__: argument for parameter, argument in type_arguments.items()},
            'interfaces_version': d3m.__version__,
            'interfaces': self._get_interfaces(),
            'hyperparams': hyperparams,
            'arguments': arguments,
            'class_methods': class_methods,
            'instance_methods': instance_methods,
            'class_attributes': class_attributes,
            'instance_attributes': instance_attributes,
        }

        if params is not None:
            primitive_code['params'] = params

        result = {
            'schema': PRIMITIVE_SCHEMA_VERSION,
            'original_python_path': '{module}.{class_name}'.format(
                module=self.primitive.__module__,
                class_name=self.primitive.__name__,
            ),
            'primitive_code': primitive_code,
            'structural_type': self.primitive,
        }

        description = inspect.cleandoc(getattr(self.primitive, '__doc__', None) or '') or None
        if description is not None:
            result['description'] = description

        digest = self._compute_primitive_digest()
        if digest is not None:
            result['digest'] = digest

        return result

    def _compute_primitive_digest(self) -> typing.Optional[str]:
        primitive_metadata = self.query()

        # We use installation metadata for digest because it uniquely identifies the content of the primitive.
        # TODO: Some primitives install extra code/data from their setup.py during installation. Could we capture that with digest as well?
        installation = primitive_metadata.get('installation', None)

        if not installation:
            return None

        # We use "to_json_structure" here and not "to_reversible_json_structure"
        # because pickled values might not be deterministic.
        to_digest = utils.to_json_structure({
            # We include primitive ID as well, so that different primitives
            # from the same package do not have the same digest.
            'id': primitive_metadata['id'],
            'installation': installation,
        })

        return utils.compute_digest(to_digest)

    # Using typing.TypeVar in type signature does not really work, so we are using type instead.
    # See: https://github.com/python/typing/issues/520
    def _get_type_arguments(self) -> typing.Dict[type, type]:
        # Importing here to prevent import cycle.
        from d3m.primitive_interfaces import base

        # This call also catches if type parameter has been overridden with a new type variable.
        # This means that we for free get to make sure type parameters from the base class stay
        # as they are expected to be. It also fetches them recursively, so one cannot hide a
        # type parameter (but can fix it to a fixed type instead of leaving it open for a
        # subclass to choose it).
        type_arguments = utils.get_type_arguments(self.primitive, unique_names=True)

        for parameter, argument in type_arguments.items():
            # Params type argument is optional and can be set to None.
            if parameter == base.Params and issubclass(argument, type(None)):
                continue

            if not utils.is_subclass(argument, parameter):
                raise exceptions.InvalidPrimitiveCodeError("Type parameter '{name}' has type '{type}' and not an expected type: {expected}".format(
                    name=parameter.__name__, type=argument, expected=parameter.__bound__,  # type: ignore
                ))

        return type_arguments

    def _resolve_type(self, obj: type, type_arguments: typing.Dict[type, type]) -> type:
        if obj in type_arguments:
            return type_arguments[obj]
        else:
            return obj

    def _get_interfaces(self) -> typing.Tuple[str, ...]:
        mro = [parent for parent in inspect.getmro(self.primitive) if parent.__module__.startswith('d3m.primitive_interfaces.')]

        interfaces: typing.List[str] = []
        for parent in mro:
            interface = utils.get_full_name(parent)
            # Remove package name.
            interface = '.'.join(interface.split('.')[2:])
            if interface not in interfaces:
                interfaces.append(interface)

        if not len(interfaces):
            raise exceptions.InvalidPrimitiveCodeError("The primitive does not implement a standard interface.")

        return tuple(interfaces)

    # Using typing.TypeVar in type signature does not really work, so we are using type instead.
    # See: https://github.com/python/typing/issues/520
    def _get_params(self, type_arguments: typing.Dict[type, type]) -> typing.Optional[typing.Dict[str, type]]:
        # Importing here to prevent import cycle.
        from d3m.primitive_interfaces import base

        params = type_arguments.get(base.Params, type(None))

        if issubclass(params, type(None)):
            return None

        return params.__params_items__  # type: ignore

    def _get_hyperparams(self, hyperparams_class: 'typing.Type[hyperparams_module.Hyperparams]') -> typing.Dict[str, typing.Dict]:
        # We check this here and not during hyper-parameter construction itself because
        # we want to require this only once it is used with a primitive. Hyper-parameters
        # might be used and constructed in other settings as well.
        for hyperparameter_name, hyperparameter in hyperparams_class.configuration.items():
            if not set(hyperparameter.semantic_types) & HYPERPARAMETER_REQUIRED_SEMANTIC_TYPES:
                raise exceptions.InvalidPrimitiveCodeError(
                    "Hyper-parameter '{hyperparameter_name}' does not contain any of required semantic types: {required}".format(
                        hyperparameter_name=hyperparameter_name,
                        required=sorted(HYPERPARAMETER_REQUIRED_SEMANTIC_TYPES),
                    ),
                )

        return hyperparams_class.to_simple_structure()

    def _get_class_attributes(self) -> typing.Dict[str, type]:
        result = {}

        for attribute_name, attribute in inspect.getmembers(self.primitive):
            if attribute_name.startswith('_'):
                continue

            if utils.is_class_method_on_class(attribute) or utils.is_instance_method_on_class(attribute):
                continue

            result[attribute_name] = type(attribute)

        result_keys = set(result.keys())
        expected_result_keys = set(EXPECTED_CLASS_ATTRIBUTES.keys())

        missing = expected_result_keys - result_keys
        if len(missing):
            raise exceptions.InvalidPrimitiveCodeError("Not all expected public class attributes exist: {missing}".format(missing=missing))

        extra = result_keys - expected_result_keys
        if len(extra):
            raise exceptions.InvalidPrimitiveCodeError("Additional unexpected public class attributes exist, consider making them private by prefixing them with '_': {extra}".format(extra=extra))

        for attribute_name, attribute in result.items():
            if not utils.is_subclass(attribute, EXPECTED_CLASS_ATTRIBUTES[attribute_name]):
                raise exceptions.InvalidPrimitiveCodeError("Class attribute '{attribute_name}' does not have an expected type.".format(attribute_name=attribute_name))

        return result

    # Using typing.TypeVar in type signature does not really work, so we are using type instead.
    # See: https://github.com/python/typing/issues/520
    def _get_arguments_and_methods(
        self, hyperparams_class: 'typing.Type[hyperparams_module.Hyperparams]', type_arguments: typing.Dict[type, type],
    ) -> typing.Tuple[typing.Dict[str, typing.Dict], typing.Dict[str, typing.Dict]]:
        # Importing here to prevent import cycle.
        from d3m.primitive_interfaces import base
        from d3m import types as types_module

        arguments: typing.Dict[str, typing.Dict] = {}
        methods: typing.Dict[str, typing.Dict] = {}

        for method_name, method in inspect.getmembers(self.primitive):
            if method_name.startswith('_') and method_name != '__init__':
                continue

            if not utils.is_instance_method_on_class(method):
                continue

            # To make get_type_hints find method's module while the primitive's
            # module is still being defined (and this method was indirectly called
            # from primitive's metaclass).
            method.im_class = self.primitive

            type_hints = utils.get_type_hints(method)

            if not type_hints:
                raise exceptions.InvalidPrimitiveCodeError("Cannot get types for method '{method_name}'.".format(method_name=method_name))

            if 'return' not in type_hints:
                raise exceptions.InvalidPrimitiveCodeError("Method '{method_name}' is missing a type for the return value.".format(method_name=method_name))

            if method_name.startswith('produce_') or method_name == 'produce':
                method_kind = PrimitiveMethodKind.PRODUCE

                if getattr(method, '__singleton__', False):
                    singleton_produce_method = True
                else:
                    singleton_produce_method = False

                method_inputs_across_samples = getattr(method, '__inputs_across_samples__', ())
            elif method_name.startswith('produce'):
                raise exceptions.InvalidPrimitiveCodeError("Produce method should start with 'produce_' and not be '{method_name}'.".format(method_name=method_name))
            else:
                method_kind = PrimitiveMethodKind.OTHER

                singleton_produce_method = None
                method_inputs_across_samples = None

                if hasattr(method, '__singleton__'):
                    raise exceptions.InvalidPrimitiveCodeError("Only produce methods can be set as singleton or not: {method_name}.".format(method_name=method_name))
                if hasattr(method, '__inputs_across_samples__'):
                    raise exceptions.InvalidPrimitiveCodeError("Only arguments of produce methods can be set to compute accross samples or not: {method_name}.".format(method_name=method_name))

            method_arguments = []

            # We skip the first argument (self).
            for argument_name, argument in list(inspect.signature(method).parameters.items())[1:]:
                if argument.kind != inspect.Parameter.KEYWORD_ONLY:
                    raise exceptions.InvalidPrimitiveCodeError("Method '{method_name}' has a non-keyword argument '{argument_name}'.".format(method_name=method_name, argument_name=argument_name))

                has_default = argument.default is not inspect.Parameter.empty

                if argument_name.startswith('_'):
                    if not has_default:
                        raise exceptions.InvalidPrimitiveCodeError("Method '{method_name}' has a non-optional private argument '{argument_name}'.".format(
                            method_name=method_name, argument_name=argument_name,
                        ))

                    continue

                if not ARGUMENT_NAME_REGEX.match(argument_name):
                    raise exceptions.InvalidPrimitiveCodeError("Method '{method_name}' has an argument with an invalid name '{argument_name}'.".format(
                        method_name=method_name, argument_name=argument_name
                    ))

                if argument_name not in type_hints:
                    raise exceptions.InvalidPrimitiveCodeError("Method '{method_name}' is missing a type for argument '{argument_name}'.".format(method_name=method_name, argument_name=argument_name))

                argument_type = self._resolve_type(type_hints[argument_name], type_arguments)

                standard_argument_description = typing.cast(
                    typing.Dict,
                    STANDARD_RUNTIME_ARGUMENTS.get(argument_name, None) or STANDARD_PIPELINE_ARGUMENTS.get(argument_name, None),
                )
                if standard_argument_description is not None:
                    try:
                        expected_type = self._get_argument_type(standard_argument_description, type_arguments)
                    except KeyError:
                        raise exceptions.InvalidPrimitiveCodeError(
                            "Method '{method_name}' has an argument '{argument_name}' for which an expected type cannot be determined. Is a type parameter missing?".format(
                                method_name=method_name, argument_name=argument_name,
                            )
                        )

                    # Types have to match here exactly. This is what class type arguments are for.
                    if argument_type != expected_type:
                        raise exceptions.InvalidPrimitiveCodeError(
                            "Method '{method_name}' has an argument '{argument_name}' with type '{argument_type}' and not an expected type: {expected_type}".format(
                                method_name=method_name, argument_name=argument_name,
                                argument_type=argument_type, expected_type=expected_type,
                            )
                        )

                    if 'default' in standard_argument_description:
                        if not has_default:
                            raise exceptions.InvalidPrimitiveCodeError(
                                "Method '{method_name}' has an argument '{argument_name}' which does not have a default value, but it should.".format(
                                    method_name=method_name, argument_name=argument_name,
                                )
                            )

                        if argument.default != standard_argument_description['default']:
                            raise exceptions.InvalidPrimitiveCodeError(
                                "Method '{method_name}' has an argument '{argument_name}' with a different default value: {argument_default} != {expected_default}.".format(
                                    method_name=method_name, argument_name=argument_name,
                                    argument_default=argument.default, expected_default=standard_argument_description['default'],
                                )
                            )

                    else:
                        if has_default:
                            raise exceptions.InvalidPrimitiveCodeError("Method '{method_name}' has an argument '{argument_name}' which has a default value, but it should not.".format(
                                method_name=method_name, argument_name=argument_name,
                            ))

                    if argument_name in STANDARD_RUNTIME_ARGUMENTS:
                        argument_kind = PrimitiveArgumentKind.RUNTIME
                    else:
                        assert argument_name in STANDARD_PIPELINE_ARGUMENTS, "argument_name not in STANDARD_PIPELINE_ARGUMENTS"
                        argument_kind = PrimitiveArgumentKind.PIPELINE

                # Constructor cannot have additional non-private custom arguments.
                elif method_name == '__init__':
                    raise exceptions.InvalidPrimitiveCodeError(
                        "Constructor cannot have non-private custom arguments, but it has an argument '{argument_name}'.".format(
                            argument_name=argument_name,
                        )
                    )

                elif argument_name in hyperparams_class.configuration:
                    # Types have to match here exactly.
                    if argument_type != hyperparams_class.configuration[argument_name].structural_type:
                        raise exceptions.InvalidPrimitiveCodeError("Method '{method_name}' has an argument '{argument_name}' overriding a hyper-parameter with a different type: {argument_type} != {hyperparameter_type}.".format(  # noqa
                            method_name=method_name, argument_name=argument_name,
                            argument_type=argument_type, hyperparameter_type=hyperparams_class.configuration[argument_name].structural_type,
                        ))

                    # Arguments overriding a hyper-parameter should not have a default value and caller should pass a value in.
                    if has_default:
                        raise exceptions.InvalidPrimitiveCodeError(
                            "Method '{method_name}' has an argument '{argument_name}' overriding a hyper-parameter which has a default value, but it should not.".format(
                                method_name=method_name, argument_name=argument_name,
                            )
                        )

                    argument_kind = PrimitiveArgumentKind.HYPERPARAMETER

                else:
                    # Any other argument should be something the rest of the pipeline can provide:
                    # a container value, data value, or another primitive.
                    expected_types: typing.Tuple[type, ...] = types_module.Container + types_module.Data + (base.PrimitiveBase,)

                    if not utils.is_subclass(argument_type, typing.Union[expected_types]):
                        raise exceptions.InvalidPrimitiveCodeError(
                            "Method '{method_name}' has an argument '{argument_name}' with type '{argument_type}' and not an expected type: {expected_types}".format(
                                method_name=method_name, argument_name=argument_name,
                                argument_type=argument_type, expected_types=expected_types
                            )
                        )

                    # It should not have a default. Otherwise it is easy to satisfy the argument
                    # (just never connect anything to it in the pipeline).
                    if has_default:
                        raise exceptions.InvalidPrimitiveCodeError("Method '{method_name}' has an argument '{argument_name}' which has a default value, but it should not.".format(
                            method_name=method_name, argument_name=argument_name,
                        ))

                    argument_kind = PrimitiveArgumentKind.PIPELINE

                method_arguments.append(argument_name)

                if argument_name in arguments:
                    if argument_type != arguments[argument_name]['type']:
                        raise exceptions.InvalidPrimitiveCodeError("Method '{method_name}' has an argument '{argument_name}' which does not match a type of a previous argument with the same name: {argument_type} != {previous_type}".format(  # noqa
                            method_name=method_name, argument_name=argument_name,
                            argument_type=argument_type, previous_type=arguments[argument_name]['type'],
                        ))

                    # This should hold because it depends only on the argument name.
                    assert argument_kind == arguments[argument_name]['kind'], "argument_kind mismatch"

                    if has_default:
                        if 'default' not in arguments[argument_name]:
                            raise exceptions.InvalidPrimitiveCodeError("Method '{method_name}' has an argument '{argument_name}' which has a default value, but a previous argument with the same name did not have a default value.".format(  # noqa
                                method_name=method_name, argument_name=argument_name,
                            ))
                        elif argument.default != arguments[argument_name]['default']:
                            raise exceptions.InvalidPrimitiveCodeError("Method '{method_name}' has an argument '{argument_name}' which does not have the same default value as a previous argument with the same name: {argument_default} != {previous_default}".format(  # noqa
                                method_name=method_name, argument_name=argument_name,
                                argument_default=argument.default,
                                previous_default=arguments[argument_name]['default'],
                            ))
                    else:
                        if 'default' in arguments[argument_name]:
                            raise exceptions.InvalidPrimitiveCodeError("Method '{method_name}' has an argument '{argument_name}' which does not have a default value, but a previous argument with the same name had a default value.".format(  # noqa
                                method_name=method_name, argument_name=argument_name,
                            ))

                else:
                    arguments[argument_name] = {
                        'type': argument_type,
                        'kind': argument_kind,
                    }

                    if has_default:
                        arguments[argument_name]['default'] = argument.default

            methods[method_name] = {
                'kind': method_kind,
                'arguments': method_arguments,
                'returns': self._resolve_type(type_hints['return'], type_arguments),
            }

            if singleton_produce_method is not None:
                methods[method_name]['singleton'] = singleton_produce_method

            if method_inputs_across_samples is not None:
                for method_input in method_inputs_across_samples:
                    if method_input not in method_arguments:
                        raise exceptions.InvalidPrimitiveCodeError("Method '{method_name}' has an argument '{method_input}' set as computing across samples, but it does not exist.".format(
                            method_name=method_name, method_input=method_input,
                        ))

                    if arguments[method_input]['kind'] != PrimitiveArgumentKind.PIPELINE:
                        raise exceptions.InvalidPrimitiveCodeError("Method '{method_name}' has an argument '{method_input}' set as computing across samples, but it is not a PIPELINE argument.".format(
                            method_name=method_name, method_input=method_input,
                        ))

                methods[method_name]['inputs_across_samples'] = method_inputs_across_samples

            description = inspect.cleandoc(getattr(method, '__doc__', None) or '') or None
            if description is not None:
                methods[method_name]['description'] = description

        return arguments, methods

    # Using typing.TypeVar in type signature does not really work, so we are using type instead.
    # See: https://github.com/python/typing/issues/520
    def _get_argument_type(self, argument_description: typing.Dict[str, typing.Any], type_arguments: typing.Dict[type, type]) -> type:
        if 'get_type' in argument_description:
            return argument_description['get_type'](type_arguments)
        else:
            return argument_description['type']

    # Using typing.TypeVar in type signature does not really work, so we are using type instead.
    # See: https://github.com/python/typing/issues/520
    def _get_class_methods(self, type_arguments: typing.Dict[type, type]) -> typing.Dict[str, typing.Dict]:
        methods: typing.Dict[str, typing.Dict] = {}

        for method_name, method in inspect.getmembers(self.primitive):
            if method_name.startswith('_'):
                continue

            if not utils.is_class_method_on_class(method):
                continue

            type_hints = utils.get_type_hints(method)

            if not type_hints:
                raise exceptions.InvalidPrimitiveCodeError("Cannot get types for method '{method_name}'.".format(method_name=method_name))

            if 'return' not in type_hints:
                raise exceptions.InvalidPrimitiveCodeError("Method '{method_name}' is missing a type for the return value.".format(method_name=method_name))

            method_arguments = {}

            for argument_name, argument in inspect.signature(method).parameters.items():
                if argument.kind != inspect.Parameter.KEYWORD_ONLY:
                    raise exceptions.InvalidPrimitiveCodeError("Method '{method_name}' has a non-keyword argument '{argument_name}'.".format(method_name=method_name, argument_name=argument_name))

                has_default = argument.default is not inspect.Parameter.empty

                if argument_name.startswith('_'):
                    if not has_default:
                        raise exceptions.InvalidPrimitiveCodeError("Method '{method_name}' has a non-optional private argument '{argument_name}'.".format(
                            method_name=method_name, argument_name=argument_name,
                        ))

                    continue

                if argument_name not in type_hints:
                    raise exceptions.InvalidPrimitiveCodeError("Method '{method_name}' is missing a type for argument '{argument_name}'.".format(method_name=method_name, argument_name=argument_name))

                argument_type = self._resolve_type(type_hints[argument_name], type_arguments)

                argument_description = {
                    'type': argument_type,
                }

                if has_default:
                    argument_description['default'] = argument.default

                method_arguments[argument_name] = argument_description

            methods[method_name] = {
                'arguments': method_arguments,
                'returns': self._resolve_type(type_hints['return'], type_arguments),
            }

            description = inspect.cleandoc(getattr(method, '__doc__', None) or '') or None
            if description is not None:
                methods[method_name]['description'] = description

        return methods

    @classmethod
    def _validate_docker_containers(cls, metadata: typing.Dict) -> None:
        installation = metadata.get('installation', [])

        containers: typing.List[str] = []

        for entry in installation:
            # We can check simply equality because metadata enumerations are equal to strings as well,
            # and "entry['type']" can be both a string or an enumeration instance.
            if entry.get('type', None) != PrimitiveInstallationType.DOCKER:
                continue

            key = entry.get('key', None)
            if key:
                containers.append(key)

        containers_set = set(containers)
        if len(containers_set) != len(containers):
            for key in containers_set:
                containers.remove(key)
            raise exceptions.InvalidMetadataError("Same Docker image key reused across multiple installation entries: {extra_keys}".format(extra_keys=containers))

    @classmethod
    def _validate_volumes(cls, metadata: typing.Dict) -> None:
        volumes: typing.List[str] = []

        for entry in cls._get_volumes(metadata):
            volumes.append(entry['key'])

        volumes_set = set(volumes)
        if len(volumes_set) != len(volumes):
            for key in volumes_set:
                volumes.remove(key)
            raise exceptions.InvalidMetadataError("Same volume key reused across multiple installation entries: {extra_keys}".format(extra_keys=volumes))

    def _validate_constructor(self, instance_methods: typing.Dict[str, typing.Dict]) -> None:
        if '__init__' not in instance_methods:
            raise exceptions.InvalidPrimitiveCodeError("Constructor is missing.")

        if 'hyperparams' not in instance_methods['__init__']['arguments']:
            raise exceptions.InvalidPrimitiveCodeError("Constructor's argument 'hyperparams' is required.")

    def _validate_multi_produce(self, instance_methods: typing.Dict[str, typing.Dict]) -> None:
        if 'produce' not in instance_methods:
            raise exceptions.InvalidPrimitiveCodeError("'produce' method is missing.")

        if 'multi_produce' not in instance_methods:
            raise exceptions.InvalidPrimitiveCodeError("'multi_produce' method is missing.")

        # Initialize with runtime arguments.
        expected_arguments = {'produce_methods', 'timeout', 'iterations'}
        for method_name, method in instance_methods.items():
            if method['kind'] != PrimitiveMethodKind.PRODUCE:
                continue

            if 'produce_methods' in method['arguments']:
                raise exceptions.InvalidPrimitiveCodeError("Produce method cannot use 'produce_methods' argument: {method_name}".format(method_name=method_name))

            expected_arguments.update(method['arguments'])

        arguments = set(instance_methods['multi_produce']['arguments'])

        missing = expected_arguments - arguments
        if len(missing):
            raise exceptions.InvalidPrimitiveCodeError(
                "'multi_produce' method arguments have to be an union of all arguments of all produce methods, but it does not accept all expected arguments: {missing}".format(
                    missing=missing,
                )
            )

        extra = arguments - expected_arguments
        if len(extra):
            raise exceptions.InvalidPrimitiveCodeError(
                "'multi_produce' method arguments have to be an union of all arguments of all produce methods, but it accepts unexpected arguments: {extra}".format(
                    extra=extra,
                )
            )

    def _validate_fit_multi_produce(self, instance_methods: typing.Dict[str, typing.Dict]) -> None:
        if 'set_training_data' not in instance_methods:
            raise exceptions.InvalidPrimitiveCodeError("'set_training_data' method is missing.")

        if 'produce' not in instance_methods:
            raise exceptions.InvalidPrimitiveCodeError("'produce' method is missing.")

        if 'fit_multi_produce' not in instance_methods:
            raise exceptions.InvalidPrimitiveCodeError("'fit_multi_produce' method is missing.")

        # Initialize with runtime arguments.
        expected_arguments = {'produce_methods', 'timeout', 'iterations'}
        for method_name, method in instance_methods.items():
            if method['kind'] == PrimitiveMethodKind.PRODUCE:
                if 'produce_methods' in method['arguments']:
                    raise exceptions.InvalidPrimitiveCodeError("Produce method cannot use 'produce_methods' argument: {method_name}".format(method_name=method_name))

                expected_arguments.update(method['arguments'])

            elif method_name == 'set_training_data':
                if 'produce_methods' in method['arguments']:
                    raise exceptions.InvalidPrimitiveCodeError("'set_training_data' method cannot use 'produce_methods' argument: {method_name}".format(method_name=method_name))

                expected_arguments.update(method['arguments'])

        arguments = set(instance_methods['fit_multi_produce']['arguments'])

        missing = expected_arguments - arguments
        if len(missing):
            raise exceptions.InvalidPrimitiveCodeError(
                "'fit_multi_produce' method arguments have to be an union of all arguments of 'set_training_data' method and all produce methods, "
                "but it does not accept all expected arguments: {missing}".format(
                    missing=missing,
                )
            )

        extra = arguments - expected_arguments
        if len(extra):
            raise exceptions.InvalidPrimitiveCodeError(
                "'fit_multi_produce' method arguments have to be an union of all arguments of 'set_training_data' method and all produce methods, but it accepts unexpected arguments: {extra}".format(
                    extra=extra,
                )
            )

    # In the past we have validated instance attributes by creating an instance of the primitive and observe
    # which instance attributes were created in a constructor. This was potentially resource intensive because
    # primitives use constructor to initialize resources they use. Moreover, it did not detect attributes
    # added outside the constructor (even if such practice is bad, it does happen). We could maybe do some
    # static analysis instead, but it could also miss attributes, or have false positives. So, instead, we
    # just document standard instance attributes and this is it.
    # See: https://gitlab.com/datadrivendiscovery/d3m/issues/158
    def _get_instance_attributes(self) -> typing.Dict[str, type]:
        # Importing here to prevent import cycle.
        from d3m.primitive_interfaces import base

        # Primitive instance attributes are standardized and fixed.
        return {
            'hyperparams': hyperparams_module.Hyperparams,
            'random_seed': int,
            'docker_containers': typing.Dict[str, base.DockerContainer],
            'volumes': typing.Dict[str, str],
            'temporary_directory': typing.Optional[str],
        }

    def get_hyperparams(self) -> 'hyperparams_module.Hyperparams':
        return self.query()['primitive_code']['class_type_arguments']['Hyperparams']

    def get_volumes(self) -> typing.Sequence[typing.Dict]:
        return self._get_volumes(self.query())

    @classmethod
    def _get_volumes(cls, metadata: typing.Dict) -> typing.Sequence[typing.Dict]:
        # We can check simply equality because metadata enumerations are equal to strings as well,
        # and "entry['type']" can be both a string or an enumeration instance.
        return [
            entry for entry in metadata.get('installation', [])
            if entry.get('key', None) and entry.get('file_digest', None) and entry.get('type', None) in [PrimitiveInstallationType.FILE, PrimitiveInstallationType.TGZ]
        ]

    # Not adhering to Liskov substitution principle: we are not returning a list.
    def to_internal_json_structure(self) -> typing.Dict:  # type: ignore
        return utils.to_reversible_json_structure(self.to_internal_simple_structure())

    # Not adhering to Liskov substitution principle: we are not returning a list.
    def to_internal_simple_structure(self) -> typing.Dict:  # type: ignore
        return super().to_internal_simple_structure()[0]['metadata']

    # Not adhering to Liskov substitution principle: we are not returning a list.
    def to_json_structure(self) -> typing.Dict:  # type: ignore
        return utils.to_json_structure(self.to_simple_structure())

    # Not adhering to Liskov substitution principle: we are not returning a list.
    def to_simple_structure(self) -> typing.Dict:  # type: ignore
        return super().to_simple_structure()[0]['metadata']


EXPECTED_CLASS_ATTRIBUTES = {
    'metadata': PrimitiveMetadata,
    'logger': logging.Logger,
}


def _get_inputs(type_arguments: typing.Dict[type, type]) -> type:
    # Importing here to prevent import cycle.
    from d3m.primitive_interfaces import base

    return type_arguments[base.Inputs]


def _get_outputs(type_arguments: typing.Dict[type, type]) -> type:
    # Importing here to prevent import cycle.
    from d3m.primitive_interfaces import base

    return type_arguments[base.Outputs]


def _get_input_labels(type_arguments: typing.Dict[type, type]) -> type:
    # Importing here to prevent import cycle.
    from d3m.primitive_interfaces import distance

    return type_arguments[distance.InputLabels]


# Arguments which can be fulfilled by other primitives in a pipeline.
STANDARD_PIPELINE_ARGUMENTS = {
    'inputs': {
        'get_type': _get_inputs,
    },
    'outputs': {
        'get_type': _get_outputs,
    },
    'input_labels': {
        'get_type': _get_input_labels,
    },
}


def _get_hyperparams(type_arguments: typing.Dict[type, type]) -> type:
    # Importing here to prevent import cycle.
    from d3m.primitive_interfaces import base

    return type_arguments[base.Hyperparams]


def _get_docker_containers(type_arguments: typing.Dict[type, type]) -> type:
    # Importing here to prevent import cycle.
    from d3m.primitive_interfaces import base

    return typing.Optional[typing.Dict[str, base.DockerContainer]]


def _get_params(type_arguments: typing.Dict[type, type]) -> type:
    # Importing here to prevent import cycle.
    from d3m.primitive_interfaces import base

    return type_arguments[base.Params]


def _get_gradient_outputs(type_arguments: typing.Dict[type, type]) -> type:
    # Importing here to prevent import cycle.
    from d3m.primitive_interfaces import base

    return base.Gradients[type_arguments[base.Outputs]]  # type: ignore


def _get_module(type_arguments: typing.Dict[type, type]) -> type:
    # Importing here to prevent import cycle.
    from d3m.primitive_interfaces import base

    return type_arguments[base.Module]


# Arguments which are meaningful only for a runtime executing a pipeline.
STANDARD_RUNTIME_ARGUMENTS = {
    'hyperparams': {
        'get_type': _get_hyperparams,
    },
    'random_seed': {
        'type': int,
        'default': 0,
    },
    'docker_containers': {
        'get_type': _get_docker_containers,
        'default': None,
    },
    'volumes': {
        'type': typing.Optional[typing.Dict[str, str]],
        'default': None,
    },
    'temporary_directory': {
        'type': typing.Optional[str],
        'default': None,
    },
    'timeout': {
        'type': typing.Optional[float],
        'default': None,
    },
    'iterations': {
        'type': typing.Optional[int],
        'default': None,
    },
    'produce_methods': {
        'type': typing.Sequence[str],
    },
    'params': {
        'get_type': _get_params,
    },
    'num_samples': {
        'type': int,
        'default': 1,
    },
    'gradient_outputs': {
        'get_type': _get_gradient_outputs,
    },
    'fine_tune':  {
        'type': bool,
        'default': False,
    },
    'fine_tune_learning_rate': {
        'type': float,
        'default': 0.00001,
    },
    'fine_tune_weight_decay': {
        'type': float,
        'default': 0.00001,
    },
    'temperature': {
        'type': float,
        'default': 0,
    },
    'input_module': {
        'get_type': _get_module,
    },
    'module': {
        'get_type': _get_module,
    },
}


def metadata_serializer(obj: Metadata) -> dict:
    data = {
        'metadata': pickle.dumps(obj),
    }

    return data


def metadata_deserializer(data: dict) -> Metadata:
    metadata = pickle.loads(data['metadata'])

    return metadata


if pyarrow_lib is not None:
    pyarrow_lib._default_serialization_context.register_type(
        Metadata, 'd3m.metadata',
        custom_serializer=metadata_serializer,
        custom_deserializer=metadata_deserializer,
    )
