"use strict";
const nodeunit_1 = require("nodeunit");
const lib_1 = require("../lib");
const util_1 = require("../lib/util");
module.exports = nodeunit_1.testCase({
    'parseMethodResourcePath': {
        'fails if path does not start with a /'(test) {
            test.throws(() => util_1.parseMethodOptionsPath('foo'), /Method options path must start with \'\/\'/);
            test.done();
        },
        'fails if there are less than two components'(test) {
            test.throws(() => util_1.parseMethodOptionsPath('/'), /Method options path must include at least two components/);
            test.throws(() => util_1.parseMethodOptionsPath('/foo'), /Method options path must include at least two components/);
            test.throws(() => util_1.parseMethodOptionsPath('/foo/'), /Invalid HTTP method ""/);
            test.done();
        },
        'fails if a non-supported http method is used'(test) {
            test.throws(() => util_1.parseMethodOptionsPath('/foo/bar'), /Invalid HTTP method "BAR"/);
            test.done();
        },
        'extracts resource path and method correctly'(test) {
            test.deepEqual(util_1.parseMethodOptionsPath('/foo/GET'), { resourcePath: '/~1foo', httpMethod: 'GET' });
            test.deepEqual(util_1.parseMethodOptionsPath('/foo/bar/GET'), { resourcePath: '/~1foo~1bar', httpMethod: 'GET' });
            test.deepEqual(util_1.parseMethodOptionsPath('/foo/*/GET'), { resourcePath: '/~1foo~1*', httpMethod: 'GET' });
            test.deepEqual(util_1.parseMethodOptionsPath('/*/GET'), { resourcePath: '/*', httpMethod: 'GET' });
            test.deepEqual(util_1.parseMethodOptionsPath('/*/*'), { resourcePath: '/*', httpMethod: '*' });
            test.deepEqual(util_1.parseMethodOptionsPath('//POST'), { resourcePath: '/', httpMethod: 'POST' });
            test.done();
        },
    },
    'parseAwsApiCall': {
        'fails if "actionParams" is set but "action" is undefined'(test) {
            test.throws(() => util_1.parseAwsApiCall(undefined, undefined, { foo: '123' }), /"actionParams" requires that "action" will be set/);
            test.done();
        },
        'fails since "action" and "path" are mutually exclusive'(test) {
            test.throws(() => util_1.parseAwsApiCall('foo', 'bar'), /"path" and "action" are mutually exclusive \(path="foo", action="bar"\)/);
            test.done();
        },
        'fails if "path" and "action" are both undefined'(test) {
            test.throws(() => util_1.parseAwsApiCall(), /Either "path" or "action" are required/);
            test.done();
        },
        '"path" mode'(test) {
            test.deepEqual(util_1.parseAwsApiCall('my/path'), { apiType: 'path', apiValue: 'my/path' });
            test.done();
        },
        '"action" mode with no parameters'(test) {
            test.deepEqual(util_1.parseAwsApiCall(undefined, 'MyAction'), { apiType: 'action', apiValue: 'MyAction' });
            test.done();
        },
        '"action" mode with parameters (url-encoded)'(test) {
            test.deepEqual(util_1.parseAwsApiCall(undefined, 'GetObject', { Bucket: 'MyBucket', Key: 'MyKey' }), {
                apiType: 'action',
                apiValue: 'GetObject&Bucket=MyBucket&Key=MyKey',
            });
            test.done();
        },
    },
    'JsonSchemaMapper.toCfnJsonSchema': {
        'maps "ref" found under properties'(test) {
            const schema = {
                type: lib_1.JsonSchemaType.OBJECT,
                properties: {
                    collection: {
                        type: lib_1.JsonSchemaType.ARRAY,
                        items: {
                            ref: '#/some/reference',
                        },
                        uniqueItems: true,
                    },
                },
                required: ['collection'],
            };
            const actual = util_1.JsonSchemaMapper.toCfnJsonSchema(schema);
            test.deepEqual(actual, {
                $schema: 'http://json-schema.org/draft-04/schema#',
                type: 'object',
                properties: {
                    collection: {
                        type: 'array',
                        items: {
                            $ref: '#/some/reference',
                        },
                        uniqueItems: true,
                    },
                },
                required: ['collection'],
            });
            test.done();
        },
        'does not map a "ref" property name'(test) {
            const schema = {
                type: lib_1.JsonSchemaType.OBJECT,
                properties: {
                    ref: {
                        type: lib_1.JsonSchemaType.ARRAY,
                        items: {
                            ref: '#/some/reference',
                        },
                        uniqueItems: true,
                    },
                },
                required: ['ref'],
            };
            const actual = util_1.JsonSchemaMapper.toCfnJsonSchema(schema);
            test.deepEqual(actual, {
                $schema: 'http://json-schema.org/draft-04/schema#',
                type: 'object',
                properties: {
                    ref: {
                        type: 'array',
                        items: {
                            $ref: '#/some/reference',
                        },
                        uniqueItems: true,
                    },
                },
                required: ['ref'],
            });
            test.done();
        },
    },
});
//# sourceMappingURL=data:application/json;base64,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