"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.VpcEndpointServiceDomainName = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const crypto = require("crypto");
const core_1 = require("@aws-cdk/core");
const custom_resources_1 = require("@aws-cdk/custom-resources");
const lib_1 = require("../lib");
// v2 - keep this import as a separate section to reduce merge conflict when forward merging with the v2 branch.
// eslint-disable-next-line
const core_2 = require("@aws-cdk/core");
/**
 * A Private DNS configuration for a VPC endpoint service.
 */
class VpcEndpointServiceDomainName extends core_2.Construct {
    // The way this class works is by using three custom resources and a TxtRecord in conjunction
    // The first custom resource tells the VPC endpoint service to use the given DNS name
    // The VPC endpoint service will then say:
    // "ok, create a TXT record using these two values to prove you own the domain"
    // The second custom resource retrieves these two values from the service
    // The TxtRecord is created from these two values
    // The third custom resource tells the VPC Endpoint Service to verify the domain ownership
    constructor(scope, id, props) {
        super(scope, id);
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_route53_VpcEndpointServiceDomainNameProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.constructor);
            }
            throw error;
        }
        const serviceUniqueId = core_1.Names.nodeUniqueId(props.endpointService.node);
        const serviceId = props.endpointService.vpcEndpointServiceId;
        this.domainName = props.domainName;
        // Make sure a user doesn't accidentally add multiple domains
        this.validateProps(props);
        VpcEndpointServiceDomainName.endpointServicesMap[serviceUniqueId] = this.domainName;
        VpcEndpointServiceDomainName.endpointServices.push(props.endpointService);
        // Enable Private DNS on the endpoint service and retrieve the AWS-generated configuration
        const privateDnsConfiguration = this.getPrivateDnsConfiguration(serviceUniqueId, serviceId, this.domainName);
        // Tell AWS to verify that this account owns the domain attached to the service
        this.verifyPrivateDnsConfiguration(privateDnsConfiguration, props.publicHostedZone);
        // Finally, don't do any of the above before the endpoint service is created
        this.node.addDependency(props.endpointService);
    }
    validateProps(props) {
        const serviceUniqueId = core_1.Names.nodeUniqueId(props.endpointService.node);
        if (serviceUniqueId in VpcEndpointServiceDomainName.endpointServicesMap) {
            const endpoint = VpcEndpointServiceDomainName.endpointServicesMap[serviceUniqueId];
            throw new Error(`Cannot create a VpcEndpointServiceDomainName for service ${serviceUniqueId}, another VpcEndpointServiceDomainName (${endpoint}) is already associated with it`);
        }
    }
    /**
     * Sets up Custom Resources to make AWS calls to set up Private DNS on an endpoint service,
     * returning the values to use in a TxtRecord, which AWS uses to verify domain ownership.
     */
    getPrivateDnsConfiguration(serviceUniqueId, serviceId, privateDnsName) {
        // The custom resource which tells AWS to enable Private DNS on the given service, using the given domain name
        // AWS will generate a name/value pair for use in a TxtRecord, which is used to verify domain ownership.
        const enablePrivateDnsAction = {
            service: 'EC2',
            action: 'modifyVpcEndpointServiceConfiguration',
            parameters: {
                ServiceId: serviceId,
                PrivateDnsName: privateDnsName,
            },
            physicalResourceId: custom_resources_1.PhysicalResourceId.of(serviceUniqueId),
        };
        const removePrivateDnsAction = {
            service: 'EC2',
            action: 'modifyVpcEndpointServiceConfiguration',
            parameters: {
                ServiceId: serviceId,
                RemovePrivateDnsName: true,
            },
        };
        const enable = new custom_resources_1.AwsCustomResource(this, 'EnableDns', {
            onCreate: enablePrivateDnsAction,
            onUpdate: enablePrivateDnsAction,
            onDelete: removePrivateDnsAction,
            policy: custom_resources_1.AwsCustomResourcePolicy.fromSdkCalls({
                resources: [
                    core_1.Fn.join(':', [
                        'arn',
                        core_1.Stack.of(this).partition,
                        'ec2',
                        core_1.Stack.of(this).region,
                        core_1.Stack.of(this).account,
                        core_1.Fn.join('/', [
                            'vpc-endpoint-service',
                            serviceId,
                        ]),
                    ]),
                ],
            }),
        });
        // Look up the name/value pair if the domain changes, or the service changes,
        // which would cause the values to be different. If the unique ID changes,
        // the resource may be entirely recreated, so we will need to look it up again.
        const lookup = hashcode(core_1.Names.uniqueId(this) + serviceUniqueId + privateDnsName);
        // Create the custom resource to look up the name/value pair generated by AWS
        // after the previous API call
        const retrieveNameValuePairAction = {
            service: 'EC2',
            action: 'describeVpcEndpointServiceConfigurations',
            parameters: {
                ServiceIds: [serviceId],
            },
            physicalResourceId: custom_resources_1.PhysicalResourceId.of(lookup),
        };
        const getNames = new custom_resources_1.AwsCustomResource(this, 'GetNames', {
            onCreate: retrieveNameValuePairAction,
            onUpdate: retrieveNameValuePairAction,
            // describeVpcEndpointServiceConfigurations can't take an ARN for granular permissions
            policy: custom_resources_1.AwsCustomResourcePolicy.fromSdkCalls({
                resources: custom_resources_1.AwsCustomResourcePolicy.ANY_RESOURCE,
            }),
        });
        // We only want to call and get the name/value pair after we've told AWS to enable Private DNS
        // If we call before then, we'll get an empty pair of values.
        getNames.node.addDependency(enable);
        // Get the references to the name/value pair associated with the endpoint service
        const name = getNames.getResponseField('ServiceConfigurations.0.PrivateDnsNameConfiguration.Name');
        const value = getNames.getResponseField('ServiceConfigurations.0.PrivateDnsNameConfiguration.Value');
        return { name, value, serviceId };
    }
    /**
     * Creates a Route53 entry and a Custom Resource which explicitly tells AWS to verify ownership
     * of the domain name attached to an endpoint service.
     */
    verifyPrivateDnsConfiguration(config, publicHostedZone) {
        // Create the TXT record in the provided hosted zone
        const verificationRecord = new lib_1.TxtRecord(this, 'DnsVerificationRecord', {
            recordName: config.name,
            values: [config.value],
            zone: publicHostedZone,
        });
        // Tell the endpoint service to verify the domain ownership
        const startVerificationAction = {
            service: 'EC2',
            action: 'startVpcEndpointServicePrivateDnsVerification',
            parameters: {
                ServiceId: config.serviceId,
            },
            physicalResourceId: custom_resources_1.PhysicalResourceId.of(core_1.Fn.join(':', [config.name, config.value])),
        };
        const startVerification = new custom_resources_1.AwsCustomResource(this, 'StartVerification', {
            onCreate: startVerificationAction,
            onUpdate: startVerificationAction,
            policy: custom_resources_1.AwsCustomResourcePolicy.fromSdkCalls({
                resources: [
                    core_1.Fn.join(':', [
                        'arn',
                        core_1.Stack.of(this).partition,
                        'ec2',
                        core_1.Stack.of(this).region,
                        core_1.Stack.of(this).account,
                        core_1.Fn.join('/', [
                            'vpc-endpoint-service',
                            config.serviceId,
                        ]),
                    ]),
                ],
            }),
        });
        // Only verify after the record has been created
        startVerification.node.addDependency(verificationRecord);
    }
}
exports.VpcEndpointServiceDomainName = VpcEndpointServiceDomainName;
_a = JSII_RTTI_SYMBOL_1;
VpcEndpointServiceDomainName[_a] = { fqn: "@aws-cdk/aws-route53.VpcEndpointServiceDomainName", version: "1.158.0" };
// Track all domain names created, so someone doesn't accidentally associate two domains with a single service
VpcEndpointServiceDomainName.endpointServices = [];
// Track all domain names created, so someone doesn't accidentally associate two domains with a single service
VpcEndpointServiceDomainName.endpointServicesMap = {};
/**
 * Hash a string
 */
function hashcode(s) {
    const hash = crypto.createHash('md5');
    hash.update(s);
    return hash.digest('hex');
}
;
//# sourceMappingURL=data:application/json;base64,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