# arqLogger

This library is intended to be used by developers seeking to send logs generated by the strategy outside Arquants infrastructure to be analyze and process.

## Installation

Use the package manager [pip](https://pip.pypa.io/en/stable/) to install arqLogger.

```bash
pip install -U arqLogger
```

In the case you used it in an Arquant Strategy. You import the package directly in the strategy and you are good to go.

## Usage
The library can be used to create the server and the client side of the connection.
 
#### Client
###### Simple Strategy
How to use arqLogger in an Arquant Strategy
```python
from arquants import Strategy
from arquants import Order

import arqLogger

class ExampleStrategy(Strategy):

    def __init__(self):
        # Init websocket logger connection
        self.logger = arqLogger.ArquantLogger(
           url="ws://ip:port/", # URL to connect 
           strategy=self, # Reference to the strategy
           strategy_id="My_Strategy_1", # Unique ID of the strategy
           param_list=locals()) # parameter list used to run the strategy, this information is send in the init message. (local function send all the variable define in the scope)
        
        # Establish a connection with the server and send the initial message
        self.logger.connect()

        
    def next(self):
        # Example on how to log Market Data message received.
        # data_list: list of n instrument to be log
        # entry_list: list with n list (one per instrument) in which we indicate the entries we want to log
        # additional: dict with additional information
        self.logger.log_md_event(data_list=[self.data0, self.data1],
                                 entry_list=[['bid_px','bid_qty','bid_px_2','bid_qty_2'], ['bid_px','bid_qty']],
                                 additional={
                                     "other_data": 0,
                                     "another_data": "Important Info",
                                 })

        # Send buy order for instrument data0 
        order = self.buy(data=self.data0, price=3000, size=10, exectype=Order.Limit)
        
        # Example on how to log when we send a New Order to the market.
        # order: new order.
        # additional: dict with additional information
        self.logger.log_new_order_response(order=order,
                                           additional={
                                                "other_data": 0,
                                                "another_data": "Important Info",
                                           })


    def notify_order(self, order):
        # Example on how to log when the strategy received an Update on an order.
        self.logger.log_er_event(order=order,
                                 additional={
                                     "important_info": 555,
                                 })
        if order.status is Order.Accepted:
            # Cancelling order
            self.cancel(order)
            # Example on how to log when we cancel an order.
            self.logger.log_cancel_order_response(order=order, 
                                                  additional={
                                                        "important_info": 555,
                                                  })
        if order.status is Order.Partial:
            # Replacing order
            replaced_order = self.replace(size=order.size, price=3001, order=order)
            # Example on how to log when we replace an order.
            self.logger.log_replace_order_response(order=replaced_order, 
                                                   additional={
                                                        "important_info": 555,
                                                   })
        # Example on how to log messages related with the strategy.
        # data: dict with the data we want to send in the log
        amount = 10
        self.ws.log_strategy(data={"internal_log": "calculating new rate.", "internal_amount": amount})        
    
    def on_pause(self):
        # Example on how to log a pause event.
        # description: describe what the strategy do when this happend
        self.logger.log_pause_event(description="Se apreto pausa, cancelando ordenes")

    def on_error(self):
        # Example on how to log an error event.
        # description: describe what the strategy do when this happend
        self.logger.log_error_event(description="Ocurrio un error en la estrategia, cancelando ordenes")     
```
#### Server
Example on how to create a websocket server using **arqLogger**
###### Simple Logging Server

How to setup a simple websocket server that print in the stdout all message received.
```python
import arqLogger

# Function to be call when a new message is received by the server
def on_new_log(log):
   print("Message received: {0}".format(log))


# start new Websocket Sever in localhost port 8001 and specify function to be call when new message arrived
arqLogger.start_websocket(8001, on_new_log)
```

###### Advance Logging Server
How to to create a server that logs all the messages received in a new file.
```python
import logging
from datetime import datetime
from logging.handlers import RotatingFileHandler

import arqLogger

# create new logger
logger = logging.getLogger("Rotating Log")
logger.setLevel(logging.INFO)

# add a rotating handler
handler = RotatingFileHandler("log_file.log", maxBytes=1000000, backupCount=5)
logger.addHandler(handler)


def on_new_log(log):
    logger.info("%s - %s" % (datetime.now(), log))


# start new Websocket Sever in localhost port 8001 and specify function to be call when new message arrived
arqLogger.start_websocket(8001, on_new_log)
```


Author/Maintainer
-----------------

* [Franco Zanuso](https://github.com/fzanuso>)
