from autotabular.pipeline.components.base import AutotabularRegressionAlgorithm
from autotabular.pipeline.constants import DENSE, PREDICTIONS, SPARSE, UNSIGNED_DATA
from ConfigSpace.configuration_space import ConfigurationSpace
from ConfigSpace.hyperparameters import CategoricalHyperparameter, UniformIntegerHyperparameter


class KNearestNeighborsRegressor(AutotabularRegressionAlgorithm):

    def __init__(self, n_neighbors, weights, p, random_state=None):
        self.n_neighbors = n_neighbors
        self.weights = weights
        self.p = p
        self.random_state = random_state

    def fit(self, X, Y):
        import sklearn.neighbors

        self.n_neighbors = int(self.n_neighbors)
        self.p = int(self.p)

        self.estimator = \
            sklearn.neighbors.KNeighborsRegressor(
                n_neighbors=self.n_neighbors,
                weights=self.weights,
                p=self.p)
        self.estimator.fit(X, Y)
        return self

    def predict(self, X):
        if self.estimator is None:
            raise NotImplementedError()
        return self.estimator.predict(X)

    @staticmethod
    def get_properties(dataset_properties=None):
        return {
            'shortname': 'KNN',
            'name': 'K-Nearest Neighbor Classification',
            'handles_regression': True,
            'handles_classification': False,
            'handles_multiclass': False,
            'handles_multilabel': False,
            'handles_multioutput': True,
            'is_deterministic': True,
            'input': (DENSE, SPARSE, UNSIGNED_DATA),
            'output': (PREDICTIONS, )
        }

    @staticmethod
    def get_hyperparameter_search_space(dataset_properties=None):
        cs = ConfigurationSpace()

        n_neighbors = UniformIntegerHyperparameter(
            name='n_neighbors', lower=1, upper=100, log=True, default_value=1)
        weights = CategoricalHyperparameter(
            name='weights',
            choices=['uniform', 'distance'],
            default_value='uniform')
        p = CategoricalHyperparameter(
            name='p', choices=[1, 2], default_value=2)

        cs.add_hyperparameters([n_neighbors, weights, p])

        return cs
