#!/usr/bin/env python3
#
# restore waypoints for an exterior lap around Spring Canyon Park in Fort Collins Colorado
#
# usage:
#    ./route_foco_park_waypoints.py
#
#
# Marc Compere, comperem@gmail.com
# created : 14 Jul 2019
# modified: 29 Jul 2020
#
# --------------------------------------------------------------
# Copyright 2018, 2019 Marc Compere
#
# This file is part of the Mobility Virtual Environment (MoVE).
# MoVE is open source software licensed under the GNU GPLv3.
#
# MoVE is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License version 3
# as published by the Free Software Foundation.
#
# MoVE is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License version 3 for more details.
#
# A copy of the GNU General Public License is included with MoVE
# in the COPYING file, or see <https://www.gnu.org/licenses/>.
# --------------------------------------------------------------

import numpy as np
import logging


# ------------------------------------------------------------------------------
# the restore_waypoint() function gets called from MoVE vehicle model
# or as a stand-alone program with __main__ below
def restore_waypoints( cfg ):
    import numpy as np
    from routes_functions import computeRouteStatistics
    import utm
    
    # create an empty route object 
    class Route:
        pass
    route = Route()
    
    # all waypoints must eventually be in an orthogonal, terrain-fixed XYZ Cartesian coordinate system
    # however, route, or path coordinates may be specified in lat/lon pairs or (X,Y) coordinates directly
    #
    # - if original route coordinates are specified in meters, then use 'meters' for route.originalUnits
    # - if original route coordinates are specified in lat/lon pairs in decimal degrees, use 'decimal_degrees'
    route.originalUnits = 'decimal_degrees' #'meters'
    route.desc = "route foco, an exterior lap around Spring Canyon Park in Ft. Collins Colorado from lat/lon recorded in SensorLog iPhone app"
    
    # -------------------------------
    # specify the route coordinates:
    # -------------------------------
    # the lines below were copy-pasted directly from this .csv: fort_collins_spring_canyon_park_1_lap_13_Jul_2019.csv
    # the .csv file was written by fort_collins_lap_preprocessor.m which works in Matlab or Octave
    #
    # for a series of points without commas, python has something called
    # multi-line strings that are indicates by three quotes in a row (single or double quotes)
    #
    # format: lat,lon,speed (m/s)
    my_multi_line_string_of_lat_lon_points = """
40.5436783987,-105.1278419687,0.0350459000
40.5436660399,-105.1278686310,1.7443240000
40.5436617903,-105.1279056417,3.3288410000
40.5436395053,-105.1279445296,3.3288410000
40.5436090137,-105.1279808581,3.0832910000
40.5435858368,-105.1280095309,3.0670290000
40.5435541516,-105.1280299657,2.8681230000
40.5435326577,-105.1280538892,2.9069180000
40.5435160629,-105.1280644841,2.9069180000
40.5434935884,-105.1280756104,2.7256840000
40.5434682494,-105.1280803800,2.6605720000
40.5434451257,-105.1280847652,2.4186960000
40.5434294897,-105.1280749925,2.7368760000
40.5434031619,-105.1280737746,2.7322630000
40.5433865279,-105.1280817501,2.7368760000
40.5433592528,-105.1280879513,2.7733500000
40.5433403804,-105.1280839757,3.0075980000
40.5433099640,-105.1280927280,3.0075980000
40.5432735799,-105.1280981011,3.0075980000
40.5432389382,-105.1281163178,3.2295710000
40.5432069255,-105.1281404857,4.4701480000
40.5431817834,-105.1281901554,4.6740610000
40.5431463377,-105.1282355027,4.6740610000
40.5431161851,-105.1282780600,4.7530440000
40.5430734243,-105.1283214909,5.0531130000
40.5430451323,-105.1283737726,5.6016570000
40.5430055855,-105.1284234062,5.8110460000
40.5429575856,-105.1284613151,5.8110460000
40.5429095113,-105.1284842406,6.0140960000
40.5428486131,-105.1284700398,6.4654100000
40.5427838076,-105.1284657151,6.4721260000
40.5427187932,-105.1284730053,6.4654100000
40.5426639714,-105.1284884819,6.3275350000
40.5426124926,-105.1285110487,6.1309270000
40.5425635898,-105.1285511477,5.6701600000
40.5425195647,-105.1285896762,5.6701600000
40.5424931878,-105.1286429302,5.6183650000
40.5424709237,-105.1287043639,5.5662140000
40.5424556821,-105.1287692094,5.1699800000
40.5424473697,-105.1288266319,5.1699800000
40.5424129774,-105.1288533815,5.1699800000
40.5423660320,-105.1288785269,5.4365840000
40.5423134648,-105.1288558922,6.3213270000
40.5422618407,-105.1288118572,6.6291090000
40.5422144198,-105.1287581129,6.6291090000
40.5421746744,-105.1286960162,6.4076190000
40.5421357889,-105.1286282150,6.4076190000
40.5421006456,-105.1285492693,7.0073900000
40.5420661917,-105.1284649617,7.6105550000
40.5420568605,-105.1283678992,7.6105550000
40.5420525075,-105.1282731183,7.6105550000
40.5420511479,-105.1281743736,7.8920760000
40.5420475855,-105.1280696080,8.0496020000
40.5420461819,-105.1279589590,8.4278840000
40.5420578044,-105.1278492508,8.4278840000
40.5420710533,-105.1277474212,8.2043190000
40.5420918572,-105.1276513871,8.1973890000
40.5421168293,-105.1275704580,8.1749320000
40.5421371875,-105.1274851058,8.2306680000
40.5421618067,-105.1273942896,8.2027120000
40.5421701376,-105.1272966142,8.2306680000
40.5421733695,-105.1271984029,8.4258460000
40.5421746593,-105.1271010349,8.0709830000
40.5421551976,-105.1270047471,8.4031280000
40.5421350943,-105.1269087545,8.3154440000
40.5420998872,-105.1268173475,8.3154440000
40.5420631472,-105.1267292951,8.3154440000
40.5420187312,-105.1266440825,8.5632090000
40.5419688695,-105.1265663061,8.5632090000
40.5419128370,-105.1264990477,8.5403050000
40.5418574183,-105.1264273867,8.5403050000
40.5417922934,-105.1263741333,8.3530650000
40.5417273852,-105.1263223069,8.4148780000
40.5416631201,-105.1262695543,8.3530650000
40.5416027733,-105.1262142135,8.2660060000
40.5415433693,-105.1261641053,8.1364740000
40.5414889018,-105.1261097567,7.7276700000
40.5414397965,-105.1260425229,7.9559970000
40.5413934546,-105.1259749293,7.7276700000
40.5413606534,-105.1258966601,7.4609730000
40.5413267519,-105.1258237483,7.0927470000
40.5412883178,-105.1257568311,7.0927470000
40.5412935899,-105.1257292412,4.0691740000
40.5413101816,-105.1256840592,4.1590530000
40.5413252470,-105.1256412569,3.8594660000
40.5413336824,-105.1255946973,3.8594660000
40.5413404693,-105.1255491365,3.7665100000
40.5413510919,-105.1254945298,3.6465520000
40.5413671131,-105.1254447666,3.6465520000
40.5413839075,-105.1253969192,2.4840990000
40.5413765098,-105.1253747198,2.4840990000
40.5413761902,-105.1253676525,1.8524470000
40.5413974331,-105.1253570388,0.7851635000
40.5414230317,-105.1253281626,0.7851635000
40.5414465982,-105.1253129167,3.3095740000
40.5414765098,-105.1252878523,5.3017140000
40.5415057996,-105.1252373227,5.3017140000
40.5415335263,-105.1251926094,5.3017140000
40.5415537582,-105.1251429597,4.8598590000
40.5415838701,-105.1250899490,5.0029040000
40.5416292161,-105.1250540638,5.0029040000
40.5416685817,-105.1250168419,5.0029040000
40.5416824507,-105.1250134117,2.4090570000
40.5416952148,-105.1250025271,3.0925160000
40.5417195968,-105.1249693544,2.6431210000
40.5417374134,-105.1249534791,2.6431210000
40.5417585080,-105.1249561614,2.6187100000
40.5417888966,-105.1249678201,4.0494480000
40.5418206662,-105.1250068042,4.0916300000
40.5418511985,-105.1250218586,4.0494480000
40.5418849100,-105.1250386963,4.0916300000
40.5419191053,-105.1250603443,4.2420500000
40.5419685485,-105.1250753470,4.7879120000
40.5420080554,-105.1250997686,4.8181260000
40.5420369584,-105.1251433072,4.8181260000
40.5420774786,-105.1251635798,4.8603410000
40.5421272132,-105.1251817025,5.6373020000
40.5421830224,-105.1252017733,6.3577870000
40.5422396389,-105.1252201338,6.3577870000
40.5423122265,-105.1252404337,6.9191070000
40.5423850124,-105.1252580016,7.5478040000
40.5424588682,-105.1252676982,7.9078730000
40.5425404283,-105.1252728559,7.9078730000
40.5426090104,-105.1252758252,8.4465180000
40.5426880719,-105.1252585998,8.7098270000
40.5427622624,-105.1252452512,8.7098270000
40.5428457898,-105.1252329325,8.5943790000
40.5429159269,-105.1252031406,8.5943790000
40.5429973427,-105.1251767821,8.7754580000
40.5430781964,-105.1251692772,8.7754580000
40.5431571264,-105.1251715706,8.7184600000
40.5432323923,-105.1251798652,8.4974830000
40.5433126802,-105.1251856863,8.4783650000
40.5433865559,-105.1252017929,8.5712870000
40.5434582389,-105.1252212921,8.4783650000
40.5435306613,-105.1252405875,8.1868030000
40.5436006998,-105.1252658918,8.1598020000
40.5436748524,-105.1252892642,8.2281150000
40.5437467364,-105.1253125183,8.1598020000
40.5438158987,-105.1253185898,8.0597390000
40.5438875710,-105.1253266384,8.0339600000
40.5439593861,-105.1253308778,7.9657160000
40.5440313500,-105.1253331832,8.0195870000
40.5441016699,-105.1253321892,7.9657160000
40.5441712842,-105.1253316600,7.7911520000
40.5442404077,-105.1253362129,7.7911520000
40.5443088712,-105.1253276993,7.8388860000
40.5443730014,-105.1253266271,7.8388860000
40.5444392596,-105.1253116225,7.6518270000
40.5445053202,-105.1252952850,7.5479900000
40.5445725969,-105.1252770522,7.2486390000
40.5446366727,-105.1252578994,7.2486390000
40.5446976207,-105.1252384811,6.9648810000
40.5447553610,-105.1252171402,6.7565020000
40.5448118909,-105.1251964227,6.5934890000
40.5448667242,-105.1251780231,6.5934890000
40.5449206691,-105.1251712730,6.2682480000
40.5449707136,-105.1251597469,6.0935020000
40.5450199144,-105.1251479410,5.7942470000
40.5450705383,-105.1251323129,5.5629070000
40.5451209801,-105.1251187386,5.5629070000
40.5451674297,-105.1251110146,5.1838710000
40.5452150626,-105.1251078853,5.1438090000
40.5452583074,-105.1251115514,5.1288260000
40.5453036827,-105.1251347720,4.8772770000
40.5453436669,-105.1251542905,4.8772770000
40.5453846338,-105.1251728674,4.8369240000
40.5454208886,-105.1252008192,4.7751890000
40.5454517665,-105.1252373313,5.3499540000
40.5454783275,-105.1252811344,5.3740900000
40.5455073014,-105.1253329415,5.3740900000
40.5455284706,-105.1254079272,5.4987550000
40.5455346799,-105.1254716310,5.8020340000
40.5455414678,-105.1255552330,5.8020340000
40.5455488333,-105.1256198517,5.5954200000
40.5455532544,-105.1256856310,5.5954200000
40.5455483734,-105.1257506555,5.2005000000
40.5455299723,-105.1258106036,5.3597690000
40.5455157556,-105.1258750890,5.2005000000
40.5454934759,-105.1259443815,5.3597690000
40.5454668078,-105.1260077378,5.6442450000
40.5454245134,-105.1260529330,5.8454870000
40.5453826904,-105.1260993899,5.6442450000
40.5453488008,-105.1261314031,5.1511190000
40.5453047364,-105.1261642018,4.9876400000
40.5452614353,-105.1261940831,5.2220900000
40.5452190706,-105.1262366280,5.1976100000
40.5451812829,-105.1262873001,5.2220900000
40.5451471843,-105.1263283190,5.3958910000
40.5451150910,-105.1263902228,5.6597900000
40.5451016603,-105.1264530169,5.5318260000
40.5450859843,-105.1265147680,5.5318260000
40.5450697616,-105.1265713096,5.3281320000
40.5450539291,-105.1266329536,5.3281320000
40.5450396981,-105.1266968105,5.3281320000
40.5450285276,-105.1267598061,5.3773490000
40.5450196843,-105.1268182565,5.1247170000
40.5450157230,-105.1268781471,5.1247170000
40.5450109976,-105.1269405478,5.0710970000
40.5450056638,-105.1269987149,5.0710970000
40.5450019760,-105.1270623804,5.1175570000
40.5450055838,-105.1271246207,5.1812950000
40.5450064292,-105.1271971252,5.1812950000
40.5449998275,-105.1272595876,5.1812950000
40.5449988951,-105.1273274337,5.0350020000
40.5449708890,-105.1273788993,5.0350020000
40.5449435400,-105.1274249557,5.0350020000
40.5449032036,-105.1274619470,5.2772720000
40.5448607490,-105.1275023910,5.0199860000
40.5448173848,-105.1275258005,5.0574540000
40.5447753994,-105.1275429820,5.0574540000
40.5447249270,-105.1275423235,5.2729130000
40.5446655997,-105.1275431813,5.3856740000
40.5446131168,-105.1275226038,5.3856740000
40.5445639954,-105.1275014380,5.3856740000
40.5445084381,-105.1274816550,5.5591260000
40.5444590405,-105.1274548739,5.6746370000
40.5444063185,-105.1274365297,5.6746370000
40.5443572323,-105.1274253684,5.4615000000
40.5443099855,-105.1274105941,5.2518480000
40.5442647483,-105.1274102600,4.9694620000
40.5442160726,-105.1274424992,5.0108830000
40.5441691192,-105.1274594835,5.0108830000
40.5441199683,-105.1274752881,5.1535770000
40.5440766036,-105.1274960275,4.8520320000
40.5440342441,-105.1275141596,4.8520320000
40.5439950088,-105.1275290364,4.6328280000
40.5439579327,-105.1275469911,4.6328280000
40.5439234785,-105.1275771883,4.8634300000
40.5438912869,-105.1276035153,4.8634300000
40.5438502202,-105.1276410599,4.6445120000
40.5438217364,-105.1276712568,4.1153500000
40.5437911495,-105.1277030961,3.6063190000
40.5437597305,-105.1277295312,3.6063190000
40.5437337764,-105.1277402133,3.3614330000
40.5437077598,-105.1277562536,2.9234450000
40.5436872037,-105.1277750725,1.6068610000
        """
    # all math and route following must be done in orthogonal coordinates, so
    # convert these lat/lon pairs to UTM coordinates which assumes a flat Earth
    # within this UTM zone (there are 60 zones around Earth)
    
    list_of_rows = my_multi_line_string_of_lat_lon_points.split('\n')
    
    # initialize native python lists
    lat=[]
    lon=[]
    X=[]
    Y=[]
    spd_mps=[]
    cnt=0
    #print('---')
    for row in list_of_rows:
        #print(row)
        if row.strip() != '': # strip() removes leading and trailing whitespace
            #print( repr(row) ) # repr() shows invisible characters to distinguish spaces from tabs and newlines
            #str=row.strip().split("\t")
            str=row.strip().split(",")
            #print('processing row {0}: [{1}]'.format(cnt,str))
            
            lat_single_pt   = float( str[0] )
            lon_single_pt   = float( str[1] )
            speed_single_pt = float( str[2] )
            
            # utm library converts from lat/lon in decimal degrees to orthogonal
            # XYZ coordinates in meters within a single UTM zone
            (X_single_pt,Y_single_pt,zone,latBand) = utm.from_latlon(lat_single_pt,lon_single_pt) # these are typically giant numbers, like (X,Y)=(489174.484, 4488110.896)
            
            # list.append() is super fast for growing native python lists dynamically
            lat.append( lat_single_pt )
            lon.append( lon_single_pt )
            X.append( X_single_pt ) # X_origin and Y_origin are also typically giant numbers, like 
            Y.append( Y_single_pt )
            spd_mps.append( speed_single_pt )
            cnt=cnt+1
    #print('---')
    
    lat     = np.array(   lat  ) # convert native python lists to numpy array objects
    lon     = np.array(   lon  )
    X       = np.array(     X  )
    Y       = np.array(     Y  )
    Z       = np.zeros( len(X) ) # (m) initialize Z with numpy array of all zeros
    spd_mps = np.array( spd_mps  ) # (m/s) commanded speed for all waypoints is from SensorLog's original GPS-based recording
    
    
    # --------------------------------------------------------------------------
    # now stuff these arrays (python lists) into the route object create at the top
    # MoVE will use this function's return 'route' object from here on out
    route.lat = lat
    route.lon = lon
    route.X = X # (m) X array in cartesian XYZ coordinates
    route.Y = Y # (m) Y array in cartesian XYZ coordinates
    route.Z = Z # (m) Z array in cartesian XYZ coordinates
    route.spd_mps = spd_mps # (m/s)
    if '__file__' in locals():
        route.source = __file__ # this file's name
    else:
        route.source = __name__ # __main__ when debugging in a terminal console
    
    route = computeRouteStatistics(route)
    
    return(route)
# ------------------------------------------------------------------------------


logging.basicConfig(
    level=logging.DEBUG,
    format='(%(threadName)-14s) %(message)s',
)



if __name__ == "__main__":
    
    import matplotlib.pyplot as plt
    import multiprocessing
    import os
    from routes_functions import computeRouteStatistics, plot_graph
    
    print("starting __main__")
    
    class Cfg:
        pass
    
    cfg = Cfg()
    #cfg.vid=101
    #cfg.L_char = 1.0 # (m) example vehicle's characteristic length
    #cfg.v_max = 20.0 # (m/s) example vehicle's maximum speed
    
    # this function is called from the vehicle model, main_veh_model.py
    route = restore_waypoints( cfg )
    
    # print full route data structure
    #for key,val in route.__dict__.items():
    #    print("{0}={1}".format(key,val))
    
    # optionally bring the plot window up in a separate process
    makePlots=True #False
    skipNthLabel=10 #1 % use 10 or 50 if waypoint labels clutter the plot
    if (makePlots==True):
        multiprocessing.Process(target=plot_graph, args=(route,skipNthLabel)).start()
        print("exiting main")
        os._exit(0) # this exits immediately with no cleanup or buffer flushing

    
    



























