# -*- coding: utf-8 -*-
################################################################################
#                                                                              #
#   This file is part of the Bibolamazi Project.                               #
#   Copyright (C) 2013-2018 by Philippe Faist                                  #
#   philippe.faist@bluewin.ch                                                  #
#                                                                              #
#   Bibolamazi is free software: you can redistribute it and/or modify         #
#   it under the terms of the GNU General Public License as published by       #
#   the Free Software Foundation, either version 3 of the License, or          #
#   (at your option) any later version.                                        #
#                                                                              #
#   Bibolamazi is distributed in the hope that it will be useful,              #
#   but WITHOUT ANY WARRANTY; without even the implied warranty of             #
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the              #
#   GNU General Public License for more details.                               #
#                                                                              #
#   You should have received a copy of the GNU General Public License          #
#   along with Bibolamazi.  If not, see <http://www.gnu.org/licenses/>.        #
#                                                                              #
################################################################################

import os
import os.path
import re
import codecs
import unicodedata
import string
import textwrap
#import collections
#import hashlib
import logging

from pybtex.database import BibliographyData #, Entry
#from pybtex.utils import OrderedCaseInsensitiveDict
import pybtex.textutils

from bibolamazi.core.bibfilter import BibFilter, BibFilterError
from bibolamazi.core.bibfilter.argtypes import CommaStrList
from bibolamazi.core import butils
from bibolamazi.core import bibusercache
from bibolamazi.core.bibusercache import tokencheckers

from .util import arxivutil
from .util import auxfile

logger = logging.getLogger(__name__)

### DON'T CHANGE THIS STRING. IT IS THE STRING THAT IS SEARCHED FOR IN AN EXISTING
### DUPFILE TO PREVENT OVERWRITING OF WRONG FILES.
### SERIOUSLY. IT DOESN'T NEED TO CHANGE.
BIBALIAS_WARNING_HEADER = """\
% NOTE: THIS FILE WAS AUTOMATICALLY GENERATED BY bibolamazi SCRIPT!
%       ANY CHANGES WILL BE LOST!
"""

BIBALIAS_HEADER = r"""%
####BIBALIAS_WARNING_HEADER####
%
% File automatically generated by bibolamazi's `duplicates` filter.
%
% You should include this file in your main LaTeX file with the command
%
%   \input{####DUP_FILE_NAME####}
%
% in your document preamble.
%

""".replace('####BIBALIAS_WARNING_HEADER####\n', BIBALIAS_WARNING_HEADER)

BIBALIAS_LATEX_DEFINITIONS = r"""

%
% The following will define the command \bibalias{<alias>}{<source>}, which will make
% the command \cite[..]{<alias>} the same as doing \cite[..]{<source>}.
%
% This code has been copied and adapted from
%    http://tex.stackexchange.com/questions/37233/
%

\makeatletter
% \bibalias{<alias>}{<source>} makes \cite{<alias>} equivalent to \cite{<source>}
\newcommand\bibalias[2]{%
  \@namedef{bibali@#1}{#2}%
}


% Code taken from from etextools.sty, apparently originally borrowed from environ.sty
\newcommand\biba@deblank[1]{\romannumeral\biba@deblank@#1/ /} % dark magic. stay away
\long\def\biba@deblank@#1 /{\biba@deblank@i#1/}
\long\def\biba@deblank@i#1/#2{\z@#1}


\newtoks\biba@toks
\let\bibalias@oldcite\cite
\def\cite{%
  \@ifnextchar[{%
    \biba@cite@optarg%
  }{%
    \biba@cite{}%
  }%
}
\newcommand\biba@cite@optarg[2][]{%
  \biba@cite{[#1]}{#2}%
}
\newcommand\biba@cite[2]{%
  \biba@toks{\bibalias@oldcite#1}%
  \def\biba@comma{}%
  \def\biba@all{}%
  \def\biba@argkeys{}%
  \@for\biba@one@:=#2\do{%
    \edef\biba@one{\expandafter\@firstofone\biba@one@\@empty}%
    \edef\biba@one{\expandafter\biba@deblank\expandafter{\biba@one}}% remove whitespace
    \edef\biba@argkeys{\biba@argkeys\biba@comma\biba@one}%
    \@ifundefined{bibali@\biba@one}{%
      \edef\biba@all{\biba@all\biba@comma\biba@one}%
    }{%
      \PackageInfo{bibalias}{%
        Replacing citation `\biba@one' with `\@nameuse{bibali@\biba@one}'
      }%
      \edef\biba@all{\biba@all\biba@comma\@nameuse{bibali@\biba@one}}%
    }%
    \def\biba@comma{,}%
  }%
  %
  % However, still write in the .aux file a dummy \citation{...} command, so that
  % filters.util.auxfile will still catch those used citations
  %
  \immediate\write\@auxout{\noexpand\bgroup\noexpand\renewcommand\noexpand\citation[1]{}\noexpand\citation{\biba@argkeys}\noexpand\egroup}%
  %
  % Now, produce the \cite command with the original keys instead of the aliases
  %
  \edef\biba@tmp{\the\biba@toks{\biba@all}}%
  \biba@tmp%
}
\makeatother


%
% Now, declare all the alias keys.
%

"""



DUPL_WARN_TOP = """

    DUPLICATE ENTRIES WARNING
    -------------------------

"""

DUPL_WARN_ENTRY = """\
    %(alias)-20s   ** duplicate of **  %(orig)s
"""
DUPL_WARN_ENTRY_MIDCOL=4+20+3+len('** duplicate of **  ') + 2; # column no. to start text of "original"
DUPL_WARN_ENTRY_BEGCOL=6; # column no. to start text of "alias"
DUPL_WARN_ENTRY_COLSEP = 4
DUPL_WARN_ENTRY_COLWIDTH = DUPL_WARN_ENTRY_MIDCOL - DUPL_WARN_ENTRY_COLSEP - DUPL_WARN_ENTRY_BEGCOL

DUPL_WARN_BOTTOM = """\
    There were %(num_dupl)d duplicates.
    -------------------------

"""



# --------------------------------------------------


# some utilities


# these words will get stripped from journal names when forming abbreviations
BORING_WORDS = (
    "a",
    "of",
    "the",
    "in",
    "on",
    "and",
    "its",
    "de",
    "et",
    "der",
    "und",
    )


def normstr(x, lower=True):
    if not isinstance(x, str):
        x = str(x.decode('utf-8'))

    x2 = unicodedata.normalize('NFKD', x).strip()
    if lower:
        x2 = x2.lower()
    # remove any unicode compositions (accents, etc.)
    x2 = re.sub(b'[^\\x00-\\x7f]', b'', x2.encode('utf-8')).decode('utf-8')
    ## additionally, remove any special LaTeX chars which may be written differently.
    #x2 = re.sub(r'\\([a-zA-Z]+|.)', '', x2)
    x2 = re.sub(r'''[\{\}\|\.\+\?\*\,\'\"\\]''', '', x2)
    x2 = re.sub(r'-+', '-', x2)
    #logger.longdebug("Normalized string: %r -> %r", x, x2)
    return x2

def getlast(pers, lower=True):
    # join last names
    last = normstr(
        str( butils.latex_to_text(" ".join(pers.prelast_names+pers.last_names)).split()[-1] ),
        lower=lower
    )
    initial = re.sub(
        '[^a-z]',
        '',
        normstr(u"".join([pybtex.textutils.abbreviate(x) for x in pers.first_names]),
                lower=lower),
        flags=re.IGNORECASE
    )[0:1] # only first initial [a-z]
    return (last, initial)

def fmtjournal(x):
    if not isinstance(x, str):
        x = str(x.decode('utf-8'))
        
    x2 = normstr(x, lower=False)

    # drop "a", "the", "on", "of"
    # -- HACK to keep the `A' in ``Physics Review A'': Trick it into thinking as
    #    ``Physics Review Aaaa'', and then only capital letters are kept anyway
    x2 = re.sub(r'\s+(a)\s*([.:,]|$)', ' Aaaaaa', x2, flags=re.IGNORECASE)
    # --
    x2 = re.sub(r'\b(' + r'|'.join(BORING_WORDS) + r')\b', '', x2, flags=re.IGNORECASE)
    #logger.longdebug('fmtjournal TEMP: %r', x2)

    x2 = re.sub(r'\b([a-z])', lambda m: m.group().capitalize(), x2)
    x2 = re.sub(r'[^A-Z]', '', x2)
    #logger.longdebug('fmtjournal TEMP final: %r', x2)
    return x2


def sanitize_doi(doi):
    if doi is None:
        return None
    return re.sub(r'^DOI\s*:?\s*', '', doi.strip().lower(), flags=re.IGNORECASE)

# -------------

# utility for comparing months:
_mon_list = ['jan', 'feb', 'mar', 'apr', 'may', 'jun', 'jul', 'aug', 'sep', 'oct', 'nov', 'dec']
_mon = dict(zip(_mon_list,range(1,12)))

def normalize_month(x):
    m = x.strip().lower()[:3]
    if m in _mon:
        return _mon[m]
    return -1







# --------------------------------------------------








HELP_AUTHOR = r"""
Philippe Faist, (C) 2013-2018, GPL 3+
"""

HELP_DESC = r"""
Detect duplicate bibtex entries and create corresponding aliases.
"""

HELP_TEXT = r"""
This filter works by writing a LaTeX file (the ``dupfile''), which contains the
commands needed to define the bibtex aliases. You may then use the LaTeX `\cite'
command with either key; duplicate entries are merged and a single entry will be
produced in the bibliography.

The dupfile, which contains the necessary definitions to make keys aliases of
their duplicate entries, is specifed via the `--dupfile' option as
`--dupfile=yourdupfile.tex' or with `-sDupfile=yourdupfile.tex'.

In your main LaTeX document, you need to add the following command in the
preamble:

  \input{yourdupfile.tex}

where of couse yourdupfile.tex is the file that you specified to this filter.

Alternatively, if you just set the warn flag on, then a duplicate file is not
created (unless the dupfile option is also given), and a warning is displayed
for each duplicate found. The duplicates *ARE* however still removed and merged
in the final bibfile.

Duplicates are detected based on several heuristics and quite extensive
testing. Recently I've been pretty much satisfied with the results, but there is
no guarantee that it won't miss some matches sometimes. The algorithm is
designed to prevent as much as possible any false matches, so that shouldn't
normally happen, but again there's no guarantee. Also, if two entries refer to
different versions of a paper where one is on the arXiv and the other is
published, they are NOT considered as duplicates.

You can inhibit specific entries from being matched as duplicates by including
'x-no-duplicate' as a keyword (in the keywords={...} field).  If this keyword is
included in an entry, it will never match as a duplicate with any other entry.

If you set -dKeepOnlyUsed, then while sorting out the duplicates, we only keep
those entries that are referred to from within the document. For this to work,
you need to have called latex/pdflatex on your document first. The document is
assumed to have the same base name as your bibolamazi file, for instance, if
your bibolamazi file is called "myfile.bibolamazi.bib", then we assume that we
can read the citations in the aux file produced by LaTeX called "myfile.aux".
If this is not the case, you may specify the base name of your LaTeX document
using the option "-sJobname=myfile". Note -dKeepOnlyUsed only works if you are
merging duplicates (i.e., you didn't set -dMergeDuplicates=False).

The dupfile will be by default self-contained, i.e. will contain all the
definitions necessary so that you can use the different cite keys transparently
with the `\cite` LaTeX command. However the implementation of the `\cite'
command is a bit minimal. For example, no spaces are allowed between its
arguments, and other commands such as `\citep' are not supported.

If you specify the `-dCustomBibalias' option, then the dupfile will only contain
a list of duplicate definitions of the form

    \bibalias{<alias>}{<original>}

without any definition of the `\bibalias' command itself. It is thus up to the
user to provide a usable `\bibalias' command, before the `\input{<dupfile>}'
invocation. Use this option to get most flexibly on how you want to treat your
aliases, but this will require a little more work from your side.
"""


# ------------------------------------------------


class DuplicatesEntryInfoCacheAccessor(bibusercache.BibUserCacheAccessor):
    def __init__(self, **kwargs):
        super().__init__(
            cache_name='duplicates_entryinfo',
            **kwargs
            )


    def initialize(self, cache_obj, **kwargs):
        #
        # Make sure we set up cache invalidation properly, to ensure that if a
        # user modifies a falsely picked-up duplicate, that the cache is
        # updated!
        #

        cache_entries_validator = tokencheckers.EntryFieldsTokenChecker(
            self.bibolamaziFile().bibliographyData(),
            store_type=True,
            store_persons=['author'],
            fields=list(set(
                # from arxivInfo
                arxivutil.arxivinfo_from_bibtex_fields +
                [
                    # 'author' is not a field in pybtex, see store_persons above
                    'note',
                    'journal',
                    'title',
                    ])),
            )

        self.cacheDic()['entries'].set_validation(cache_entries_validator)

    def prepare_entry_cache(self, key, a, arxivaccess):

        entriescache = self.cacheDic()['entries']

        if key in entriescache and entriescache[key]:
            # already in cache, all fine
            return

        cache_a = entriescache[key]

        cache_a['pers'] = [ getlast(pers) for pers in a.persons.get('author',[]) ]

        cache_a['arxivinfo'] = arxivaccess.getArXivInfo(a.key)

        note = a.fields.get('note', '')
        cache_a['note_cleaned'] = (arxivutil.stripArXivInfoInNote(note) if note else "")
        
        cache_a['j_abbrev'] = fmtjournal(a.fields.get('journal', ''))

        def cleantitle(title):
            title = unicodedata.normalize('NFKD', str(butils.latex_to_text(title).lower()))
            # remove any unicode compositions (accents, etc.)
            title = re.sub(b'[^\\x00-\\x7f]', b'', title.encode('utf-8')).decode('utf-8')
            # remove any unusual characters
            title = re.sub(r'[^a-zA-Z0-9 ]', '', title)
            # remove any inline math
            title = re.sub(r'\$[^$]+\$', '', title)
            title = re.sub(r'\\\(.+?\\\)', '', title)
            # clean up whitespace
            title = re.sub(r'\s+', ' ', title)
            return title.strip()

        cache_a['title_clean'] = cleantitle(a.fields.get('title', ''))


    # PhF -- turns out to be slower with cached comparison !!! Probably because
    # we gotta compute a sh*tload of hashes?


    # def _pairkey_entries(self, entriescache, key1, key2):
    #     # do NOT use python's built-in hash(), it can change from one process to
    #     # another.  Use md5, which will always give the same hash
    #     return hashlib.md5(
    #             ('%s\n%s\n%s\n%s'%(str(key1), str(entriescache.token_for(key1)),
    #                                str(key2), str(entriescache.token_for(key2))) ).encode('utf-8')
    #         ).digest()

    # def query_cache_comparison(self, key1, key2):

    #     entriescache = self.cacheDic()['entries']
    #     comparisonscache =  self.cacheDic()['comparisons']

    #     if key1 not in entriescache or key2 not in entriescache:
    #         raise ValueError("store_cache_comparison: entries must already be in entries cache (%s and %s)",
    #                          key1, key2)

    #     pairkey = self._pairkey_entries(entriescache, key1, key2)

    #     if pairkey not in comparisonscache:
    #         return None

    #     return comparisonscache[pairkey]

    # def store_cache_comparison(self, key1, key2, is_duplicate):

    #     entriescache = self.cacheDic()['entries']
    #     comparisonscache =  self.cacheDic()['comparisons']

    #     if key1 not in entriescache or key2 not in entriescache:
    #         raise ValueError("store_cache_comparison: entries must already be in entries cache (%s and %s)",
    #                          key1, key2)

    #     pairkey = self._pairkey_entries(entriescache, key1, key2)

    #     comparisonscache[pairkey] = bool(is_duplicate)

        

        
        


    def get_entry_cache(self, key):
        return self.cacheDic()['entries'][key]





    



# ------------------------------------------------



rx_conflictkey = re.compile(r'^(?P<origkey>.*)\.conflictkey\.\d+$', flags=re.IGNORECASE)
rx_keyword_aka = re.compile(r'(aka|previously)--(?P<aliaskey>[^ ,;]+)',
                            flags=re.IGNORECASE)
rx_keyword_nodup = re.compile(r'\bx-no-duplicate\b', flags=re.IGNORECASE)


#AliasPair = collections.namedtuple("AliasPair", ('alias', 'origkey', 'is_extra',), )
# need a mutable version:
class AliasPair:
    def __init__(self, aliaskey, origkey, is_extra=False):
        super().__init__()
        self.aliaskey = aliaskey
        self.origkey = origkey
        self.is_extra = is_extra # CANNOT CHANGE if used in AliasList
    def __repr__(self):
        return "AliasPair(%r,%r,is_extra=%r)"%(self.aliaskey,self.origkey,self.is_extra)

class AliasList:
    #
    # Idea: store AliasPair() objects and update them as we add more alias
    # objects.
    #
    def __init__(self):
        super().__init__()
        # The elements in these containers are AliasPair() objects that are
        # shared among the containers
        self.aliases = []
        self.aliases_by_aliaskey = {}
        self.aliases_by_origkey = {}
        self.extra_aliases = []

    def add_alias(self, alias):
        # update our database for this alias (eg, resolve dependencies)
        self.add_virtual_alias(alias)

        # and add the alias
        self.aliases.append(alias)
        if alias.is_extra:
            self.extra_aliases.append(alias)
        self.aliases_by_aliaskey[alias.aliaskey] = alias
        self._register_alias_in_origkey_dic(alias)

    def add_virtual_alias(self, alias):
        #
        # update current aliases as if we were going to add this alias to the
        # aliases.
        #
        if alias.aliaskey in self.aliases_by_aliaskey:
            raise ValueError("Alias is already defined")
        if alias.aliaskey in self.aliases_by_origkey:
            # this aliaskey is used as the target origkey for another alias,
            # update that one.
            #
            # We have otheralias.origkey == alias.aliaskey
            otheraliases = self.aliases_by_origkey[alias.aliaskey]
            logger.longdebug("due to alias %r->%r, updating other aliases %r that pointed to %r",
                             alias.aliaskey, alias.origkey, otheraliases, alias.aliaskey)
            del self.aliases_by_origkey[alias.aliaskey]
            for otheralias in otheraliases:
                otheralias.origkey = alias.origkey
                self._register_alias_in_origkey_dic(alias)

            
    def _register_alias_in_origkey_dic(self, alias):
        if alias.origkey not in self.aliases_by_origkey:
            self.aliases_by_origkey[alias.origkey] = [ alias ]
        else:
            self.aliases_by_origkey[alias.origkey].append(alias)



class DuplicatesFilter(BibFilter):

    helpauthor = HELP_AUTHOR
    helpdescription = HELP_DESC
    helptext = HELP_TEXT

    def __init__(self,
                 dupfile=None,
                 merge_duplicates=True,
                 ensure_conflict_keys_are_duplicates=True,
                 create_alias_from_aka_keyword=True,
                 warn=False,
                 custom_bibalias=False,
                 keep_only_used=None,
                 jobname=None,
                 keep_only_used_in_jobname=None,
                 jobname_search_dirs=None,
                 ignore_fields_warning=None,
                 *args):
        r"""
        DuplicatesFilter constructor.

        *dupfile: the name of a file to write latex code for defining duplicates
               to. This file will be overwritten!!

        *merge_duplicates(bool): Whether to actually merge the duplicates in the
               bibliography data or not. True by default. Set this to False and
               set -dWarn for instance if you would like to have a warning only
               without actually any duplicate culling in the resulting
               bibliography.

        *ensure_conflict_keys_are_duplicates(bool): If true (the default), then
               keys of the form "xxxxxx.conflictkey.N" are verified to indeed be
               duplicates of the corresponding "xxxxxx" entry. These conflict
               key entries are created automatically when two bibtex sources
               have entries with the same key.

        *create_alias_from_aka_keyword(bool): if set to true, then the
               'keywords' field of all entries are scanned for any keywords of
               the form 'aka--XYZ' (also 'previously--XYZ').  If such a
               "keyword" is found, then an alias to that entry is created with
               the bibtex name 'XYZ' when writing to the bibalias file set with
               -sDupfile.  Note that the aliases collected in this way are not
               included in the warning emitted by -dWarn.

        *warn(bool): if this flag is set, a warning is issued for every
               duplicate entry found in the database.

        *custom_bibalias(bool): if set to TRUE, then no latex definitions will
               be generated in the file given in `dupfile', and will rely on a
               user-defined implementation of `\bibalias`.

        *keep_only_used(bool): if TRUE, then only keep entries which are
               referenced in the LaTeX file.  Use the -sJobname option to
               specify which LaTeX jobname to look at.  Note: This option has no
               effect if the `merge_duplicates' option is off.

        *jobname: If -dKeepOnlyUsed is specified, then look for used citations
               in the LaTeX file with this base name.  See also the only_used
               filter.  The corresponding AUX file is searched for and analyzed.
               If -sJobname is not specified, then the LaTeX file name is
               guessed from the bibolamazi file name, as for the only_used
               filter.

        *ignore_fields_warning(CommaStrList): Normally, warnings are issued if
               some fields differ between entries that are detected as
               duplicates.  List some fields here for which you don't care. (By
               default, this is a reasonable collection of fields which normally
               don't matter, e.g., 'file'.)

        *keep_only_used_in_jobname: OBSOLTE

        *jobname_search_dirs(CommaStrList): (use with -sJobname) search for the
               AUX file in the given directories, as for the only_used filter.
               Paths are absolute or relative to bibolamazi file.
        """

        super().__init__()

        self.dupfile = dupfile

        self.merge_duplicates = butils.getbool(merge_duplicates)

        self.create_alias_from_aka_keyword = \
            butils.getbool(create_alias_from_aka_keyword)

        self.warn = butils.getbool(warn)
        
        self.custom_bibalias = butils.getbool(custom_bibalias)
        
        self.ensure_conflict_keys_are_duplicates = \
            butils.getbool(ensure_conflict_keys_are_duplicates)

        if len(args) == 1:
            if self.dupfile is None:
                self.dupfile = args[0]
            else:
                raise BibFilterError(
                    "duplicates",
                    "Repeated values given for dupfile: one as an option (`%s'), "
                    "the other as a positional argument (`%s')"%(self.dupfile, args[0])
                )
        elif len(args) != 0:
            raise BibFilterError(
                "duplicates",
                "Received unexpected positional arguments (at most one expected, "
                "the dupfile name): [%s]"%(",".join(["%s"%(x) for x in args]))
            )

        self.keep_only_used = False
        self.jobname = None
        
        if keep_only_used_in_jobname:
            logger.warning(
                "duplicates filter: the -sKeepOnlyUsedInJobname=<jobname> option is now obsolete. "
                "Please replace this option by '-dKeepOnlyUsed -sJobname=<jobname>'"
            )
            self.keep_only_used = True
            self.jobname = keep_only_used_in_jobname

        if keep_only_used is not None:
            if keep_only_used_in_jobname:
                logger.warning(
                    "duplicates filter: options -sKeepOnlyUsedInJobname=<jobname> and "
                    " -dKeepOnlyUsed may not be specified simultaneously"
                )
            self.keep_only_used = keep_only_used
            self.jobname = jobname

        if jobname_search_dirs is not None:
            jobname_search_dirs = CommaStrList(jobname_search_dirs)
        self.jobname_search_dirs = jobname_search_dirs

        if ignore_fields_warning:
            self.ignore_fields_warning = [ x.lower() for x in CommaStrList(ignore_fields_warning)]
        else:
            self.ignore_fields_warning = [
                # NOTE: must be all lowercase!
                
                # seriously, no latex document cares.
                'file', 'keywords', 'mendeley-tags',

                # these are already taken into account when detecting
                # duplicates
                'eprint', 'arxivid', 'archiveprefix',
            ]

        # ### PhF: Is this used at all or is this leftover code ???
        self.cache_entries_validator = None

        #if (not self.dupfile and not self.warn):
        #    logger.warning(
        #        "bibolamazi duplicates filter: no action will be taken as neither -sDupfile or"+
        #        " -dWarn are given!"
        #    )

        logger.debug('duplicates: dupfile=%r, warn=%r' % (dupfile, warn))


    def getRunningMessage(self):
        if (self.dupfile):
            return ("Processing duplicate entries. Don't forget to insert `\\input{%s}' in "
                    "your LaTeX file!" %(self.dupfile) )
        return "Processing duplicate entries"
    

    def action(self):
        return BibFilter.BIB_FILTER_BIBOLAMAZIFILE


    def requested_cache_accessors(self):
        return [
            DuplicatesEntryInfoCacheAccessor,
            arxivutil.ArxivInfoCacheAccessor,
            arxivutil.ArxivFetchedAPIInfoCacheAccessor,
            ]


    def compare_entries(self, akey, bkey, a, b, dupl_entryinfo_cache_accessor):
        """
        Returns a tuple `(is_same, reason)` of a boolean and a string. The `reason`
        is a human-readable explanations of why the entries are considered the
        same or not.
        """

        # compare author list first

        logger.longdebug('Comparing entries %s and %s', akey, bkey)

        # PhF -- turns out to be slower with cached comparison !!! Probably
        # because we gotta compute a sh*tload of hashes?
        #
        # r = dupl_entryinfo_cache_accessor.query_cache_comparison(akey, bkey)
        # if r is not None:
        #     return r, "Cached comparison result; clear cache to re-compare"

        cache_a = dupl_entryinfo_cache_accessor.get_entry_cache(akey)
        cache_b = dupl_entryinfo_cache_accessor.get_entry_cache(bkey)

        tf, reason = self._do_compare_entries(a, b, cache_a, cache_b)

        # dupl_entryinfo_cache_accessor.store_cache_comparison(akey, bkey, tf)

        return tf, reason

    def _do_compare_entries(self, a, b, cache_a, cache_b):

        apers = cache_a['pers']
        bpers = cache_b['pers']

        # see if either entry has keyword 'x-no-duplicate' preventing it from
        # matching as a duplicate
        m1 = rx_keyword_nodup.search(a.fields.get('keywords', ''))
        m2 = rx_keyword_nodup.search(b.fields.get('keywords', ''))
        if m1 is not None or m2 is not None:
            return (False,
                    ("Entry ‘%s’ is explicitly marked as non-duplicate "
                     "('x-no-duplicate' keyword)") %(
                        a.key if m1 else b.key
                    ))

        pending_pos_match_warning = []
        def pos_match():
            for w in pending_pos_match_warning:
                logger.warning(w)

        if (len(apers) != len(bpers)):
            return False, "Author list lengths %d and %d differ"%(len(apers), len(bpers))

        for k in range(len(apers)):
            (lasta, ina) = apers[k]
            (lastb, inb) = bpers[k]
            # use Levenshtein distance to detect possible typos or alternative spellings
            # (e.g. Koenig vs Konig). Allow approx. one such typo per 8 characters.
            lev_dist = levenshtein(lasta, lastb)
            if (lev_dist > (1+int(len(lasta)/8)) or (ina and inb and ina != inb)):
                return False, "Authors %r and %r differ"%((lasta, ina), (lastb, inb))
            if lev_dist > 0:
                pending_pos_match_warning.append(
                    "Duplicate entries {} and {} have possible typo in author name: \"{}\" vs \"{}\""
                    .format(a.key, b.key, str(a.persons.get('author',[])[k]),
                            str(b.persons.get('author',[])[k]))
                )


        logger.longdebug("Author list matches! %r and %r ",apers,bpers)

        def compare_neq_fld(x, y, fld, filt=lambda x: x):
            xval = x.get(fld, None)
            yval = y.get(fld, None)
            try:
                xval = xval.strip()
            except AttributeError:
                pass
            try:
                yval = yval.strip()
            except AttributeError:
                pass

            return xval is not None and yval is not None and filt(xval) != filt(yval) 

        # authors are the same. check year
        if (compare_neq_fld(a.fields, b.fields, 'year')):
            logger.longdebug
            return False, "Years %r and %r differ"%(a.fields.get('year', None),
                                                    b.fields.get('year', None))

        if (compare_neq_fld(a.fields, b.fields, 'month', filt=normalize_month)):
            return False, "Months %r and %r differ"%(a.fields.get('month', None),
                                                     b.fields.get('month', None))

        doi_a = sanitize_doi(a.fields.get('doi'))
        doi_b = sanitize_doi(b.fields.get('doi'))
        if (doi_a and doi_b and doi_a != doi_b):
            return False, "DOI's %r and %r differ"%(doi_a, doi_b)
        if (doi_a and doi_a == doi_b):
            pos_match()
            return True, "DOI's %r and %r are the same"%(doi_a, doi_b)

        arxiv_a = cache_a['arxivinfo']
        arxiv_b = cache_b['arxivinfo']

        logger.longdebug("  arxiv_a=%r, arxiv_b=%r", arxiv_a, arxiv_b)
        
        if (arxiv_a and arxiv_b and
            'arxivid' in arxiv_a and 'arxivid' in arxiv_b and
            arxiv_a['arxivid'] != arxiv_b['arxivid']):
            return False, "arXiv IDS %r and %r differ"%(arxiv_a['arxivid'], arxiv_b['arxivid'])
        if (arxiv_a and arxiv_b and
            'arxivid' in arxiv_a and 'arxivid' in arxiv_b and
            arxiv_a['arxivid'] == arxiv_b['arxivid']):
            pos_match()
            return True, "arXiv IDS %r and %r same"%(arxiv_a['arxivid'], arxiv_b['arxivid'])


        # if they have different notes, then they're different entries
        note_cl_a = cache_a['note_cleaned']
        note_cl_b = cache_b['note_cleaned']
        if (note_cl_a and note_cl_b and note_cl_a != note_cl_b):
            return False, "Notes (cleaned up) %r and %r differ"%(note_cl_a, note_cl_b)

        # create abbreviations of the journals by keeping only the uppercase letters
        j_abbrev_a = cache_a['j_abbrev']
        j_abbrev_b = cache_b['j_abbrev']
        # don't require them to be equal, but just that they have good
        # overlap... e.g. "PNAS" and "PNASUSA" allow also one typo per approx. 4
        # chars
        if ( j_abbrev_a and j_abbrev_b and
             (levenshtein(j_abbrev_a[:len(j_abbrev_b)], j_abbrev_b[:len(j_abbrev_a)])
              > (1+int(min(len(j_abbrev_a),len(j_abbrev_b))/4))
             ) ):
            return False, "Journal (parsed & simplified) %r and %r differ"%(j_abbrev_a, j_abbrev_b)

        if ( compare_neq_fld(a.fields, b.fields, 'volume') ):
            return False, "Volumes %r and %r differ"%(a.fields.get('volume', None),
                                                      b.fields.get('volume', None))

        if ( compare_neq_fld(a.fields, b.fields, 'number') ):
            return False, "Numbers %r and %r differ"%(a.fields.get('number', None),
                                                      b.fields.get('number', None))

        titlea = cache_a['title_clean']
        titleb = cache_b['title_clean']

        if (titlea and titleb and titlea != titleb):
            return False, "Titles %r and %r differ"%(titlea, titleb)

        # ### Unreliable. Bad for arxiv entries and had some other
        # ### bugs. (E.g. "123--5" vs "123--125" vs "123")
        #
        #if ( compare_neq_fld(a.fields, b.fields, 'pages') ):
        #    print("pages differ!")
        #    import pdb; pdb.set_trace()
        #    return False, "pages differ ....."

        #logger.longdebug("Entries %s and %s match.", a.key, b.key)

        # make sure we don't report duplicates for entries that have basically
        # no fields (e.g. .PATCH entries).  Calculate a simple score that
        # specifies "how much information" the entry provides based on which we
        # could detect a duplicate, and set a simple threshold.
        def dupl_relevant_fields_score(c):
            def f(x):
                return 1 if c[x] else 0
            return (
                0
                + 10*f('pers')
                +  5*f('j_abbrev')
                +  5*f('year')
                +  5*f('title_clean')
                +  2*f('volume')
                +  2*f('number')
                +  5*f('note_cleaned')
            )
        threshold = 6
        if not (dupl_relevant_fields_score(cache_a) >= threshold and
                dupl_relevant_fields_score(cache_b) >= threshold):
            return False, "Insufficient information to determine if entries are duplicates"

        # well at this point the publications are pretty much duplicates
        pos_match()
        return True, "Entries do not differ on the relevant fields"
        

    def update_entry_with_duplicate(self, origkey, origentry, duplkey, duplentry):
        """
        Merges definitions present in the duplicate entry, which are not present in the
        original. A very simple update-if-not-present mechanism is done, and no full-blown
        merge is performed. Simply entries which are not already present in the original
        are updated.
        """

        _mergefields = ['keywords','note','annote']
        _splitrx = re.compile(r'[;,]\s*') # no capture please!! because we use rx.split(...)

        for (fk, fval) in duplentry.fields.items():
            # field not in original -- add it
            if fk not in origentry.fields or not origentry.fields[fk].strip():
                origentry.fields[fk] = fval
                continue

            origval = origentry.fields[fk]

            if fval.strip().lower() == origval.strip().lower():
                # fields agree on value -- no action needed
                continue

            # field is a list of keywords field -- simply merge all entries
            if fk in _mergefields:
                m = _splitrx.search(origval)
                if m:
                    splitchars = m.group()
                else:
                    splitchars = ','

                thelist = _splitrx.split(origval)
                thelistlower = [x.lower() for x in thelist]
                for newval in _splitrx.split(fval):
                    if newval and newval.lower() not in thelistlower:
                        thelist.append(newval)
                        thelistlower.append(newval.lower())
                origentry.fields[fk] = splitchars.join(thelist)
                continue

            # fields differ, but we don't know how to merge them. Warn the user
            # if it's a field that is not listed as an unimportant field
            if fk.lower() not in self.ignore_fields_warning:
                def trunc_str(x, w=50):
                    if len(x) <= w: return x
                    return x[:(w-3)]+'...'
                f1val = trunc_str(origentry.fields[fk])
                f2val = trunc_str(fval)
                logger.warning("Duplicate entries %s and %s differ on field %s. (\"%s\" vs \"%s\")",
                               origkey, duplkey, fk, f1val, f2val)
            

    def _get_used_citations(self, bibolamazifile):
        jobname = auxfile.get_action_jobname(self.jobname, bibolamazifile)
        #logger.debug("Getting list of used citations from %s.aux.", jobname)
        used_citations = auxfile.get_all_auxfile_citations(
            jobname, bibolamazifile, self.name(),
            self.jobname_search_dirs, return_set=True
        )
        return used_citations
        
    def _write_to_dupfile(self, bibolamazifile, aliases):
        #
        # write definitions to the dupfile
        #
        dupfilepath = os.path.join(bibolamazifile.fdir(), self.dupfile)
        check_overwrite_dupfile(dupfilepath)

        dupstrlist = []

        with codecs.open(dupfilepath, 'w', 'utf-8') as dupf:

            dupf.write(BIBALIAS_HEADER.replace('####DUP_FILE_NAME####', self.dupfile))

            if not self.custom_bibalias:
                dupf.write(BIBALIAS_LATEX_DEFINITIONS)

            # Note: Sort alias pairs in some way (e.g. alphabetically according
            # to (alias, original)), to avoid diffs in VCS's
            for alias in sorted(aliases.aliases,
                                key=lambda x: (x.aliaskey,x.origkey)):
                dupf.write((r'\bibalias{%s}{%s}' % (alias.aliaskey, alias.origkey)) + "\n")
                dupstrlist.append("\t%s is an alias of %s" % (alias.aliaskey,alias.origkey))

            dupf.write('\n\n')

        # issue debug message
        logger.debug("wrote duplicates to file: \n" + "\n".join(dupstrlist))


    def filter_bibolamazifile(self, bibolamazifile):
        #
        # bibdata is a pybtex.database.BibliographyData object
        #

        #if (not self.dupfile and not self.warn):
        #    logger.warning("duplicates filter: No action is being taken because neither "
        #                   "-sDupfile= nor -dWarn have been requested.")
        #    return

        bibdata = bibolamazifile.bibliographyData()

        used_citations = None
        
        if self.keep_only_used:
            used_citations = self._get_used_citations(bibolamazifile)

        aliases = AliasList()

        arxivaccess = arxivutil.setup_and_get_arxiv_accessor(bibolamazifile)

        dupl_entryinfo_cache_accessor = self.cacheAccessor(DuplicatesEntryInfoCacheAccessor)

        for (key, entry) in bibdata.entries.items():
            #cache_entries[key] = {}
            dupl_entryinfo_cache_accessor.prepare_entry_cache(key, entry, arxivaccess)


        newbibdata = BibliographyData()
        unused = BibliographyData()
        #unused_respawned = set() # because del unused.entries[key] is not implemented ... :(

        # def copy_entry(entry):
        #     #return copy.deepcopy(entry) # too deep ...
        #     newpers = {}
        #     for role, plist in entry.persons.items():
        #         newpers[role] = [copy.deepcopy(p) for p in plist]
        #     return Entry(type_=entry.type,
        #                  fields=entry.fields.items(), # will create own Fielddict
        #                  persons=newpers,
        #                  collection=entry.collection
        #                  )


        # Strategy: go through the list of entries, and each time keeping it if it is new,
        # or updating the original and registering the alias if it is a duplicate.
        #
        # With only_used, the situation is a little trickier as we cannot just discard the
        # entries as they are filtered: indeed, they might be duplicates of a used entry,
        # with which one should merge the bib information.
        #
        # So the full algorithm does not immediately discard the unused keys, but rather
        # keeps them in an `unused` list. If they are later required, they are respawned
        # into the actual new list.
        #

        def iter_over_bibdata(bibdata):
            # Iterate while respecting the order of the elements in the bibdata.
            # This might not be guaranteed with .items() or iteritems() (?)
            for k in bibdata.entries:
                yield k, bibdata.entries[k]
            #

        if self.ensure_conflict_keys_are_duplicates:
            # first, go through the whole bibliography, and make sure that any
            # entry of the form 'xxxxx.conflictkey.N' is a duplicate of 'xxxxx'
            confldup_entries_to_remove = []
            for (key, entry) in iter_over_bibdata(bibdata):
                m = rx_conflictkey.match(key)
                if not m: # not a conflictkey entry
                    continue

                origkey = m.group('origkey')

                logger.longdebug("Looking at conflictkey entry %s", key)

                if origkey not in bibdata.entries:
                    logger.warning("Entry with conflict key %s has no corresponding entry %s",
                                   key, origkey)
                    continue

                origentry = bibdata.entries[origkey]

                same, reason = self.compare_entries(
                    key, origkey, entry, origentry,
                    dupl_entryinfo_cache_accessor
                )
                if not same:
                    logger.warning("Entry with conflict key %s is NOT a duplicate of entry %s: %s",
                                   key, origkey, reason)

                # entries are proper duplicates as expected, remove the conflictkey entry
                logger.debug("Removing conflictkey entry %s as it is really a duplicate of %s",
                             key, origkey)

                # do merge the entries, in case they are not exact copies and
                # one has more information than the other
                self.update_entry_with_duplicate(origkey, origentry, key, entry)

                # don't delete the entry here, because it will mess up the for loop iteration!
                #del bibdata.entries[key]
                confldup_entries_to_remove.append(key)
            for k in confldup_entries_to_remove:
                del bibdata.entries[k]

        if self.merge_duplicates or self.warn:

            #
            # examine all the entries, to see if they have any "aka--..."
            # keywords for which we would want to create corresponding aliases.
            # Collect these into the aliases list as extra aliases.
            #
            # Do this now, before we start discarding unused entries which might
            # have an alias that is used
            #
            if self.create_alias_from_aka_keyword:
                logger.debug("create_alias_from_aka_keyword=True, scanning keywords")
                for k, e in iter_over_bibdata(bibdata):
                    if 'keywords' in e.fields:
                        for m in rx_keyword_aka.finditer(e.fields['keywords']):
                            aliaskey = m.group('aliaskey')
                            if aliaskey in bibdata.entries:
                                logger.warning("Can't create alias '%s' to '%s', alias key name "
                                               "already exists in database as another entry",
                                               aliaskey, k)
                                continue
                            if used_citations is not None and aliaskey not in used_citations:
                                # alias is not used, drop it
                                logger.debug("ignoring extra alias %r -> %r because it's not used",
                                             aliaskey, k)
                            else:
                                logger.debug("extra alias: %r -> %r", aliaskey, k)
                                aliases.add_alias( AliasPair(aliaskey, k, is_extra=True) )


            for (key, entry) in iter_over_bibdata(bibdata):
                #
                # examine this entry, see if it is a duplicate of an entry that
                # we've already seen.
                #
                
                #
                # search the newbibdata object, in case this entry already exists.
                #
                #logger.longdebug('inspecting new entry %s ...', key)
                is_duplicate_of = None
                duplicate_original_is_unused = False
                for (nkey, nentry) in newbibdata.entries.items():
                    same, reason = self.compare_entries(
                        key, nkey, entry, nentry,
                        dupl_entryinfo_cache_accessor
                    )
                    if same:
                        logger.debug("Entry %s is duplicate of existing entry %s: %s",
                                     key, nkey, reason)
                        is_duplicate_of = nkey
                        break
                    
                for (nkey, nentry) in unused.entries.items():
                    same, reason = self.compare_entries(
                        key, nkey, entry, nentry,
                        dupl_entryinfo_cache_accessor
                    )
                    if same:
                        logger.debug("Entry %s is duplicate of entry %s: %s",
                                     key, nkey, reason)
                        is_duplicate_of = nkey
                        duplicate_original_is_unused = True
                        break

                #
                # if it's a duplicate
                #
                if is_duplicate_of is not None:
                    if duplicate_original_is_unused:
                        self.update_entry_with_duplicate(is_duplicate_of,
                                                         unused.entries[is_duplicate_of],
                                                         key, entry)
                        aliases.add_virtual_alias( AliasPair(key, is_duplicate_of) )
                    else:
                        # a duplicate of a key we have used. So update the original ...
                        self.update_entry_with_duplicate(is_duplicate_of,
                                                         newbibdata.entries[is_duplicate_of],
                                                         key, entry)
                        # ... and register the alias.
                        aliases.add_alias( AliasPair(key, is_duplicate_of) )

                    if duplicate_original_is_unused and used_citations and key in used_citations:
                        # if we had set the original in the unused list, but we need the
                        # alias, then respawn the original to the newbibdata so we can refer
                        # to it. Bonus: use the name with which we have referred to it, so we
                        # don't need to register any duplicate.
                        logger.debug("Resuscitating entry %s as a duplicate is a used citation",
                                     is_duplicate_of)
                        ue = unused.entries[is_duplicate_of]
                        del unused.entries[is_duplicate_of]
                        ue.key = key
                        newbibdata.add_entry(key, ue)
                        #unused_respawned.add(is_duplicate_of)
                else:
                    if used_citations is not None and key not in used_citations:
                        # new entry, but we don't want it. So add it to the unused list.
                        unused.add_entry(key, entry)
                    else:
                        # new entry and we want it. So add it to the main newbibdata list.
                        newbibdata.add_entry(key, entry)

            #
            # now, make sure that all originals corresponding to "extra" aliases
            # (e.g., caused by "aka--X" keywords) are in the newbibdata
            # and not in the unused list.
            #
            logger.debug("all aliases = %r", aliases.aliases)
            for alias in aliases.extra_aliases:
                if alias.origkey in unused.entries:
                    # the seemingly unused entry is actually used! put it back.
                    logger.debug("Resuscitating entry %s as it is used by an extra alias",
                                 alias.origkey)
                    newbibdata.add_entry(alias.origkey, unused.entries[alias.origkey])
                    del unused.entries[alias.origkey]

            #
            # check if we are discarding any entries of the form XXX.PATCH, and
            # warn the user
            #
            rx_patch = re.compile('[.]PATCH([.].*)?$')
            for e in unused.entries.keys():
                if rx_patch.search(e):
                    logger.warning(
                        "Entry ‘%s’ is being discarded because it's not used, but it looks like "
                        "a patch for another entry.  If this is the case you should apply "
                        "patches (\"filter: apply_patches\") before running the "
                        "duplicates filter.",
                        e
                    )
            

            # output aliases to the duplicates file

            if self.dupfile:
                self._write_to_dupfile(bibolamazifile, aliases)

            if (self.warn and aliases.aliases):
                #
                # prepare formatting for a single duplicate info pair
                #
                def warnline(dupalias, duporiginal):
                    def fmt(key, entry, cache_entry):
                        s = ", ".join(string.capwords('%s, %s' % (x[0], "".join(x[1])))
                                      for x in cache_entry['pers'])
                        if 'title_clean' in cache_entry and cache_entry['title_clean']:
                            s += ', "' + (cache_entry['title_clean']).capitalize() + '"'
                        if 'j_abbrev' in cache_entry and cache_entry['j_abbrev']:
                            s += ', ' + cache_entry['j_abbrev']

                        f = entry.fields
                        if f.get('month',None) and f.get('year',None):
                            s += ', ' + f['month'] + ' ' + f['year']
                        elif f.get('month', None):
                            s += ', ' + f['month'] + ' <unknown year>'
                        elif f.get('year', None):
                            s += ', ' + f['year']

                        if 'doi' in entry.fields and entry.fields['doi']:
                            s += ', doi:'+entry.fields['doi']
                        if 'arxivinfo' in cache_entry and cache_entry['arxivinfo']:
                            s += ', arXiv:'+cache_entry['arxivinfo']['arxivid']
                        if 'note_cleaned' in cache_entry and cache_entry['note_cleaned']:
                            s += '; ' + cache_entry['note_cleaned']

                        return s

                    tw = textwrap.TextWrapper(width=DUPL_WARN_ENTRY_COLWIDTH)

                    fmtalias = fmt(dupalias, bibdata.entries[dupalias],
                                   dupl_entryinfo_cache_accessor.get_entry_cache(dupalias))
                    fmtorig = fmt(duporiginal, bibdata.entries[duporiginal],
                                  dupl_entryinfo_cache_accessor.get_entry_cache(duporiginal))
                    linesalias = tw.wrap(fmtalias)
                    linesorig = tw.wrap(fmtorig)
                    maxlines = max(len(linesalias), len(linesorig))
                    return (DUPL_WARN_ENTRY % { 'alias': dupalias,
                                                'orig': duporiginal
                                                }
                            + "\n".join( ('%s%s%s%s' %(' '*DUPL_WARN_ENTRY_BEGCOL,
                                                       linealias +
                                                       ' '*(DUPL_WARN_ENTRY_COLWIDTH-len(linealias)),
                                                       ' '*DUPL_WARN_ENTRY_COLSEP,
                                                       lineorig)
                                          for (linealias, lineorig) in
                                          zip(linesalias + ['']*(maxlines-len(linesalias)),
                                              linesorig + ['']*(maxlines-len(linesorig)))) )
                            + "\n\n"
                            )

                # Now glue together a mega-string of all lines, and emit a
                # warning with it to our standard logger.
                #
                # Note: no warning is emitted for extra aliases
                # (e.g. "aka--" keywords) --- what would be the point of
                # that?
                
                warnlines = [
                    warnline(alias.aliaskey, alias.origkey)
                    for alias in aliases.aliases
                    if not alias.is_extra
                ]

                logger.warning(DUPL_WARN_TOP  +
                               "".join(warnlines) +
                               DUPL_WARN_BOTTOM % {'num_dupl': len(warnlines)})


            if self.merge_duplicates:
                #
                # set the new bibdata, without the duplicates --- DON'T DO THIS,
                # BECAUSE CACHES MAY HAVE KEPT A POINTER TO THE BIBDATA::
                #bibolamazifile.setBibliographyData(newbibdata)
                #
                # Instead, update bibolamazifile's bibliographyData() object itself.
                #
                bibolamazifile.setEntries(newbibdata.entries.items())

            # end if self.merge_duplicates or self.warn
        
        return


def bibolamazi_filter_class():
    return DuplicatesFilter





# utility: taken from http://hetland.org/coding/python/levenshtein.py
#
# used to detect small differences in names which can result from either typos, or
# alternative spellings (e.g. 'Konig' vs 'Koenig')
def levenshtein(a, b):
    "Calculates the Levenshtein distance between a and b."
    n, m = len(a), len(b)
    if n > m:
        # Make sure n <= m, to use O(min(n,m)) space
        a,b = b,a
        n,m = m,n
        
    current = range(n+1)
    for i in range(1,m+1):
        previous, current = current, [i]+[0]*n
        for j in range(1,n+1):
            add, delete = previous[j]+1, current[j-1]+1
            change = previous[j-1]
            if a[j-1] != b[i-1]:
                change = change + 1
            current[j] = min(add, delete, change)
            
    return current[n]




def check_overwrite_dupfile(dupfilepath):
    if (not os.path.exists(dupfilepath)):
        return
    # path item exists (but could be dir, etc.)
    if (not os.path.isfile(dupfilepath)):
        raise BibFilterError('duplicates', "Can't overwrite non-file path `%s'"% (dupfilepath))
    
    with codecs.open(dupfilepath, 'r') as f:
        head_content = u''
        for countline in range(10):
            head_content += f.readline()

    if BIBALIAS_WARNING_HEADER not in head_content:
        raise BibFilterError(
            'duplicates',
            ("File `%s' does not seem to have been generated by bibolamazi. "
             "Won't overwrite. Please remove the file manually.")%(dupfilepath))
