# -*- coding: utf-8 -*-
################################################################################
#                                                                              #
#   This file is part of the Bibolamazi Project.                               #
#   Copyright (C) 2013 by Philippe Faist                                       #
#   philippe.faist@bluewin.ch                                                  #
#                                                                              #
#   Bibolamazi is free software: you can redistribute it and/or modify         #
#   it under the terms of the GNU General Public License as published by       #
#   the Free Software Foundation, either version 3 of the License, or          #
#   (at your option) any later version.                                        #
#                                                                              #
#   Bibolamazi is distributed in the hope that it will be useful,              #
#   but WITHOUT ANY WARRANTY; without even the implied warranty of             #
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the              #
#   GNU General Public License for more details.                               #
#                                                                              #
#   You should have received a copy of the GNU General Public License          #
#   along with Bibolamazi.  If not, see <http://www.gnu.org/licenses/>.        #
#                                                                              #
################################################################################

import logging

#import os
#import re

from pybtex.utils import OrderedCaseInsensitiveDict

from bibolamazi.core.bibfilter import BibFilter #, BibFilterError
from bibolamazi.core.bibfilter.argtypes import CommaStrList

from .util import auxfile

logger = logging.getLogger(__name__)


HELP_AUTHOR = r"""
Philippe Faist, (C) 2013, GPL 3+
"""

HELP_DESC = r"""
Discard all BibTeX entries that are not used in a LaTeX document
"""

HELP_TEXT = r"""
This filter detects which citations you have referenced in a LaTeX document, and keeps
only those bibtex entries; the rest is discarded. (Of course this doesn't modify your
original sources.)

For this to work, you need to have called latex/pdflatex on your document first.

By default, the latex document name is guessed from the bibolamazi file name.
For instance, if your bibolamazi file is called "myfile.bibolamazi.bib", then we
assume that we can read the citations in the aux file produced by LaTeX called
"myfile.aux" (for instance, with a LaTeX file callsed "myfile.tex").  If this is
not the case, you may specify the base name of your LaTeX document using the
option "-sJobname=myfile". In case the `mydocument.aux' file is not in the same
directory as the bibolamazi file, you may specify an additional directory (or
several additional directories) in which to search while looking for the aux
file with the option `-sSearchDirs=...'.

WARNING: If you are also using the `duplicates' filter, then you shouldn't use
this filter. The `duplicates' filter itself provides the required functionality:
it has a `-dKeepOnlyUsed' option which does the same thing as the present
filter.

"""



class OnlyUsedFilter(BibFilter):

    helpauthor = HELP_AUTHOR
    helpdescription = HELP_DESC
    helptext = HELP_TEXT

    def __init__(self, jobname=None, search_dirs=[]):
        r"""
        OnlyUsedFilter constructor.

        Arguments:

          - jobname: the base name of the latex file. Will search for
              jobname.aux and look for \citation{..} commands as they are
              generated by latex.  If this option is not provided, then by
              default the base name of the bibolamazi file is used

          - search_dirs(CommaStrList): the .aux file will be searched for in
              this list of directories; separate directories with commas
              e.g. 'path/to/dir1,path/to/dir2' (escape commas and backslashes
              with a backslash)
        """

        super().__init__()

        self.jobname = jobname
        self.search_dirs = CommaStrList(search_dirs)

        if (not self.search_dirs):
            self.search_dirs = ['.', '_cleanlatexfiles'] # also for my cleanlatex utility :)

        logger.debug('only_used: jobname=%r', jobname)


    def getRunningMessage(self):
        return u"only_used: filtering entries ..."

    
    def action(self):
        return BibFilter.BIB_FILTER_BIBOLAMAZIFILE


    def filter_bibolamazifile(self, bibolamazifile):

        jobname = auxfile.get_action_jobname(self.jobname, bibolamazifile)

        citations = auxfile.get_all_auxfile_citations(jobname, bibolamazifile, self.name(), self.search_dirs,
                                                      return_set=True)

        logger.longdebug("set of citations: %r", citations)


        bibdata = bibolamazifile.bibliographyData()
        
        newentries = OrderedCaseInsensitiveDict()

        for key,entry in bibdata.entries.items():
            if key in citations:
                newentries[key] = entry

        logger.longdebug("the new database has entries %r" %(newentries.keys()))

        bibolamazifile.setEntries(newentries.items())

        return


def bibolamazi_filter_class():
    return OnlyUsedFilter


