#Standart library imports
import os
import traceback
import time
import stat
import subprocess
import re
from pathlib import Path
from typing import Any, Tuple

# Third party imports
import wget

# Local imports
from selenium_driver_updater._setting import setting

from selenium_driver_updater.browsers._chromeBrowser import ChromeBrowser

from selenium_driver_updater.util.extractor import Extractor
from selenium_driver_updater.util.requests_getter import RequestsGetter
from selenium_driver_updater.util.logger import logger

from selenium_driver_updater.util.exceptions import DriverVersionInvalidException

class ChromeDriver():
    """Class for working with Selenium chromedriver binary"""

    _tmp_folder_path = 'tmp'

    def __init__(self, **kwargs):

        self.setting : Any = setting

        self.path : str = str(kwargs.get('path'))

        self.upgrade : bool = bool(kwargs.get('upgrade'))

        self.chmod : bool = bool(kwargs.get('chmod'))

        self.check_driver_is_up_to_date : bool = bool(kwargs.get('check_driver_is_up_to_date'))

        self.system_name = ''
        self.filename = ''

        #assign of specific os
        specific_system = str(kwargs.get('system_name', ''))
        specific_system = specific_system.replace('win64', 'win32')
        if specific_system:
            self.system_name = f"chromedriver_{specific_system}.zip"

        self.setting['ChromeDriver']['LastReleasePlatform'] = 'chromedriver'

        #assign of filename
        specific_filename = str(kwargs.get('filename'))
        if specific_filename:
            self.filename = specific_filename + self.setting['Program']['DriversFileFormat']

        self.setting['ChromeDriver']['LastReleasePlatform'] += self.setting['Program']['DriversFileFormat']

        self.chromedriver_path : str =  self.path + self.setting['ChromeDriver']['LastReleasePlatform'] if not specific_filename else self.path + self.filename

        self.version = str(kwargs.get('version'))

        self.info_messages = bool(kwargs.get('info_messages'))

        self.extractor = Extractor
        self.requests_getter = RequestsGetter

        kwargs.update(path=self.chromedriver_path)
        self.chromebrowser = ChromeBrowser(**kwargs)

    def main(self) -> str:
        """Main function, checks for the latest version, downloads or updates chromedriver binary or
        downloads specific version of chromedriver.

        Returns:
            str

            driver_path (str) : Path where chromedriver was downloaded or updated.

        """
        driver_path : str = ''

        self.chromebrowser.main()

        if not self.version:

            #additional checking for main versions to equal - for example, chromedriver version main is 90 and chrome browser is still 89
            is_equal, latest_version_driver, latest_version_browser = self.__compare_latest_version_main_chromedriver_and_latest_version_main_chrome_browser()

            if is_equal:

                driver_path = self.__check_if_chromedriver_is_up_to_date()

            if not is_equal:

                message = (f' Problem with chromedriver latest_version_driver:'
                    f'{latest_version_driver} latest_version_browser: {latest_version_browser}\n'
                    ' It often happen when new version of chromedriver released, but new version of chrome browser is not\n'
                    ' Trying to download the latest previous version of chromedriver')
                logger.error(message)

                driver_path = self.__download_driver(previous_version=True)

        else:

            driver_path = self.__download_driver(version=self.version)

        return driver_path

    def __compare_latest_version_main_chromedriver_and_latest_version_main_chrome_browser(self) -> Tuple[bool, str, str]:
        """Compares latest main version of chromedriver and latest main version of chrome browser"""
        is_equal : bool = False
        latest_version_chromedriver_main : str = ''
        latest_version_browser_main : str = ''

        latest_version_chromedriver = self.__get_latest_version_chromedriver(no_messages=True)

        latest_version_browser = self.chromebrowser._ChromeBrowser__get_latest_version_chrome_browser(no_messages=True)

        latest_version_chromedriver_main = latest_version_chromedriver.split('.', maxsplit=1)[0]
        latest_version_browser_main = latest_version_browser.split('.', maxsplit=1)[0]

        if int(latest_version_chromedriver_main) <= int(latest_version_browser_main):
            is_equal = True

        return is_equal, latest_version_chromedriver_main, latest_version_browser_main

    def __check_if_chromedriver_is_up_to_date(self) -> str:
        """Сhecks for the latest version, downloads or updates chromedriver binary

        Returns:
            str

            driver_path (str) : Path where chromedriver was downloaded or updated.

        """
        driver_path : str = ''

        if self.check_driver_is_up_to_date and not self.system_name:

            is_driver_up_to_date, current_version, latest_version = self.__compare_current_version_and_latest_version()

            if is_driver_up_to_date:
                return self.chromedriver_path

        driver_path = self.__download_driver()

        if self.check_driver_is_up_to_date and not self.system_name:

            is_driver_up_to_date, current_version, latest_version = self.__compare_current_version_and_latest_version()

            if not is_driver_up_to_date:

                message = f'Problem with updating chromedriver current_version: {current_version} latest_version: {latest_version}'
                logger.error(message)
                message = 'Trying to download previous latest version of chromedriver'
                logger.info(message)

                driver_path = self.__download_driver(previous_version=True)

        return driver_path

    def __get_latest_version_chromedriver(self, no_messages : bool = False) -> str:
        """Gets latest chromedriver version

        Returns:
            str

            latest_version (str)  : Latest version of chromedriver.

        """

        latest_version : str = ''

        url = self.setting["ChromeDriver"]["LinkLastRelease"]
        json_data = self.requests_getter.get_result_by_request(url=url)

        latest_version = str(json_data)

        if not no_messages:

            logger.info(f'Latest version of chromedriver: {latest_version}')

        return latest_version

    def __delete_current_chromedriver_for_current_os(self) -> None:
        """Deletes chromedriver from folder if parameter "upgrade" is True"""

        if Path(self.chromedriver_path).exists():

            logger.info(f'Deleted existing chromedriver chromedriver_path: {self.chromedriver_path}')
            Path(self.chromedriver_path).unlink()

    def __get_current_version_chromedriver(self) -> str:
        """Gets current chromedriver version via command in terminal

        Returns:
            str

            driver_version (str) : Current chromedriver version.

        Raises:

            OSError: Occurs when chromedriver made for another CPU type

        """

        driver_version : str = ''
        driver_version_terminal : str = ''

        try:

            if Path(self.chromedriver_path).exists():

                with subprocess.Popen([self.chromedriver_path, '--version'], stdout=subprocess.PIPE) as process:
                    driver_version_terminal = process.communicate()[0].decode('UTF-8')

                find_string = re.findall(self.setting["Program"]["wedriverVersionPattern"], driver_version_terminal)
                driver_version = find_string[0] if len(find_string) > 0 else ''

                logger.info(f'Current version of chromedriver: {driver_version}')

        except OSError:
            message_run = f'OSError error: {traceback.format_exc()}' #probably [Errno 86] Bad CPU type in executable:
            logger.error(message_run)
            return driver_version

        return driver_version

    def __compare_current_version_and_latest_version(self) -> Tuple[bool, str, str]:
        """Compares current version of chromedriver to latest version

        Returns:
            Tuple of bool, str and bool

            result_run (bool)           : True if function passed correctly, False otherwise.
            message_run (str)           : Returns an error message if an error occurs in the function.
            is_driver_up_to_date (bool) : If true current version of chromedriver is up to date. Defaults to False.

        """

        is_driver_up_to_date : bool = False
        current_version : str = ''
        latest_version : str = ''

        current_version = self.__get_current_version_chromedriver()

        if not current_version:
            return is_driver_up_to_date, current_version, latest_version

        latest_version = self.__get_latest_version_chromedriver()

        if current_version == latest_version:
            is_driver_up_to_date = True
            message = f'Your existing chromedriver is up to date. current_version: {current_version} latest_version: {latest_version}'
            logger.info(message)

        return is_driver_up_to_date, current_version, latest_version

    def __chmod_driver(self) -> None:
        """Tries to give chromedriver needed permissions"""

        if Path(self.chromedriver_path).exists():

            logger.info('Trying to give chromedriver needed permissions')

            st = os.stat(self.chromedriver_path)
            os.chmod(self.chromedriver_path, st.st_mode | stat.S_IEXEC)

            logger.info('Needed rights for chromedriver were successfully issued')

    def __get_latest_previous_version_chromedriver_via_requests(self) -> str:
        """Gets previous latest chromedriver version


        Returns:
            str

            latest_version_previous (str)   : Latest previous version of chromedriver.

        """

        latest_version_previous : str = ''

        url = self.setting["ChromeDriver"]["LinkLastRelease"]
        json_data = self.requests_getter.get_result_by_request(url=url)

        latest_version = str(json_data)
        latest_version_main = latest_version.split(".", maxsplit=1)[0]

        latest_version_main_previous = int(latest_version_main) - 1

        url = self.setting["ChromeDriver"]["LinkLatestReleaseSpecificVersion"].format(latest_version_main_previous)
        json_data = self.requests_getter.get_result_by_request(url=url)

        latest_version_previous = str(json_data)

        logger.info(f'Latest previous version of chromedriver: {latest_version_previous}')

        return latest_version_previous

    def __check_if_version_is_valid(self, url : str) -> None:
        """Checks the specified version for existence.

        Args:
            url (str)           : Full download url of chromedriver.

        """

        archive_name : str = url.split("/")[len(url.split("/"))-1]

        find_string = re.findall(self.setting["Program"]["wedriverVersionPattern"], url)
        driver_version = find_string[0] if len(find_string) > 0 else ''

        url_test_valid = self.setting["ChromeDriver"]["LinkCheckVersionIsValid"].format(driver_version)
        version_valid : str = f"{driver_version}/{archive_name}"

        json_data = self.requests_getter.get_result_by_request(url=url_test_valid)

        if not version_valid in json_data:
            message = f'Wrong version or system_name was specified. version_valid: {version_valid} driver_version: {driver_version} url: {url}'
            raise DriverVersionInvalidException(message)

    def __download_driver(self, version : str = '', previous_version : bool = False) -> str:
        """Function to download, delete or upgrade current chromedriver

        Args:
            version (str)               : Specific chromedriver version to download. Defaults to empty string.
            previous_version (boll)     : If true, chromedriver latest previous version will be downloaded. Defaults to False.

        Returns:
            str

            driver_path (str) : Path to unzipped driver.

        """

        url : str = ''
        latest_previous_version : str = ''
        latest_version : str = ''
        driver_path : str = ''

        if self.upgrade:

            self.__delete_current_chromedriver_for_current_os()

        if version:

            url = self.setting["ChromeDriver"]["LinkLastReleaseFile"].format(version)
            logger.info(f'Started download chromedriver specific_version: {version}')

        elif previous_version:

            latest_previous_version = self.__get_latest_previous_version_chromedriver_via_requests()

            url = self.setting["ChromeDriver"]["LinkLastReleaseFile"].format(latest_previous_version)
            logger.info(f'Started download chromedriver latest_previous_version: {latest_previous_version}')

        else:

            latest_version = self.__get_latest_version_chromedriver()

            url = self.setting["ChromeDriver"]["LinkLastReleaseFile"].format(latest_version)
            logger.info(f'Started download chromedriver latest_version: {latest_version}')

        if self.system_name:
            url = url.replace(url.split("/")[-1], '')
            url = url + self.system_name

            logger.info(f'Started downloading chromedriver for specific system: {self.system_name}')

        if any([version, self.system_name ,latest_previous_version]):
            self.__check_if_version_is_valid(url=url)

        archive_name = url.split("/")[-1]
        out_path = self.path + archive_name

        if Path(out_path).exists():
            Path(out_path).unlink()

        logger.info(f'Started download chromedriver by url: {url}')

        if self.info_messages:
            archive_path = wget.download(url=url, out=out_path)
        else:
            archive_path = wget.download(url=url, out=out_path, bar=None)

        time.sleep(2)

        logger.info(f'Chromedriver was downloaded to path: {archive_path}')

        out_path : str = self.path

        parameters = dict(archive_path=archive_path, out_path=out_path)

        if not self.filename:

            self.extractor.extract_and_detect_archive_format(**parameters)

        else:

            filename = self.setting['ChromeDriver']['LastReleasePlatform']
            parameters.update(dict(filename=filename, filename_replace=self.filename))
            
            self.extractor.extract_all_zip_archive_with_specific_name(**parameters)

        if Path(archive_path).exists():
            Path(archive_path).unlink()

        driver_path = self.chromedriver_path

        logger.info(f'Chromedriver was successfully unpacked by path: {driver_path}')

        if self.chmod:

            self.__chmod_driver()

        return driver_path
