"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Rule = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const constructs_1 = require("constructs");
const events_generated_1 = require("./events.generated");
const schedule_1 = require("./schedule");
const util_1 = require("./util");
/**
 * Defines an EventBridge Rule in this stack.
 *
 * @resource AWS::Events::Rule
 */
class Rule extends core_1.Resource {
    constructor(scope, id, props = {}) {
        super(scope, id, {
            physicalName: props.ruleName,
        });
        this.targets = new Array();
        this.eventPattern = {};
        /** Set to keep track of what target accounts and regions we've already created event buses for */
        this._xEnvTargetsAdded = new Set();
        jsiiDeprecationWarnings._aws_cdk_aws_events_RuleProps(props);
        if (props.eventBus && props.schedule) {
            throw new Error('Cannot associate rule with \'eventBus\' when using \'schedule\'');
        }
        this.description = props.description;
        this.scheduleExpression = props.schedule && props.schedule.expressionString;
        const resource = new events_generated_1.CfnRule(this, 'Resource', {
            name: this.physicalName,
            description: this.description,
            state: props.enabled == null ? 'ENABLED' : (props.enabled ? 'ENABLED' : 'DISABLED'),
            scheduleExpression: this.scheduleExpression,
            eventPattern: core_1.Lazy.any({ produce: () => this._renderEventPattern() }),
            targets: core_1.Lazy.any({ produce: () => this.renderTargets() }),
            eventBusName: props.eventBus && props.eventBus.eventBusName,
        });
        this.ruleArn = this.getResourceArnAttribute(resource.attrArn, {
            service: 'events',
            resource: 'rule',
            resourceName: this.physicalName,
        });
        this.ruleName = this.getResourceNameAttribute(resource.ref);
        this.addEventPattern(props.eventPattern);
        for (const target of props.targets || []) {
            this.addTarget(target);
        }
    }
    /**
     * Import an existing EventBridge Rule provided an ARN
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param eventRuleArn Event Rule ARN (i.e. arn:aws:events:<region>:<account-id>:rule/MyScheduledRule).
     */
    static fromEventRuleArn(scope, id, eventRuleArn) {
        const parts = core_1.Stack.of(scope).splitArn(eventRuleArn, core_1.ArnFormat.SLASH_RESOURCE_NAME);
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.ruleArn = eventRuleArn;
                this.ruleName = parts.resourceName || '';
            }
        }
        return new Import(scope, id);
    }
    /**
     * Adds a target to the rule. The abstract class RuleTarget can be extended to define new
     * targets.
     *
     * No-op if target is undefined.
     */
    addTarget(target) {
        var _b, _c, _d;
        jsiiDeprecationWarnings._aws_cdk_aws_events_IRuleTarget(target);
        if (!target) {
            return;
        }
        // Simply increment id for each `addTarget` call. This is guaranteed to be unique.
        const autoGeneratedId = `Target${this.targets.length}`;
        const targetProps = target.bind(this, autoGeneratedId);
        const inputProps = targetProps.input && targetProps.input.bind(this);
        const roleArn = (_b = targetProps.role) === null || _b === void 0 ? void 0 : _b.roleArn;
        const id = targetProps.id || autoGeneratedId;
        if (targetProps.targetResource) {
            const targetStack = core_1.Stack.of(targetProps.targetResource);
            const targetAccount = ((_c = targetProps.targetResource.env) === null || _c === void 0 ? void 0 : _c.account) || targetStack.account;
            const targetRegion = ((_d = targetProps.targetResource.env) === null || _d === void 0 ? void 0 : _d.region) || targetStack.region;
            const sourceStack = core_1.Stack.of(this);
            const sourceAccount = sourceStack.account;
            const sourceRegion = sourceStack.region;
            // if the target is in a different account or region and is defined in this CDK App
            // we can generate all the needed components:
            // - forwarding rule in the source stack (target: default event bus of the receiver region)
            // - eventbus permissions policy (creating an extra stack)
            // - receiver rule in the target stack (target: the actual target)
            if (!util_1.sameEnvDimension(sourceAccount, targetAccount) || !util_1.sameEnvDimension(sourceRegion, targetRegion)) {
                // cross-account and/or cross-region event - strap in, this works differently than regular events!
                // based on:
                // https://docs.aws.amazon.com/eventbridge/latest/userguide/eb-cross-account.html
                // for cross-account or cross-region events, we require a concrete target account and region
                if (!targetAccount || core_1.Token.isUnresolved(targetAccount)) {
                    throw new Error('You need to provide a concrete account for the target stack when using cross-account or cross-region events');
                }
                if (!targetRegion || core_1.Token.isUnresolved(targetRegion)) {
                    throw new Error('You need to provide a concrete region for the target stack when using cross-account or cross-region events');
                }
                if (core_1.Token.isUnresolved(sourceAccount)) {
                    throw new Error('You need to provide a concrete account for the source stack when using cross-account or cross-region events');
                }
                // Don't exactly understand why this code was here (seems unlikely this rule would be violated), but
                // let's leave it in nonetheless.
                const sourceApp = this.node.root;
                if (!sourceApp || !core_1.App.isApp(sourceApp)) {
                    throw new Error('Event stack which uses cross-account or cross-region targets must be part of a CDK app');
                }
                const targetApp = constructs_1.Node.of(targetProps.targetResource).root;
                if (!targetApp || !core_1.App.isApp(targetApp)) {
                    throw new Error('Target stack which uses cross-account or cross-region event targets must be part of a CDK app');
                }
                if (sourceApp !== targetApp) {
                    throw new Error('Event stack and target stack must belong to the same CDK app');
                }
                // The target of this Rule will be the default event bus of the target environment
                this.ensureXEnvTargetEventBus(targetStack, targetAccount, targetRegion, id);
                // The actual rule lives in the target stack. Other than the account, it's identical to this one,
                // but only evaluated at render time (via a special subclass).
                //
                // FIXME: the MirrorRule is a bit silly, forwarding the exact same event to another event bus
                // and trigger on it there (there will be issues with construct references, for example). Especially
                // in the case of scheduled events, we will just trigger both rules in parallel in both environments.
                //
                // A better solution would be to have the source rule add a unique token to the the event,
                // and have the mirror rule trigger on that token only (thereby properly separating triggering, which
                // happens in the source env; and activating, which happens in the target env).
                //
                // Don't have time to do that right now.
                const mirrorRuleScope = this.obtainMirrorRuleScope(targetStack, targetAccount, targetRegion);
                new MirrorRule(mirrorRuleScope, `${core_1.Names.uniqueId(this)}-${id}`, {
                    targets: [target],
                    eventPattern: this.eventPattern,
                    schedule: this.scheduleExpression ? schedule_1.Schedule.expression(this.scheduleExpression) : undefined,
                    description: this.description,
                }, this);
                return;
            }
        }
        // Here only if the target does not have a targetResource defined.
        // In such case we don't have to generate any extra component.
        // Note that this can also be an imported resource (i.e: EventBus target)
        this.targets.push({
            id,
            arn: targetProps.arn,
            roleArn,
            ecsParameters: targetProps.ecsParameters,
            httpParameters: targetProps.httpParameters,
            kinesisParameters: targetProps.kinesisParameters,
            runCommandParameters: targetProps.runCommandParameters,
            batchParameters: targetProps.batchParameters,
            deadLetterConfig: targetProps.deadLetterConfig,
            retryPolicy: targetProps.retryPolicy,
            sqsParameters: targetProps.sqsParameters,
            input: inputProps && inputProps.input,
            inputPath: inputProps && inputProps.inputPath,
            inputTransformer: (inputProps === null || inputProps === void 0 ? void 0 : inputProps.inputTemplate) !== undefined ? {
                inputTemplate: inputProps.inputTemplate,
                inputPathsMap: inputProps.inputPathsMap,
            } : undefined,
        });
    }
    /**
     * Adds an event pattern filter to this rule. If a pattern was already specified,
     * these values are merged into the existing pattern.
     *
     * For example, if the rule already contains the pattern:
     *
     *    {
     *      "resources": [ "r1" ],
     *      "detail": {
     *        "hello": [ 1 ]
     *      }
     *    }
     *
     * And `addEventPattern` is called with the pattern:
     *
     *    {
     *      "resources": [ "r2" ],
     *      "detail": {
     *        "foo": [ "bar" ]
     *      }
     *    }
     *
     * The resulting event pattern will be:
     *
     *    {
     *      "resources": [ "r1", "r2" ],
     *      "detail": {
     *        "hello": [ 1 ],
     *        "foo": [ "bar" ]
     *      }
     *    }
     *
     */
    addEventPattern(eventPattern) {
        jsiiDeprecationWarnings._aws_cdk_aws_events_EventPattern(eventPattern);
        if (!eventPattern) {
            return;
        }
        util_1.mergeEventPattern(this.eventPattern, eventPattern);
    }
    /**
     * Not private only to be overrideen in CopyRule.
     *
     * @internal
     */
    _renderEventPattern() {
        return util_1.renderEventPattern(this.eventPattern);
    }
    validate() {
        if (Object.keys(this.eventPattern).length === 0 && !this.scheduleExpression) {
            return ['Either \'eventPattern\' or \'schedule\' must be defined'];
        }
        return [];
    }
    renderTargets() {
        if (this.targets.length === 0) {
            return undefined;
        }
        return this.targets;
    }
    /**
     * Make sure we add the target environments event bus as a target, and the target has permissions set up to receive our events
     *
     * For cross-account rules, uses a support stack to set up a policy on the target event bus.
     */
    ensureXEnvTargetEventBus(targetStack, targetAccount, targetRegion, id) {
        // the _actual_ target is just the event bus of the target's account
        // make sure we only add it once per account per region
        const key = `${targetAccount}:${targetRegion}`;
        if (this._xEnvTargetsAdded.has(key)) {
            return;
        }
        this._xEnvTargetsAdded.add(key);
        const eventBusArn = targetStack.formatArn({
            service: 'events',
            resource: 'event-bus',
            resourceName: 'default',
            region: targetRegion,
            account: targetAccount,
        });
        // For some reason, cross-region requires a Role (with `PutEvents` on the
        // target event bus) while cross-account doesn't
        const roleArn = !util_1.sameEnvDimension(targetRegion, core_1.Stack.of(this).region)
            ? this.crossRegionPutEventsRole(eventBusArn).roleArn
            : undefined;
        this.targets.push({
            id,
            arn: eventBusArn,
            roleArn,
        });
        // Add a policy to the target Event Bus to allow the source account/region to publish into it.
        //
        // Since this Event Bus permission needs to be deployed before the stack containing the Rule is deployed
        // (as EventBridge verifies whether you have permissions to the targets on rule creation), this needs
        // to be in a support stack.
        const sourceApp = this.node.root;
        const sourceAccount = core_1.Stack.of(this).account;
        // If different accounts, we need to add the permissions to the target eventbus
        //
        // For different region, no need for a policy on the target event bus (but a need
        // for a role).
        if (!util_1.sameEnvDimension(sourceAccount, targetAccount)) {
            const stackId = `EventBusPolicy-${sourceAccount}-${targetRegion}-${targetAccount}`;
            let eventBusPolicyStack = sourceApp.node.tryFindChild(stackId);
            if (!eventBusPolicyStack) {
                eventBusPolicyStack = new core_1.Stack(sourceApp, stackId, {
                    env: {
                        account: targetAccount,
                        region: targetRegion,
                    },
                    // The region in the stack name is rather redundant (it will always be the target region)
                    // Leaving it in for backwards compatibility.
                    stackName: `${targetStack.stackName}-EventBusPolicy-support-${targetRegion}-${sourceAccount}`,
                });
                new events_generated_1.CfnEventBusPolicy(eventBusPolicyStack, 'GivePermToOtherAccount', {
                    action: 'events:PutEvents',
                    statementId: `Allow-account-${sourceAccount}`,
                    principal: sourceAccount,
                });
            }
            // deploy the event bus permissions before the source stack
            core_1.Stack.of(this).addDependency(eventBusPolicyStack);
        }
    }
    /**
     * Return the scope where the mirror rule should be created for x-env event targets
     *
     * This is the target resource's containing stack if it shares the same region (owned
     * resources), or should be a fresh support stack for imported resources.
     *
     * We don't implement the second yet, as I have to think long and hard on whether we
     * can reuse the existing support stack or not, and I don't have time for that right now.
     */
    obtainMirrorRuleScope(targetStack, targetAccount, targetRegion) {
        // for cross-account or cross-region events, we cannot create new components for an imported resource
        // because we don't have the target stack
        if (util_1.sameEnvDimension(targetStack.account, targetAccount) && util_1.sameEnvDimension(targetStack.region, targetRegion)) {
            return targetStack;
        }
        // For now, we don't do the work for the support stack yet
        throw new Error('Cannot create a cross-account or cross-region rule for an imported resource (create a stack with the right environment for the imported resource)');
    }
    /**
     * Obtain the Role for the EventBridge event
     *
     * If a role already exists, it will be returned. This ensures that if multiple
     * events have the same target, they will share a role.
     * @internal
     */
    crossRegionPutEventsRole(eventBusArn) {
        const id = 'EventsRole';
        let role = this.node.tryFindChild(id);
        if (!role) {
            role = new aws_iam_1.Role(this, id, {
                roleName: core_1.PhysicalName.GENERATE_IF_NEEDED,
                assumedBy: new aws_iam_1.ServicePrincipal('events.amazonaws.com'),
            });
        }
        role.addToPrincipalPolicy(new aws_iam_1.PolicyStatement({
            actions: ['events:PutEvents'],
            resources: [eventBusArn],
        }));
        return role;
    }
}
exports.Rule = Rule;
_a = JSII_RTTI_SYMBOL_1;
Rule[_a] = { fqn: "@aws-cdk/aws-events.Rule", version: "1.148.0" };
/**
 * A rule that mirrors another rule
 */
class MirrorRule extends Rule {
    constructor(scope, id, props, source) {
        super(scope, id, props);
        this.source = source;
    }
    _renderEventPattern() {
        return this.source._renderEventPattern();
    }
    /**
     * Override validate to be a no-op
     *
     * The rules are never stored on this object so there's nothing to validate.
     *
     * Instead, we mirror the other rule at render time.
     */
    validate() {
        return [];
    }
}
//# sourceMappingURL=data:application/json;base64,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