import logging

import simplejson
from trosnoth.messages.base import AgentRequest, ServerCommand, ClientCommand
from trosnoth.utils.utils import timeNow

log = logging.getLogger()


class PlayerHasMacGuffin(ServerCommand):
    idString = b'mguf'
    fields = 'player_id', 'macguffin_id'
    packspec = 'cB'

    def applyOrderToWorld(self, world):
        player = world.getPlayer(self.player_id)
        world.macguffin_manager.received_transfer_message(player, self.macguffin_id)


class PlayerHasTrosballMsg(ServerCommand):
    idString = b'ball'
    fields = 'playerId'
    packspec = 'c'

    def applyOrderToWorld(self, world):
        player = world.getPlayer(self.playerId)
        world.trosballManager.gotPlayerHasTrosballMsg(player)

    def applyOrderToLocalState(self, localState, world):
        localState.revertTrosball()


class TrosballPositionMsg(ServerCommand):
    idString = b'bing'
    fields = 'xpos', 'ypos', 'xvel', 'yvel', 'inNet'
    packspec = 'ffff?'
    inNet = False

    def applyOrderToWorld(self, world):
        world.trosballManager.gotTrosballPositionMsg(
            (self.xpos, self.ypos), (self.xvel, self.yvel), self.inNet)

    def applyOrderToLocalState(self, localState, world):
        if localState.localTrosball:
            if localState.localTrosball.realShotStarted:
                localState.revertTrosball()
            else:
                localState.matchTrosball()


class ThrowTrosballMsg(AgentRequest):
    idString = b'pass'
    fields = 'tickId'
    packspec = 'H'

    def clientValidate(self, localState, world, sendResponse):
        if not localState.player:
            return False
        return localState.player.hasTrosball()

    def applyRequestToLocalState(self, localState):
        localState.trosballThrown()

    def serverApply(self, game, agent):
        if agent.player and agent.player.hasTrosball():
            game.world.trosballManager.throwTrosball()


class AchievementUnlockedMsg(ServerCommand):
    idString = b'Achm'
    fields = 'playerId', 'achievementId'
    packspec = 'c*'


class UpdateClockStateMsg(ServerCommand):
    idString = b'clok'
    fields = 'showing', 'counting', 'upwards', 'value', 'flashBelow'
    packspec = 'bbbff'

    def applyOrderToWorld(self, world):
        if world.isServer:
            # These messages are generated by World.clock.propagateToClients,
            # based off the existing state of the clock, so we don't need to
            #  set it again on the server.
            return
        world.clock.value = self.value
        world.clock.flashBelow = self.flashBelow
        world.clock.setMode(
            showing=self.showing,
            counting=self.counting,
            upwards=self.upwards,
        )


class PlaySoundMsg(ServerCommand):
    idString = b'noyz'
    fields = 'filename'
    packspec = '*'


class UpdateGameInfoMsg(ServerCommand):
    idString = b'info'
    fields = 'info'
    packspec = '*'

    @classmethod
    def build(cls, title, info, botGoal):
        return cls(simplejson.dumps([title, info, botGoal]).encode('utf-8'))

    def applyOrderToLocalState(self, localState, world):
        (
            localState.userTitle,
            localState.userInfo,
            localState.botGoal,
        ) = simplejson.loads(self.info.decode('utf-8'))
        localState.onGameInfoChanged()


class UpdateScoreBoardModeMsg(ServerCommand):
    idString = b'xorz'
    fields = 'teamScoresEnabled', 'playerScoresEnabled'
    packspec = 'bb'

    def applyOrderToWorld(self, world):
        world.scoreboard.gotUpdateScoreBoardModeMsg(
            self.teamScoresEnabled,
            self.playerScoresEnabled,
        )


class SetTeamAbilitiesMsg(ServerCommand):
    idString = b'tAbl'
    fields = 'teamId', 'data'
    packspec = 'c*'

    def applyOrderToWorld(self, world):
        world.getTeam(self.teamId).abilities.applyMessage(self.data)


class SetTeamScoreMsg(ServerCommand):
    idString = b'tXor'
    fields = 'teamId', 'score'
    packspec = 'cf'

    def applyOrderToWorld(self, world):
        team = world.getTeam(self.teamId)
        if team:
            world.scoreboard.gotTeamScoreMsg(team, self.score)


class SetPlayerAbilitiesMsg(ServerCommand):
    idString = b'pAbl'
    fields = 'playerId', 'data'
    packspec = 'c*'

    def applyOrderToWorld(self, world):
        world.getPlayer(self.playerId).abilities.applyMessage(self.data)


class SetPlayerScoreMsg(ServerCommand):
    idString = b'pXor'
    fields = 'playerId', 'score'
    packspec = 'cf'

    def applyOrderToWorld(self, world):
        player = world.getPlayer(self.playerId)
        if player:
            world.scoreboard.gotPlayerScoreMsg(player, self.score)



class SetUIOptionsMsg(ServerCommand):
    idString = b'uiOp'
    fields = 'data'
    packspec = '*'
    isControl = True

    def applyOrderToWorld(self, world):
        world.uiOptions.applyMessage(self.data)


class SetWorldAbilitiesMsg(ServerCommand):
    idString = b'able'
    fields = 'data'
    packspec = '*'

    def applyOrderToWorld(self, world):
        world.abilities.gotSetWorldAbilitiesMsg(
            simplejson.loads(self.data.decode('utf-8')))


class PingMsg(ClientCommand):
    idString = b'ping'
    fields = 'data'
    packspec = '*'
    isControl = True

    def applyRequestToLocalState(self, localState):
        localState.pings[self.data] = timeNow()

    def serverApply(self, game, agent):
        agent.messageToAgent(self)

    def applyOrderToLocalState(self, localState, world):
        now = timeNow()
        try:
            sendTime = localState.pings.pop(self.data)
        except KeyError:
            return
        localState.gotPingTime(now - sendTime)
