"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.AcceleratorSecurityGroup = void 0;
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const custom_resources_1 = require("@aws-cdk/custom-resources");
/**
 * (experimental) The security group used by a Global Accelerator to send traffic to resources in a VPC.
 *
 * @experimental
 */
class AcceleratorSecurityGroup {
    /**
     * (experimental) Lookup the Global Accelerator security group at CloudFormation deployment time.
     *
     * As of this writing, Global Accelerators (AGA) create a single security group per VPC. AGA security groups are shared
     * by all AGAs in an account. Additionally, there is no CloudFormation mechanism to reference the AGA security groups.
     *
     * This makes creating security group rules which allow traffic from an AGA complicated in CDK. This lookup will identify
     * the AGA security group for a given VPC at CloudFormation deployment time, and lets you create rules for traffic from AGA
     * to other resources created by CDK.
     *
     * @experimental
     */
    static fromVpc(scope, id, vpc, endpointGroup) {
        // The security group name is always 'GlobalAccelerator'
        const globalAcceleratorSGName = 'GlobalAccelerator';
        // How to reference the security group name in the response from EC2
        const ec2ResponseSGIdField = 'SecurityGroups.0.GroupId';
        // The AWS Custom Resource that make a call to EC2 to get the security group ID, for the given VPC
        const lookupAcceleratorSGCustomResource = new custom_resources_1.AwsCustomResource(scope, id + 'CustomResource', {
            onCreate: {
                service: 'EC2',
                action: 'describeSecurityGroups',
                parameters: {
                    Filters: [
                        {
                            Name: 'group-name',
                            Values: [
                                globalAcceleratorSGName,
                            ],
                        },
                        {
                            Name: 'vpc-id',
                            Values: [
                                vpc.vpcId,
                            ],
                        },
                    ],
                },
                // We get back a list of responses, but the list should be of length 0 or 1
                // Getting no response means no resources have been linked to the AGA
                physicalResourceId: custom_resources_1.PhysicalResourceId.fromResponse(ec2ResponseSGIdField),
            },
            policy: custom_resources_1.AwsCustomResourcePolicy.fromSdkCalls({
                resources: custom_resources_1.AwsCustomResourcePolicy.ANY_RESOURCE,
            }),
        });
        // Look up the security group ID
        const sg = aws_ec2_1.SecurityGroup.fromSecurityGroupId(scope, id, lookupAcceleratorSGCustomResource.getResponseField(ec2ResponseSGIdField));
        // We add a dependency on the endpoint group, guaranteeing that CloudFormation won't
        // try and look up the SG before AGA creates it. The SG is created when a VPC resource
        // is associated with an AGA
        lookupAcceleratorSGCustomResource.node.addDependency(endpointGroup);
        return sg;
    }
    constructor() { }
}
exports.AcceleratorSecurityGroup = AcceleratorSecurityGroup;
//# sourceMappingURL=data:application/json;base64,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