"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.PlaceIndex = exports.IntendedUse = exports.DataSource = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const iam = require("aws-cdk-lib/aws-iam");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_location_1 = require("aws-cdk-lib/aws-location");
/**
 * Data source for a place index
 */
var DataSource;
(function (DataSource) {
    /**
     * Esri
     *
     * @see https://docs.aws.amazon.com/location/latest/developerguide/esri.html
     */
    DataSource["ESRI"] = "Esri";
    /**
     * HERE
     *
     * @see https://docs.aws.amazon.com/location/latest/developerguide/HERE.html
     */
    DataSource["HERE"] = "Here";
})(DataSource = exports.DataSource || (exports.DataSource = {}));
/**
 * Intend use for the results of an operation
 */
var IntendedUse;
(function (IntendedUse) {
    /**
     * The results won't be stored
     */
    IntendedUse["SINGLE_USE"] = "SingleUse";
    /**
     * The result can be cached or stored in a database
     */
    IntendedUse["STORAGE"] = "Storage";
})(IntendedUse = exports.IntendedUse || (exports.IntendedUse = {}));
class PlaceIndexBase extends aws_cdk_lib_1.Resource {
    /**
     * Grant the given principal identity permissions to perform the actions on this place index.
     */
    grant(grantee, ...actions) {
        return iam.Grant.addToPrincipal({
            grantee: grantee,
            actions: actions,
            resourceArns: [this.placeIndexArn],
        });
    }
    /**
     * Grant the given identity permissions to search using this index
     */
    grantSearch(grantee) {
        return this.grant(grantee, 'geo:SearchPlaceIndexForPosition', 'geo:SearchPlaceIndexForSuggestions', 'geo:SearchPlaceIndexForText');
    }
}
/**
 * A Place Index
 *
 * @see https://docs.aws.amazon.com/location/latest/developerguide/places-concepts.html
 */
class PlaceIndex extends PlaceIndexBase {
    constructor(scope, id, props = {}) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_location_alpha_PlaceIndexProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, PlaceIndex);
            }
            throw error;
        }
        if (props.placeIndexName && !aws_cdk_lib_1.Token.isUnresolved(props.placeIndexName) && !/^[-.\w]{1,100}$/.test(props.placeIndexName)) {
            throw new Error(`Invalid place index name. The place index name must be between 1 and 100 characters and contain only alphanumeric characters, hyphens, periods and underscores. Received: ${props.placeIndexName}`);
        }
        super(scope, id, {
            physicalName: props.placeIndexName ?? aws_cdk_lib_1.Lazy.string({ produce: () => this.generateUniqueId() }),
        });
        const placeIndex = new aws_location_1.CfnPlaceIndex(this, 'Resource', {
            indexName: this.physicalName,
            dataSource: props.dataSource ?? DataSource.ESRI,
            dataSourceConfiguration: props.intendedUse
                ? { intendedUse: props.intendedUse }
                : undefined,
            description: props.description,
        });
        this.placeIndexName = placeIndex.ref;
        this.placeIndexArn = placeIndex.attrArn;
        this.placeIndexCreateTime = placeIndex.attrCreateTime;
        this.placeIndexUpdateTime = placeIndex.attrUpdateTime;
    }
    /**
     * Use an existing place index by name
     */
    static fromPlaceIndexName(scope, id, placeIndexName) {
        const placeIndexArn = aws_cdk_lib_1.Stack.of(scope).formatArn({
            service: 'geo',
            resource: 'place-index',
            resourceName: placeIndexName,
        });
        return PlaceIndex.fromPlaceIndexArn(scope, id, placeIndexArn);
    }
    /**
     * Use an existing place index by ARN
     */
    static fromPlaceIndexArn(scope, id, placeIndexArn) {
        const parsedArn = aws_cdk_lib_1.Stack.of(scope).splitArn(placeIndexArn, aws_cdk_lib_1.ArnFormat.SLASH_RESOURCE_NAME);
        if (!parsedArn.resourceName) {
            throw new Error(`Place Index Arn ${placeIndexArn} does not have a resource name.`);
        }
        class Import extends PlaceIndexBase {
            constructor() {
                super(...arguments);
                this.placeIndexName = parsedArn.resourceName;
                this.placeIndexArn = placeIndexArn;
            }
        }
        return new Import(scope, id, {
            account: parsedArn.account,
            region: parsedArn.region,
        });
    }
    generateUniqueId() {
        const name = aws_cdk_lib_1.Names.uniqueId(this);
        if (name.length > 100) {
            return name.substring(0, 50) + name.substring(name.length - 50);
        }
        return name;
    }
}
exports.PlaceIndex = PlaceIndex;
_a = JSII_RTTI_SYMBOL_1;
PlaceIndex[_a] = { fqn: "@aws-cdk/aws-location-alpha.PlaceIndex", version: "2.61.1-alpha.0" };
//# sourceMappingURL=data:application/json;base64,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