import numpy as np
import logging

# custom modules
import graph_ltpl

# custom packages
import trajectory_planning_helpers as tph


def main_online_path_gen(graph_base: graph_ltpl.data_objects.GraphBase.GraphBase,
                         start_node: tuple,
                         obj_veh: list,
                         obj_zone: list,
                         action_sets=True,
                         last_action_id: str = None,
                         max_solutions=1,
                         const_path_seg: np.ndarray = None,
                         pos_est: np.ndarray = None,
                         last_solution_nodes: list = None,
                         w_last_edges: list = ()) -> tuple:
    """
    The main function to be called iteratively in an online environment to calculate path solutions. The following steps
    are executed in an iterative manner:

    * Represent all objects (e.g. vehicles) within the planning range in the graph
    * Define action sets to be generated by the planner (e.g. keep straight, overtake on the left, ...) and modify graph
      for each action primitive correspondingly
    * Execute path search for each action set
    * Calculate curvature continuous spline for each solution

    :param graph_base:              reference to the GraphBase object instance holding relevant parameters
    :param start_node:              tuple of layer and node id (a tracking of the position and fixed trajectories should
                                    be handled in the main script
    :param obj_veh:                 list of objects holding info like "pos", "ids", "vel", "radius"
    :param obj_zone:                list of objects holding information about the blocked zones
    :param action_sets:             flag, whether action sets should be used or not (else just a single solution)
    :param last_action_id:          string holding the previously executed action id
    :param max_solutions:           maximum number of path solutions to be returned (NOTE: 1 is the fastest!)
    :param const_path_seg:          path segment (x,y,psi,kappa,s) in the path planning process, which remains constant
    :param pos_est:                 estimated position of the ego vehicle
    :param last_solution_nodes:     nodes of the last valid solution -> used to reduce cost along last valid path
    :param w_last_edges:            online cost reduction per previously planned path segment
    :returns:
        * **action_set_nodes** -    dict for each action containing a list with one or multiple arrays holding tuples of
          node id sequences for determined local solutions
        * **action_set_node_idx** - dict for each action containing a list with one or multiple arrays holding the
          indexes of the node candidates in the coords arrays
        * **action_set_coeff** -    dict for each action containing a list with one or multiple arrays holding the
          spline coefficients Nx8 (N spline segments) for the determined local solutions
        * **action_set_path_param** - dict for each action containing a numpy array with one or multiple arrays holding
          the x-, y-coordinates, psi, kappa and el-length for the determined local solutions
        * **closest_obj_index** -   index of the object in the provided array, which is closest upfront ego-vehicle
          ("None", if no object in planning range)

    :Authors:
        * Tim Stahl <tim.stahl@tum.de>

    :Created on:
        07.11.2018

    """

    # ------------------------------------------------------------------------------------------------------------------
    # - Handle objects within the planning horizon ---------------------------------------------------------------------
    # ------------------------------------------------------------------------------------------------------------------
    # Generate local node template for the current action_set (block edges occupied by objects)
    end_layer, closest_obj_index, closest_obj_node = graph_ltpl.online_graph.src.gen_local_node_template.\
        gen_local_node_template(graph_base=graph_base,
                                start_node=start_node,
                                obj_veh=obj_veh,
                                obj_zone=obj_zone,
                                last_solution_nodes=last_solution_nodes,
                                w_last_edges=w_last_edges)

    # check for objects in constant path segment (if provided)
    obj_in_const_path = False
    object_besides_const_path = False
    if const_path_seg is not None and np.size(const_path_seg, axis=0) >= 2:
        # get start pos (use pos_est if provided, else use start of constant path [may contain passed path])
        if pos_est is not None:
            pos_start = pos_est
        else:
            pos_start = const_path_seg[0, 0:2]

        s_start, _ = graph_ltpl.helper_funcs.src.get_s_coord.get_s_coord(ref_line=graph_base.raceline,
                                                                         pos=pos_start,
                                                                         s_array=graph_base.s_raceline,
                                                                         closed=True)

        s_end, _ = graph_ltpl.helper_funcs.src.get_s_coord.get_s_coord(ref_line=graph_base.raceline,
                                                                       pos=const_path_seg[-1, 0:2],
                                                                       s_array=graph_base.s_raceline,
                                                                       closed=True)

        smallest_obj_dist = np.Inf
        for obj_idx, vehicle in enumerate(obj_veh):
            s_obj, _ = graph_ltpl.helper_funcs.src.get_s_coord.get_s_coord(ref_line=graph_base.raceline,
                                                                           pos=vehicle.get_pos(),
                                                                           s_array=graph_base.s_raceline,
                                                                           closed=True)

            # if object within constant path segment
            if s_start <= s_obj <= s_end or (s_start > s_end and (s_obj > s_start or s_obj < s_end)):
                object_besides_const_path = True

                # calculate object distance
                if s_obj < s_start:
                    obj_dist = s_obj + graph_base.s_raceline[-1] - s_start
                else:
                    obj_dist = s_obj - s_start

                if closest_obj_index is None or obj_dist < smallest_obj_dist:
                    closest_obj_index = obj_idx
                    smallest_obj_dist = obj_dist

                # check if constant path intersects with object
                obstacle_ref = np.power(vehicle.get_radius() + graph_base.veh_width / 2, 2)
                distances2 = np.power(const_path_seg[:, 0] - vehicle.get_pos()[0], 2) + np.power(
                    const_path_seg[:, 1] - vehicle.get_pos()[1], 2)
                if any(distances2 <= obstacle_ref):
                    obj_in_const_path = True

    # -- DEFINE ACTION SETS TO BE GENERATED ----------------------------------------------------------------------------
    # NOTE: Ensure to first list the straight/follow options (if the planning horizon is reduced there, it will be also
    #       reduced for the other primitives
    # if an object is located in the constant path section (either overtaking or follow mode)
    if action_sets and (obj_in_const_path or object_besides_const_path):
        action_set_filters = ["planning_range"]
        action_set_goal_layer = end_layer
        action_set_names = ["follow"]

        # if currently in an overtaking maneuver-> offer straight with overtaking action ID
        if not obj_in_const_path and (last_action_id == "left" or last_action_id == "right"):
            action_set_filters.append("default")
            action_set_names.append(last_action_id)

        # if object only besides path, but no overtaking active -> offer left and right
        elif not obj_in_const_path:
            # array of all filters
            action_set_filters.extend(["default", "default"])
            action_set_names.extend(["left", "right"])

    # if action sets enabled and a vehicle in the planning horizon
    elif action_sets and closest_obj_index is not None and closest_obj_node is not None:
        # -- generate node filters for each action set (overtake left, overtake right, follow) --
        # overtake left
        block_set = list(range(closest_obj_node[1], graph_base.nodes_in_layer[closest_obj_node[0]]))
        graph_base.remove_nodes_filter(layer_ids=[closest_obj_node[0]] * len(block_set),
                                       node_ids=block_set,
                                       applied_filter="overtake_left_filter",
                                       base="default")

        # overtake right
        block_set = list(range(0, closest_obj_node[1]))
        graph_base.remove_nodes_filter(layer_ids=[closest_obj_node[0]] * len(block_set),
                                       node_ids=block_set,
                                       applied_filter="overtake_right_filter",
                                       base="default")

        # straight
        # block_set = list(range(0, graph_base.nodes_in_layer[closest_obstacle_node[0]]))
        # block_filter = copy.deepcopy(local_node_filter)
        # graph_base.update_node_filter(layer_ids=[closest_obstacle_node[0]] * len(block_set),
        #                               node_ids=block_set,
        #                               applied_filter=block_filter)
        # array of all filters
        action_set_filters = ["planning_range", "overtake_left_filter", "overtake_right_filter"]
        action_set_goal_layer = end_layer
        action_set_names = ["follow", "left", "right"]
    else:
        action_set_filters = ["default"]
        action_set_goal_layer = end_layer
        action_set_names = ["straight"]

    # ------------------------------------------------------------------------------------------------------------------
    # - Path search and assembly of spline segments --------------------------------------------------------------------
    # ------------------------------------------------------------------------------------------------------------------

    # Init output variables
    action_set_nodes = dict()
    action_set_node_idx = dict()
    action_set_coeff = dict()
    action_set_path_param = dict()  # x, y, psi, kappa, el length (tailing zero)
    action_set_red_len = dict()  # dict holding a True value for primitives with a reduced planning horizon

    mod_action_set_goal_layer = action_set_goal_layer
    for action_set_filter, action_set_name in \
            zip(action_set_filters, action_set_names):

        # activate action_set_filter
        graph_base.activate_filter(applied_filter=action_set_filter)

        # Check if  object is not on planning start layer (except for follow)
        # if closest_obj_node is not None and start_node[0] == closest_obj_node[0] and action_set_name != "follow":
        #     lat_displ = abs(start_node[1] - closest_obj_node[1]) * graph_base.lat_resolution - \
        #                 graph_base.veh_width / 2 - obj_veh[closest_obj_index].get_radius()
        #     if lat_displ < 0:
        #         continue

        loc_path_nodes_list = None

        while True:
            # check if start layer is equal to goal layer
            if mod_action_set_goal_layer == start_node[0]:
                break

            # Trigger graph search (if goal node exists)
            _, loc_path_nodes_list = graph_base.search_graph_layer(start_layer=start_node[0],
                                                                   start_node=start_node[1],
                                                                   end_layer=mod_action_set_goal_layer,
                                                                   max_solutions=max_solutions)

            # exit if solution is found or when planning a special maneuver (e.g. overtaking -> use same goal as s/f)
            if loc_path_nodes_list is not None or not (action_set_name == "follow" or action_set_name == "straight"):
                break
            else:
                mod_action_set_goal_layer -= 1
                if mod_action_set_goal_layer < 0:
                    mod_action_set_goal_layer = graph_base.num_layers - 1

        # If planning horizon was reduced (either by vehicles or end of open track reached)
        reduced_horizon = (mod_action_set_goal_layer != action_set_goal_layer
                           or (not graph_base.closed and action_set_goal_layer == graph_base.num_layers - 1))
        if reduced_horizon:
            # Check if vehicle is still in range
            obj_in_mod_range = (closest_obj_node is not None
                                and ((start_node[0] <= closest_obj_node[0] <= mod_action_set_goal_layer)
                                     or (start_node[0] > mod_action_set_goal_layer
                                         and (closest_obj_node[0] >= start_node[0]
                                              or closest_obj_node[0] <= mod_action_set_goal_layer))))
            if not obj_in_const_path and closest_obj_node is not None and not obj_in_mod_range:
                # rename follow mode to straight and remove overtaking options
                if action_set_name == "follow" or action_set_name == "straight":
                    action_set_name = "straight"
                    logging.getLogger("local_trajectory_logger"). \
                        info("No feasible solution for '" + action_set_name + "'! Reduced planning horizon!")
                else:
                    # remove all other options
                    loc_path_nodes_list = None
            else:
                logging.getLogger("local_trajectory_logger"). \
                    info("No feasible solution for '" + action_set_name + "'! Reduced planning horizon!")

        # If no solution is found (blocked graph), end iteration and leave action set list empty
        if loc_path_nodes_list is None:
            logging.getLogger("local_trajectory_logger").\
                debug("Action set '" + action_set_name + "' is empty! No path solution was found.")
        else:
            # Store node sequence in action set
            action_set_nodes[action_set_name] = loc_path_nodes_list

            # init dict entry for action set (in order to allow multiple trajectories per set)
            action_set_node_idx[action_set_name] = []
            action_set_coeff[action_set_name] = []
            action_set_path_param[action_set_name] = []
            action_set_red_len[action_set_name] = []

            # For every path in list
            for loc_path_nodes in loc_path_nodes_list:
                loc_path_node_idx = [0]

                # spline_coeff_mat = np.zeros((len(loc_path_nodes)-1, 8))

                # Initialize list for all path components (since length is not beforehand, lists are faster than numpy)
                spline_param_fuse = np.empty((0, 5))
                dists = np.empty(0)

                previous_node = None

                # Assemble data from path segments in one array per path
                for count, current_node in enumerate(loc_path_nodes):
                    if previous_node is not None:
                        spline_coeff, spline_param, offline_cost, spline_length = \
                            graph_base.get_edge(start_layer=previous_node[0],
                                                start_node=previous_node[1],
                                                end_layer=current_node[0],
                                                end_node=current_node[1])

                        # Sort spline coefficients in large numpy matrix Nx8
                        # spline_coeff_mat[count-1, :] = np.array(spline_coeff).flatten()

                        # Remove last value of coordinate, psi and curvature, since they appear twice at transitions
                        # between two spline segments -> indicator that is one except in the last iteration
                        if count != len(loc_path_nodes) - 1:
                            rmv_el = -1
                        else:
                            rmv_el = None

                        # Add the coordinates to a single array
                        spline_param_fuse = np.append(spline_param_fuse, spline_param[:rmv_el, :], axis=0)
                        dists = np.append(dists, spline_length)

                        # Provide indexes of loc_path_nodes in loc_path_coords
                        loc_path_node_idx.append(np.size(spline_param_fuse, axis=0) - 1 * (rmv_el is None))

                    previous_node = current_node

                # Calculate curvature continous spline
                if const_path_seg is not None:
                    psi_s = const_path_seg[-1, 2]
                else:
                    psi_s = spline_param_fuse[0, 2]

                spline_coeff_mat = np.column_stack(tph.calc_splines.
                                                   calc_splines(path=spline_param_fuse[loc_path_node_idx, 0:2],
                                                                psi_s=psi_s,
                                                                psi_e=spline_param_fuse[-1, 2],
                                                                el_lengths=dists)[0:2])

                # Calculate new curvature, coordinates and headings based on splines
                spline_param_fuse[:, 0:2], tmp_spline_inds, tmp_t_values, _ = \
                    tph.interp_splines.interp_splines(coeffs_x=spline_coeff_mat[:, :4],
                                                      coeffs_y=spline_coeff_mat[:, 4:],
                                                      incl_last_point=True,
                                                      stepnum_fixed=(np.diff(loc_path_node_idx) + 1).tolist())

                spline_param_fuse[:, 2], spline_param_fuse[:, 3] = tph.calc_head_curv_an.\
                    calc_head_curv_an(coeffs_x=spline_coeff_mat[:, :4],
                                      coeffs_y=spline_coeff_mat[:, 4:],
                                      ind_spls=tmp_spline_inds,
                                      t_spls=tmp_t_values)

                # add trajectory candidate to action set
                action_set_coeff[action_set_name].append(spline_coeff_mat)
                action_set_node_idx[action_set_name].append(loc_path_node_idx)
                action_set_path_param[action_set_name].append(spline_param_fuse)
                action_set_red_len[action_set_name].append(reduced_horizon)

    # deactivate present filters
    graph_base.deactivate_filter()

    return (action_set_nodes, action_set_node_idx, action_set_coeff, action_set_path_param, action_set_red_len,
            closest_obj_index)
