"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CfnInclude = void 0;
const core = require("@aws-cdk/core");
const cfn_parse = require("@aws-cdk/core/lib/cfn-parse");
const cfn_type_to_l1_mapping = require("./cfn-type-to-l1-mapping");
const futils = require("./file-utils");
/**
 * Construct to import an existing CloudFormation template file into a CDK application.
 * All resources defined in the template file can be retrieved by calling the {@link getResource} method.
 * Any modifications made on the returned resource objects will be reflected in the resulting CDK template.
 */
class CfnInclude extends core.CfnElement {
    constructor(scope, id, props) {
        var _a;
        super(scope, id);
        this.conditions = {};
        this.resources = {};
        this.parameters = {};
        this.mappings = {};
        this.rules = {};
        this.hooks = {};
        this.outputs = {};
        this.nestedStacks = {};
        this.parametersToReplace = props.parameters || {};
        // read the template into a JS object
        this.template = futils.readYamlSync(props.templateFile);
        this.preserveLogicalIds = (_a = props.preserveLogicalIds) !== null && _a !== void 0 ? _a : true;
        // check if all user specified parameter values exist in the template
        for (const logicalId of Object.keys(this.parametersToReplace)) {
            if (!(logicalId in (this.template.Parameters || {}))) {
                throw new Error(`Parameter with logical ID '${logicalId}' was not found in the template`);
            }
        }
        // instantiate the Mappings
        this.mappingsScope = new core.Construct(this, '$Mappings');
        for (const mappingName of Object.keys(this.template.Mappings || {})) {
            this.createMapping(mappingName);
        }
        // instantiate all parameters
        for (const logicalId of Object.keys(this.template.Parameters || {})) {
            this.createParameter(logicalId);
        }
        // instantiate the conditions
        this.conditionsScope = new core.Construct(this, '$Conditions');
        for (const conditionName of Object.keys(this.template.Conditions || {})) {
            this.getOrCreateCondition(conditionName);
        }
        // instantiate the rules
        this.rulesScope = new core.Construct(this, '$Rules');
        for (const ruleName of Object.keys(this.template.Rules || {})) {
            this.createRule(ruleName);
        }
        this.nestedStacksToInclude = props.loadNestedStacks || {};
        // instantiate all resources as CDK L1 objects
        for (const logicalId of Object.keys(this.template.Resources || {})) {
            this.getOrCreateResource(logicalId);
        }
        // verify that all nestedStacks have been instantiated
        for (const nestedStackId of Object.keys(props.loadNestedStacks || {})) {
            if (!(nestedStackId in this.resources)) {
                throw new Error(`Nested Stack with logical ID '${nestedStackId}' was not found in the template`);
            }
        }
        // instantiate the Hooks
        this.hooksScope = new core.Construct(this, '$Hooks');
        for (const hookName of Object.keys(this.template.Hooks || {})) {
            this.createHook(hookName);
        }
        const outputScope = new core.Construct(this, '$Ouputs');
        for (const logicalId of Object.keys(this.template.Outputs || {})) {
            this.createOutput(logicalId, outputScope);
        }
    }
    /**
     * Returns the low-level CfnResource from the template with the given logical ID.
     * Any modifications performed on that resource will be reflected in the resulting CDK template.
     *
     * The returned object will be of the proper underlying class;
     * you can always cast it to the correct type in your code:
     *
     *     // assume the template contains an AWS::S3::Bucket with logical ID 'Bucket'
     *     const cfnBucket = cfnTemplate.getResource('Bucket') as s3.CfnBucket;
     *     // cfnBucket is of type s3.CfnBucket
     *
     * If the template does not contain a resource with the given logical ID,
     * an exception will be thrown.
     *
     * @param logicalId the logical ID of the resource in the CloudFormation template file
     */
    getResource(logicalId) {
        const ret = this.resources[logicalId];
        if (!ret) {
            throw new Error(`Resource with logical ID '${logicalId}' was not found in the template`);
        }
        return ret;
    }
    /**
     * Returns the CfnCondition object from the 'Conditions'
     * section of the CloudFormation template with the given name.
     * Any modifications performed on that object will be reflected in the resulting CDK template.
     *
     * If a Condition with the given name is not present in the template,
     * throws an exception.
     *
     * @param conditionName the name of the Condition in the CloudFormation template file
     */
    getCondition(conditionName) {
        const ret = this.conditions[conditionName];
        if (!ret) {
            throw new Error(`Condition with name '${conditionName}' was not found in the template`);
        }
        return ret;
    }
    /**
     * Returns the CfnParameter object from the 'Parameters'
     * section of the included template.
     * Any modifications performed on that object will be reflected in the resulting CDK template.
     *
     * If a Parameter with the given name is not present in the template,
     * throws an exception.
     *
     * @param parameterName the name of the parameter to retrieve
     */
    getParameter(parameterName) {
        const ret = this.parameters[parameterName];
        if (!ret) {
            throw new Error(`Parameter with name '${parameterName}' was not found in the template`);
        }
        return ret;
    }
    /**
     * Returns the CfnMapping object from the 'Mappings' section of the included template.
     * Any modifications performed on that object will be reflected in the resulting CDK template.
     *
     * If a Mapping with the given name is not present in the template,
     * an exception will be thrown.
     *
     * @param mappingName the name of the Mapping in the template to retrieve
     */
    getMapping(mappingName) {
        const ret = this.mappings[mappingName];
        if (!ret) {
            throw new Error(`Mapping with name '${mappingName}' was not found in the template`);
        }
        return ret;
    }
    /**
     * Returns the CfnOutput object from the 'Outputs'
     * section of the included template.
     * Any modifications performed on that object will be reflected in the resulting CDK template.
     *
     * If an Output with the given name is not present in the template,
     * throws an exception.
     *
     * @param logicalId the name of the output to retrieve
     */
    getOutput(logicalId) {
        const ret = this.outputs[logicalId];
        if (!ret) {
            throw new Error(`Output with logical ID '${logicalId}' was not found in the template`);
        }
        return ret;
    }
    /**
     * Returns the CfnRule object from the 'Rules'
     * section of the CloudFormation template with the given name.
     * Any modifications performed on that object will be reflected in the resulting CDK template.
     *
     * If a Rule with the given name is not present in the template,
     * an exception will be thrown.
     *
     * @param ruleName the name of the Rule in the CloudFormation template
     */
    getRule(ruleName) {
        const ret = this.rules[ruleName];
        if (!ret) {
            throw new Error(`Rule with name '${ruleName}' was not found in the template`);
        }
        return ret;
    }
    /**
     * Returns the CfnHook object from the 'Hooks'
     * section of the included CloudFormation template with the given logical ID.
     * Any modifications performed on the returned object will be reflected in the resulting CDK template.
     *
     * If a Hook with the given logical ID is not present in the template,
     * an exception will be thrown.
     *
     * @param hookLogicalId the logical ID of the Hook in the included CloudFormation template's 'Hooks' section
     */
    getHook(hookLogicalId) {
        const ret = this.hooks[hookLogicalId];
        if (!ret) {
            throw new Error(`Hook with logical ID '${hookLogicalId}' was not found in the template`);
        }
        return ret;
    }
    /**
     * Returns a loaded NestedStack with name logicalId.
     * For a nested stack to be returned by this method,
     * it must be specified either in the {@link CfnIncludeProps.loadNestedStacks} property,
     * or through the {@link loadNestedStack} method.
     *
     * @param logicalId the ID of the stack to retrieve, as it appears in the template
     */
    getNestedStack(logicalId) {
        if (!this.nestedStacks[logicalId]) {
            if (!this.template.Resources[logicalId]) {
                throw new Error(`Nested Stack with logical ID '${logicalId}' was not found in the template`);
            }
            else if (this.template.Resources[logicalId].Type !== 'AWS::CloudFormation::Stack') {
                throw new Error(`Resource with logical ID '${logicalId}' is not a CloudFormation Stack`);
            }
            else {
                throw new Error(`Nested Stack '${logicalId}' was not included in the parent template. ` +
                    'To retrieve an included nested stack, it must be specified either in the `loadNestedStacks` property, or through the `loadNestedStack` method');
            }
        }
        return this.nestedStacks[logicalId];
    }
    /**
     * Includes a template for a child stack inside of this parent template.
     * A child with this logical ID must exist in the template,
     * and be of type AWS::CloudFormation::Stack.
     * This is equivalent to specifying the value in the {@link CfnIncludeProps.loadNestedStacks}
     * property on object construction.
     *
     * @param logicalId the ID of the stack to retrieve, as it appears in the template
     * @param nestedStackProps the properties of the included child Stack
     * @returns the same {@link IncludedNestedStack} object that {@link getNestedStack} returns for this logical ID
     */
    loadNestedStack(logicalId, nestedStackProps) {
        if (logicalId in this.nestedStacks) {
            throw new Error(`Nested Stack '${logicalId}' was already included in its parent template`);
        }
        const cfnStack = this.resources[logicalId];
        if (!cfnStack) {
            throw new Error(`Nested Stack with logical ID '${logicalId}' was not found in the template`);
        }
        if (cfnStack instanceof core.CfnStack) {
            // delete the old CfnStack child - one will be created by the NestedStack object
            this.node.tryRemoveChild(logicalId);
            // remove the previously created CfnStack resource from the resources map
            delete this.resources[logicalId];
            // createNestedStack() (called by getOrCreateResource()) expects this to be filled
            this.nestedStacksToInclude[logicalId] = nestedStackProps;
            this.getOrCreateResource(logicalId);
            return this.nestedStacks[logicalId];
        }
        else {
            throw new Error(`Nested Stack with logical ID '${logicalId}' is not an AWS::CloudFormation::Stack resource`);
        }
    }
    /** @internal */
    _toCloudFormation() {
        const ret = {};
        for (const section of Object.keys(this.template)) {
            const self = this;
            const finder = {
                findResource(lId) {
                    return self.resources[lId];
                },
                findRefTarget(elementName) {
                    var _a;
                    return (_a = self.resources[elementName]) !== null && _a !== void 0 ? _a : self.parameters[elementName];
                },
                findCondition(conditionName) {
                    return self.conditions[conditionName];
                },
                findMapping(mappingName) {
                    return self.mappings[mappingName];
                },
            };
            const cfnParser = new cfn_parse.CfnParser({
                finder,
                parameters: this.parametersToReplace,
            });
            switch (section) {
                case 'Conditions':
                case 'Mappings':
                case 'Resources':
                case 'Parameters':
                case 'Rules':
                case 'Hooks':
                case 'Outputs':
                    // these are rendered as a side effect of instantiating the L1s
                    break;
                default:
                    ret[section] = cfnParser.parseValue(this.template[section]);
            }
        }
        return ret;
    }
    createMapping(mappingName) {
        const cfnParser = new cfn_parse.CfnParser({
            finder: {
                findCondition() { throw new Error('Referring to Conditions in Mapping definitions is not allowed'); },
                findMapping() { throw new Error('Referring to other Mappings in Mapping definitions is not allowed'); },
                findRefTarget() { throw new Error('Using Ref expressions in Mapping definitions is not allowed'); },
                findResource() { throw new Error('Using GetAtt expressions in Mapping definitions is not allowed'); },
            },
            parameters: {},
        });
        const cfnMapping = new core.CfnMapping(this.mappingsScope, mappingName, {
            mapping: cfnParser.parseValue(this.template.Mappings[mappingName]),
        });
        this.mappings[mappingName] = cfnMapping;
        this.overrideLogicalIdIfNeeded(cfnMapping, mappingName);
    }
    createParameter(logicalId) {
        if (logicalId in this.parametersToReplace) {
            return;
        }
        const expression = new cfn_parse.CfnParser({
            finder: {
                findResource() { throw new Error('Using GetAtt expressions in Parameter definitions is not allowed'); },
                findRefTarget() { throw new Error('Using Ref expressions in Parameter definitions is not allowed'); },
                findCondition() { throw new Error('Referring to Conditions in Parameter definitions is not allowed'); },
                findMapping() { throw new Error('Referring to Mappings in Parameter definitions is not allowed'); },
            },
            parameters: {},
        }).parseValue(this.template.Parameters[logicalId]);
        const cfnParameter = new core.CfnParameter(this, logicalId, {
            type: expression.Type,
            default: expression.Default,
            allowedPattern: expression.AllowedPattern,
            allowedValues: expression.AllowedValues,
            constraintDescription: expression.ConstraintDescription,
            description: expression.Description,
            maxLength: expression.MaxLength,
            maxValue: expression.MaxValue,
            minLength: expression.MinLength,
            minValue: expression.MinValue,
            noEcho: expression.NoEcho,
        });
        this.overrideLogicalIdIfNeeded(cfnParameter, logicalId);
        this.parameters[logicalId] = cfnParameter;
    }
    createRule(ruleName) {
        const self = this;
        const cfnParser = new cfn_parse.CfnParser({
            finder: {
                findRefTarget(refTarget) {
                    // only parameters can be referenced in Rules
                    return self.parameters[refTarget];
                },
                findResource() { throw new Error('Using GetAtt expressions in Rule definitions is not allowed'); },
                findCondition(conditionName) {
                    return self.conditions[conditionName];
                },
                findMapping(mappingName) {
                    return self.mappings[mappingName];
                },
            },
            parameters: this.parametersToReplace,
            context: cfn_parse.CfnParsingContext.RULES,
        });
        const ruleProperties = cfnParser.parseValue(this.template.Rules[ruleName]);
        const rule = new core.CfnRule(this.rulesScope, ruleName, {
            ruleCondition: ruleProperties.RuleCondition,
            assertions: ruleProperties.Assertions,
        });
        this.rules[ruleName] = rule;
        this.overrideLogicalIdIfNeeded(rule, ruleName);
    }
    createHook(hookName) {
        var _a;
        const self = this;
        const cfnParser = new cfn_parse.CfnParser({
            finder: {
                findResource(lId) {
                    return self.resources[lId];
                },
                findRefTarget(elementName) {
                    var _a;
                    return (_a = self.resources[elementName]) !== null && _a !== void 0 ? _a : self.parameters[elementName];
                },
                findCondition(conditionName) {
                    return self.conditions[conditionName];
                },
                findMapping(mappingName) {
                    return self.mappings[mappingName];
                },
            },
            parameters: this.parametersToReplace,
        });
        const hookAttributes = this.template.Hooks[hookName];
        let hook;
        switch (hookAttributes.Type) {
            case 'AWS::CodeDeploy::BlueGreen':
                hook = core.CfnCodeDeployBlueGreenHook._fromCloudFormation(this.hooksScope, hookName, hookAttributes, {
                    parser: cfnParser,
                });
                break;
            default: {
                const hookProperties = (_a = cfnParser.parseValue(hookAttributes.Properties)) !== null && _a !== void 0 ? _a : {};
                hook = new core.CfnHook(this.hooksScope, hookName, {
                    type: hookAttributes.Type,
                    properties: hookProperties,
                });
            }
        }
        this.hooks[hookName] = hook;
        this.overrideLogicalIdIfNeeded(hook, hookName);
    }
    createOutput(logicalId, scope) {
        const self = this;
        const outputAttributes = new cfn_parse.CfnParser({
            finder: {
                findResource(lId) {
                    return self.resources[lId];
                },
                findRefTarget(elementName) {
                    var _a;
                    return (_a = self.resources[elementName]) !== null && _a !== void 0 ? _a : self.parameters[elementName];
                },
                findCondition(conditionName) {
                    return self.conditions[conditionName];
                },
                findMapping(mappingName) {
                    return self.mappings[mappingName];
                },
            },
            parameters: this.parametersToReplace,
        }).parseValue(this.template.Outputs[logicalId]);
        const cfnOutput = new core.CfnOutput(scope, logicalId, {
            value: outputAttributes.Value,
            description: outputAttributes.Description,
            exportName: outputAttributes.Export ? outputAttributes.Export.Name : undefined,
            condition: (() => {
                if (!outputAttributes.Condition) {
                    return undefined;
                }
                else if (this.conditions[outputAttributes.Condition]) {
                    return self.getCondition(outputAttributes.Condition);
                }
                throw new Error(`Output with name '${logicalId}' refers to a Condition with name ` +
                    `'${outputAttributes.Condition}' which was not found in this template`);
            })(),
        });
        this.overrideLogicalIdIfNeeded(cfnOutput, logicalId);
        this.outputs[logicalId] = cfnOutput;
    }
    getOrCreateCondition(conditionName) {
        if (conditionName in this.conditions) {
            return this.conditions[conditionName];
        }
        const self = this;
        const cfnParser = new cfn_parse.CfnParser({
            finder: {
                findResource() { throw new Error('Using GetAtt in Condition definitions is not allowed'); },
                findRefTarget(elementName) {
                    // only Parameters can be referenced in the 'Conditions' section
                    return self.parameters[elementName];
                },
                findCondition(cName) {
                    return cName in (self.template.Conditions || {})
                        ? self.getOrCreateCondition(cName)
                        : undefined;
                },
                findMapping(mappingName) {
                    return self.mappings[mappingName];
                },
            },
            context: cfn_parse.CfnParsingContext.CONDITIONS,
            parameters: this.parametersToReplace,
        });
        const cfnCondition = new core.CfnCondition(this.conditionsScope, conditionName, {
            expression: cfnParser.parseValue(this.template.Conditions[conditionName]),
        });
        this.overrideLogicalIdIfNeeded(cfnCondition, conditionName);
        this.conditions[conditionName] = cfnCondition;
        return cfnCondition;
    }
    getOrCreateResource(logicalId) {
        const ret = this.resources[logicalId];
        if (ret) {
            return ret;
        }
        const resourceAttributes = this.template.Resources[logicalId];
        // fail early for resource attributes we don't support yet
        const knownAttributes = [
            'Type', 'Properties', 'Condition', 'DependsOn', 'Metadata', 'Version',
            'CreationPolicy', 'UpdatePolicy', 'DeletionPolicy', 'UpdateReplacePolicy',
        ];
        for (const attribute of Object.keys(resourceAttributes)) {
            if (!knownAttributes.includes(attribute)) {
                throw new Error(`The '${attribute}' resource attribute is not supported by cloudformation-include yet. ` +
                    'Either remove it from the template, or use the CdkInclude class from the core package instead.');
            }
        }
        const self = this;
        const finder = {
            findCondition(conditionName) {
                return self.conditions[conditionName];
            },
            findMapping(mappingName) {
                return self.mappings[mappingName];
            },
            findResource(lId) {
                if (!(lId in (self.template.Resources || {}))) {
                    return undefined;
                }
                return self.getOrCreateResource(lId);
            },
            findRefTarget(elementName) {
                if (elementName in self.parameters) {
                    return self.parameters[elementName];
                }
                return this.findResource(elementName);
            },
        };
        const cfnParser = new cfn_parse.CfnParser({
            finder,
            parameters: this.parametersToReplace,
        });
        let l1Instance;
        if (this.nestedStacksToInclude[logicalId]) {
            l1Instance = this.createNestedStack(logicalId, cfnParser);
        }
        else {
            const l1ClassFqn = cfn_type_to_l1_mapping.lookup(resourceAttributes.Type);
            // The AWS::CloudFormation::CustomResource type corresponds to the CfnCustomResource class.
            // Unfortunately, it's quite useless; it only has a single property, ServiceToken.
            // For that reason, even the CustomResource class from @core doesn't use it!
            // So, special-case the handling of this one resource type
            if (l1ClassFqn && resourceAttributes.Type !== 'AWS::CloudFormation::CustomResource') {
                const options = {
                    parser: cfnParser,
                };
                const [moduleName, ...className] = l1ClassFqn.split('.');
                const module = require(moduleName); // eslint-disable-line @typescript-eslint/no-require-imports
                const jsClassFromModule = module[className.join('.')];
                l1Instance = jsClassFromModule._fromCloudFormation(this, logicalId, resourceAttributes, options);
            }
            else {
                l1Instance = new core.CfnResource(this, logicalId, {
                    type: resourceAttributes.Type,
                    properties: cfnParser.parseValue(resourceAttributes.Properties),
                });
                cfnParser.handleAttributes(l1Instance, resourceAttributes, logicalId);
            }
        }
        this.overrideLogicalIdIfNeeded(l1Instance, logicalId);
        this.resources[logicalId] = l1Instance;
        return l1Instance;
    }
    createNestedStack(nestedStackId, cfnParser) {
        const templateResources = this.template.Resources || {};
        const nestedStackAttributes = templateResources[nestedStackId] || {};
        if (nestedStackAttributes.Type !== 'AWS::CloudFormation::Stack') {
            throw new Error(`Nested Stack with logical ID '${nestedStackId}' is not an AWS::CloudFormation::Stack resource`);
        }
        if (nestedStackAttributes.CreationPolicy) {
            throw new Error('CreationPolicy is not supported by the AWS::CloudFormation::Stack resource');
        }
        if (nestedStackAttributes.UpdatePolicy) {
            throw new Error('UpdatePolicy is not supported by the AWS::CloudFormation::Stack resource');
        }
        const nestedStackProps = cfnParser.parseValue(nestedStackAttributes.Properties);
        const nestedStack = new core.NestedStack(this, nestedStackId, {
            parameters: this.parametersForNestedStack(nestedStackProps.Parameters, nestedStackId),
            notificationArns: nestedStackProps.NotificationArns,
            timeout: nestedStackProps.Timeout,
        });
        const template = new CfnInclude(nestedStack, nestedStackId, this.nestedStacksToInclude[nestedStackId]);
        this.nestedStacks[nestedStackId] = { stack: nestedStack, includedTemplate: template };
        // we know this is never undefined for nested stacks
        const nestedStackResource = nestedStack.nestedStackResource;
        cfnParser.handleAttributes(nestedStackResource, nestedStackAttributes, nestedStackId);
        return nestedStackResource;
    }
    parametersForNestedStack(parameters, nestedStackId) {
        var _a;
        if (parameters == null) {
            return undefined;
        }
        const parametersToReplace = (_a = this.nestedStacksToInclude[nestedStackId].parameters) !== null && _a !== void 0 ? _a : {};
        const ret = {};
        for (const paramName of Object.keys(parameters)) {
            if (!(paramName in parametersToReplace)) {
                ret[paramName] = parameters[paramName];
            }
        }
        return ret;
    }
    overrideLogicalIdIfNeeded(element, id) {
        if (this.preserveLogicalIds) {
            element.overrideLogicalId(id);
        }
    }
}
exports.CfnInclude = CfnInclude;
//# sourceMappingURL=data:application/json;base64,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