"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const path = require("path");
const assert_1 = require("@aws-cdk/assert");
require("@aws-cdk/assert/jest");
const core = require("@aws-cdk/core");
const inc = require("../lib");
describe('CDK Include', () => {
    let stack;
    beforeEach(() => {
        stack = new core.Stack();
    });
    test('throws a validation exception for a template with a missing required top-level resource property', () => {
        expect(() => {
            includeTestTemplate(stack, 'bucket-policy-without-bucket.json');
        }).toThrow(/missing required property: bucket/);
    });
    test('throws a validation exception for a template with a resource property expecting an array assigned the wrong type', () => {
        includeTestTemplate(stack, 'bucket-with-cors-rules-not-an-array.json');
        expect(() => {
            assert_1.SynthUtils.synthesize(stack);
        }).toThrow(/corsRules: "CorsRules!" should be a list/);
    });
    test('throws a validation exception for a template with a null array element of a complex type with required fields', () => {
        includeTestTemplate(stack, 'bucket-with-cors-rules-null-element.json');
        expect(() => {
            assert_1.SynthUtils.synthesize(stack);
        }).toThrow(/allowedMethods: required but missing/);
    });
    test('throws a validation exception for a template with a missing nested resource property', () => {
        includeTestTemplate(stack, 'bucket-with-invalid-cors-rule.json');
        expect(() => {
            assert_1.SynthUtils.synthesize(stack);
        }).toThrow(/allowedOrigins: required but missing/);
    });
    test("throws a validation exception for a template with a DependsOn that doesn't exist", () => {
        expect(() => {
            includeTestTemplate(stack, 'non-existent-depends-on.json');
        }).toThrow(/Resource 'Bucket2' depends on 'Bucket1' that doesn't exist/);
    });
    test("throws a validation exception for a template referencing a Condition in the Conditions section that doesn't exist", () => {
        expect(() => {
            includeTestTemplate(stack, 'non-existent-condition-in-conditions.json');
        }).toThrow(/Referenced Condition with name 'AlwaysFalse' was not found in the template/);
    });
    test('throws a validation exception for a template using Fn::GetAtt in the Conditions section', () => {
        expect(() => {
            includeTestTemplate(stack, 'getatt-in-conditions.json');
        }).toThrow(/Using GetAtt in Condition definitions is not allowed/);
    });
    test("throws a validation exception for a template referencing a Condition resource attribute that doesn't exist", () => {
        expect(() => {
            includeTestTemplate(stack, 'non-existent-condition.json');
        }).toThrow(/Resource 'Bucket' uses Condition 'AlwaysFalseCond' that doesn't exist/);
    });
    test("throws a validation exception for a template referencing a Condition in an If expression that doesn't exist", () => {
        expect(() => {
            includeTestTemplate(stack, 'non-existent-condition-in-if.json');
        }).toThrow(/Condition 'AlwaysFalse' used in an Fn::If expression does not exist in the template/);
    });
    test("throws an exception when encountering a CFN function it doesn't support", () => {
        expect(() => {
            includeTestTemplate(stack, 'only-codecommit-repo-using-cfn-functions.json');
        }).toThrow(/Unsupported CloudFormation function 'Fn::ValueOfAll'/);
    });
    test('throws a validation exception when encountering an unrecognized resource attribute', () => {
        expect(() => {
            includeTestTemplate(stack, 'non-existent-resource-attribute.json');
        }).toThrow(/The 'NonExistentResourceAttribute' resource attribute is not supported by cloudformation-include yet/);
    });
    test("throws a validation exception when encountering a Ref-erence to a template element that doesn't exist", () => {
        expect(() => {
            includeTestTemplate(stack, 'ref-ing-a-non-existent-element.json');
        }).toThrow(/Element used in Ref expression with logical ID: 'DoesNotExist' not found/);
    });
    test("throws a validation exception when encountering a GetAtt reference to a resource that doesn't exist", () => {
        expect(() => {
            includeTestTemplate(stack, 'getting-attribute-of-a-non-existent-resource.json');
        }).toThrow(/Resource used in GetAtt expression with logical ID: 'DoesNotExist' not found/);
    });
    test("throws a validation exception when an Output references a Condition that doesn't exist", () => {
        expect(() => {
            includeTestTemplate(stack, 'output-referencing-nonexistant-condition.json');
        }).toThrow(/Output with name 'SomeOutput' refers to a Condition with name 'NonexistantCondition' which was not found in this template/);
    });
    test("throws a validation exception when a Resource property references a Mapping that doesn't exist", () => {
        expect(() => {
            includeTestTemplate(stack, 'non-existent-mapping.json');
        }).toThrow(/Mapping used in FindInMap expression with name 'NonExistentMapping' was not found in the template/);
    });
    test("throws a validation exception when a Rule references a Parameter that isn't in the template", () => {
        expect(() => {
            includeTestTemplate(stack, 'rule-referencing-a-non-existent-parameter.json');
        }).toThrow(/Rule references parameter 'Subnets' which was not found in the template/);
    });
    test("throws a validation exception when Fn::Sub in string form uses a key that isn't in the template", () => {
        expect(() => {
            includeTestTemplate(stack, 'fn-sub-key-not-in-template-string.json');
        }).toThrow(/Element referenced in Fn::Sub expression with logical ID: 'AFakeResource' was not found in the template/);
    });
    test('throws a validation exception when Fn::Sub has an empty ${} reference', () => {
        expect(() => {
            includeTestTemplate(stack, 'fn-sub-${}-only.json');
        }).toThrow(/Element referenced in Fn::Sub expression with logical ID: '' was not found in the template/);
    });
    test("throws an exception for a template with a non-number string passed to a property with type 'number'", () => {
        includeTestTemplate(stack, 'alphabetical-string-passed-to-number.json');
        expect(() => {
            assert_1.SynthUtils.synthesize(stack);
        }).toThrow(/"abc" should be a number/);
    });
    test('throws an exception for a template with a short-form Fn::GetAtt whose string argument does not contain a dot', () => {
        expect(() => {
            includeTestTemplate(stack, 'short-form-get-att-no-dot.yaml');
        }).toThrow(/Short-form Fn::GetAtt must contain a '.' in its string argument, got: 'Bucket1Arn'/);
    });
});
function includeTestTemplate(scope, testTemplate) {
    return new inc.CfnInclude(scope, 'MyScope', {
        templateFile: _testTemplateFilePath(testTemplate),
    });
}
function _testTemplateFilePath(testTemplate) {
    return path.join(__dirname, 'test-templates', 'invalid', testTemplate);
}
//# sourceMappingURL=data:application/json;base64,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