"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const path = require("path");
const assert_1 = require("@aws-cdk/assert");
require("@aws-cdk/assert/jest");
const core = require("@aws-cdk/core");
const inc = require("../lib");
const futils = require("../lib/file-utils");
/* eslint-disable quote-props */
/* eslint-disable quotes */
describe('CDK Include for nested stacks', () => {
    let stack;
    beforeEach(() => {
        stack = new core.Stack();
    });
    test('can ingest a template with one child', () => {
        const parentTemplate = new inc.CfnInclude(stack, 'ParentStack', {
            templateFile: testTemplateFilePath('parent-one-child.json'),
            loadNestedStacks: {
                'ChildStack': {
                    templateFile: testTemplateFilePath('grandchild-import-stack.json'),
                },
            },
        });
        const childStack = parentTemplate.getNestedStack('ChildStack');
        expect(childStack.stack).toMatchTemplate(loadTestFileToJsObject('grandchild-import-stack.json'));
    });
    test('can ingest a template with two children', () => {
        const parentTemplate = new inc.CfnInclude(stack, 'ParentStack', {
            templateFile: testTemplateFilePath('parent-two-children.json'),
            loadNestedStacks: {
                'ChildStack': {
                    templateFile: testTemplateFilePath('grandchild-import-stack.json'),
                },
                'AnotherChildStack': {
                    templateFile: testTemplateFilePath('grandchild-import-stack.json'),
                },
            },
        });
        const childStack = parentTemplate.getNestedStack('ChildStack');
        const anotherChildStack = parentTemplate.getNestedStack('AnotherChildStack');
        expect(childStack.stack).toMatchTemplate(loadTestFileToJsObject('grandchild-import-stack.json'));
        expect(anotherChildStack.stack).toMatchTemplate(loadTestFileToJsObject('grandchild-import-stack.json'));
    });
    test('can ingest a template with one child and one grandchild', () => {
        const parentTemplate = new inc.CfnInclude(stack, 'ParentStack', {
            templateFile: testTemplateFilePath('parent-two-children.json'),
            loadNestedStacks: {
                'ChildStack': {
                    templateFile: testTemplateFilePath('child-import-stack.json'),
                    loadNestedStacks: {
                        'GrandChildStack': {
                            templateFile: testTemplateFilePath('grandchild-import-stack.json'),
                        },
                    },
                },
            },
        });
        const childStack = parentTemplate.getNestedStack('ChildStack');
        const grandChildStack = childStack.includedTemplate.getNestedStack('GrandChildStack');
        expect(childStack.stack).toMatchTemplate(loadTestFileToJsObject('child-import-stack.expected.json'));
        expect(grandChildStack.stack).toMatchTemplate(loadTestFileToJsObject('grandchild-import-stack.json'));
    });
    test('throws an error when provided a nested stack that is not present in the template', () => {
        expect(() => {
            new inc.CfnInclude(stack, 'ParentStack', {
                templateFile: testTemplateFilePath('parent-two-children.json'),
                loadNestedStacks: {
                    'FakeStack': {
                        templateFile: testTemplateFilePath('child-import-stack.json'),
                    },
                },
            });
        }).toThrow(/Nested Stack with logical ID 'FakeStack' was not found in the template/);
    });
    test('throws an exception when NestedStacks contains an ID that is not a CloudFormation::Stack in the template', () => {
        expect(() => {
            new inc.CfnInclude(stack, 'ParentStack', {
                templateFile: testTemplateFilePath('child-import-stack.json'),
                loadNestedStacks: {
                    'BucketImport': {
                        templateFile: testTemplateFilePath('grandchild-import-stack.json'),
                    },
                },
            });
        }).toThrow(/Nested Stack with logical ID 'BucketImport' is not an AWS::CloudFormation::Stack resource/);
    });
    test('throws an exception when the nestedStack resource uses the CreationPolicy attribute', () => {
        expect(() => {
            new inc.CfnInclude(stack, 'ParentStack', {
                templateFile: testTemplateFilePath('parent-creation-policy.json'),
                loadNestedStacks: {
                    'ChildStack': {
                        templateFile: testTemplateFilePath('grandchild-import-stack.json'),
                    },
                },
            });
        }).toThrow(/CreationPolicy is not supported by the AWS::CloudFormation::Stack resource/);
    });
    test('throws an exception when the nested stack resource uses the UpdatePolicy attribute', () => {
        expect(() => {
            new inc.CfnInclude(stack, 'ParentStack', {
                templateFile: testTemplateFilePath('parent-update-policy.json'),
                loadNestedStacks: {
                    'ChildStack': {
                        templateFile: testTemplateFilePath('grandchild-import-stack.json'),
                    },
                },
            });
        }).toThrow(/UpdatePolicy is not supported by the AWS::CloudFormation::Stack resource/);
    });
    test('throws an exception when a nested stack refers to a Condition that does not exist in the template', () => {
        expect(() => {
            new inc.CfnInclude(stack, 'ParentStack', {
                templateFile: testTemplateFilePath('parent-invalid-condition.json'),
                loadNestedStacks: {
                    'ChildStack': {
                        templateFile: testTemplateFilePath('grandchild-import-stack.json'),
                    },
                },
            });
        }).toThrow(/Resource 'ChildStack' uses Condition 'FakeCondition' that doesn't exist/);
    });
    test('throws an exception when a nested stacks depends on a resource that does not exist in the template', () => {
        expect(() => {
            new inc.CfnInclude(stack, 'ParentStack', {
                templateFile: testTemplateFilePath('parent-bad-depends-on.json'),
                loadNestedStacks: {
                    'ChildStack': {
                        templateFile: testTemplateFilePath('child-import-stack.json'),
                    },
                },
            });
        }).toThrow(/Resource 'ChildStack' depends on 'AFakeResource' that doesn't exist/);
    });
    test('throws an exception when an ID was passed in loadNestedStacks that is a resource type not in the CloudFormation schema', () => {
        expect(() => {
            new inc.CfnInclude(stack, 'Template', {
                templateFile: testTemplateFilePath('custom-resource.json'),
                loadNestedStacks: {
                    'CustomResource': {
                        templateFile: testTemplateFilePath('whatever.json'),
                    },
                },
            });
        }).toThrow(/Nested Stack with logical ID 'CustomResource' is not an AWS::CloudFormation::Stack resource/);
    });
    test('can modify resources in nested stacks', () => {
        const parent = new inc.CfnInclude(stack, 'ParentStack', {
            templateFile: testTemplateFilePath('child-import-stack.json'),
            loadNestedStacks: {
                'GrandChildStack': {
                    templateFile: testTemplateFilePath('grandchild-import-stack.json'),
                },
            },
        });
        const childTemplate = parent.getNestedStack('GrandChildStack').includedTemplate;
        const bucket = childTemplate.getResource('BucketImport');
        bucket.bucketName = 'modified-bucket-name';
        expect(childTemplate.stack).toHaveResource('AWS::S3::Bucket', { BucketName: 'modified-bucket-name' });
    });
    test('can use a condition', () => {
        const parent = new inc.CfnInclude(stack, 'ParentStack', {
            templateFile: testTemplateFilePath('parent-valid-condition.json'),
            loadNestedStacks: {
                'ChildStack': {
                    templateFile: testTemplateFilePath('grandchild-import-stack.json'),
                },
            },
        });
        const alwaysFalseCondition = parent.getCondition('AlwaysFalseCond');
        expect(parent.getResource('ChildStack').cfnOptions.condition).toBe(alwaysFalseCondition);
    });
    test('asset parameters generated in parent and child are identical', () => {
        new inc.CfnInclude(stack, 'ParentStack', {
            templateFile: testTemplateFilePath('parent-one-child.json'),
            loadNestedStacks: {
                'ChildStack': {
                    templateFile: testTemplateFilePath('grandchild-import-stack.json'),
                },
            },
        });
        const assetParam = 'AssetParameters5dc7d4a99cfe2979687dc74f2db9fd75f253b5505a1912b5ceecf70c9aefba50S3BucketEAA24F0C';
        const assetParamKey = 'AssetParameters5dc7d4a99cfe2979687dc74f2db9fd75f253b5505a1912b5ceecf70c9aefba50S3VersionKey1194CAB2';
        expect(stack).toMatchTemplate({
            "Parameters": {
                [assetParam]: {
                    "Type": "String",
                    "Description": "S3 bucket for asset \"5dc7d4a99cfe2979687dc74f2db9fd75f253b5505a1912b5ceecf70c9aefba50\"",
                },
                [assetParamKey]: {
                    "Type": "String",
                    "Description": "S3 key for asset version \"5dc7d4a99cfe2979687dc74f2db9fd75f253b5505a1912b5ceecf70c9aefba50\"",
                },
                "AssetParameters5dc7d4a99cfe2979687dc74f2db9fd75f253b5505a1912b5ceecf70c9aefba50ArtifactHash9C417847": {
                    "Type": "String",
                    "Description": "Artifact hash for asset \"5dc7d4a99cfe2979687dc74f2db9fd75f253b5505a1912b5ceecf70c9aefba50\"",
                },
            },
            "Resources": {
                "ChildStack": {
                    "Type": "AWS::CloudFormation::Stack",
                    "Properties": {
                        "TemplateURL": {
                            "Fn::Join": ["", [
                                    "https://s3.",
                                    { "Ref": "AWS::Region" },
                                    ".",
                                    { "Ref": "AWS::URLSuffix" },
                                    "/",
                                    { "Ref": assetParam },
                                    "/",
                                    {
                                        "Fn::Select": [
                                            0,
                                            {
                                                "Fn::Split": [
                                                    "||",
                                                    { "Ref": assetParamKey },
                                                ],
                                            },
                                        ],
                                    },
                                    {
                                        "Fn::Select": [
                                            1,
                                            {
                                                "Fn::Split": [
                                                    "||",
                                                    { "Ref": assetParamKey },
                                                ],
                                            },
                                        ],
                                    },
                                ]],
                        },
                        "Parameters": {
                            "MyBucketParameter": "some-magic-bucket-name",
                        },
                    },
                },
            },
        });
    });
    test('templates with nested stacks that were not provided in the loadNestedStacks property are left unmodified', () => {
        new inc.CfnInclude(stack, 'ParentStack', {
            templateFile: testTemplateFilePath('parent-two-children.json'),
        });
        expect(stack).toMatchTemplate(loadTestFileToJsObject('parent-two-children.json'));
    });
    test('getNestedStack() throws an exception when getting a resource that does not exist in the template', () => {
        const parentTemplate = new inc.CfnInclude(stack, 'ParentStack', {
            templateFile: testTemplateFilePath('parent-two-children.json'),
            loadNestedStacks: {
                'ChildStack': {
                    templateFile: testTemplateFilePath('child-import-stack.json'),
                },
            },
        });
        expect(() => {
            parentTemplate.getNestedStack('FakeStack');
        }).toThrow(/Nested Stack with logical ID 'FakeStack' was not found/);
    });
    test('getNestedStack() throws an exception when getting a resource that exists in the template, but is not a Stack', () => {
        const parentTemplate = new inc.CfnInclude(stack, 'ParentStack', {
            templateFile: testTemplateFilePath('parent-two-children.json'),
            loadNestedStacks: {
                'ChildStack': {
                    templateFile: testTemplateFilePath('child-import-stack.json'),
                },
            },
        });
        const childTemplate = parentTemplate.getNestedStack('ChildStack').includedTemplate;
        expect(() => {
            childTemplate.getNestedStack('BucketImport');
        }).toThrow(/Resource with logical ID 'BucketImport' is not a CloudFormation Stack/);
    });
    test('getNestedStack() throws an exception when getting a nested stack that exists in the template, but was not specified in the props', () => {
        const parentTemplate = new inc.CfnInclude(stack, 'ParentStack', {
            templateFile: testTemplateFilePath('parent-two-children.json'),
            loadNestedStacks: {
                'ChildStack': {
                    templateFile: testTemplateFilePath('child-import-stack.json'),
                },
            },
        });
        expect(() => {
            parentTemplate.getNestedStack('AnotherChildStack');
        }).toThrow(/Nested Stack 'AnotherChildStack' was not included in the parent template/);
    });
    test('correctly handles renaming of references across nested stacks', () => {
        const parentTemplate = new inc.CfnInclude(stack, 'ParentStack', {
            templateFile: testTemplateFilePath('cross-stack-refs.json'),
            loadNestedStacks: {
                'ChildStack': {
                    templateFile: testTemplateFilePath('child-import-stack.json'),
                },
            },
        });
        const cfnBucket = parentTemplate.getResource('Bucket');
        cfnBucket.overrideLogicalId('DifferentBucket');
        const parameter = parentTemplate.getParameter('Param');
        parameter.overrideLogicalId('DifferentParameter');
        expect(stack).toHaveResourceLike('AWS::CloudFormation::Stack', {
            "Parameters": {
                "Param1": {
                    "Ref": "DifferentParameter",
                },
                "Param2": {
                    "Fn::GetAtt": ["DifferentBucket", "Arn"],
                },
            },
        });
    });
    test('returns the CfnStack object from getResource() for a nested stack that was not in the loadNestedStacks property', () => {
        const cfnTemplate = new inc.CfnInclude(stack, 'ParentStack', {
            templateFile: testTemplateFilePath('parent-two-children.json'),
        });
        const childStack1 = cfnTemplate.getResource('ChildStack');
        expect(childStack1).toBeInstanceOf(core.CfnStack);
    });
    test('returns the CfnStack object from getResource() for a nested stack that was in the loadNestedStacks property', () => {
        const cfnTemplate = new inc.CfnInclude(stack, 'ParentStack', {
            templateFile: testTemplateFilePath('parent-one-child.json'),
            loadNestedStacks: {
                'ChildStack': {
                    templateFile: testTemplateFilePath('child-import-stack.json'),
                },
            },
        });
        const childStack1 = cfnTemplate.getResource('ChildStack');
        expect(childStack1).toBeInstanceOf(core.CfnStack);
    });
    test("handles Metadata, DeletionPolicy, and UpdateReplacePolicy attributes of the nested stack's resource", () => {
        const cfnTemplate = new inc.CfnInclude(stack, 'ParentStack', {
            templateFile: testTemplateFilePath('parent-with-attributes.json'),
            loadNestedStacks: {
                'ChildStack': {
                    templateFile: testTemplateFilePath('child-import-stack.json'),
                },
                'AnotherChildStack': {
                    templateFile: testTemplateFilePath('child-import-stack.json'),
                },
            },
        });
        expect(stack).toHaveResourceLike('AWS::CloudFormation::Stack', {
            "Metadata": {
                "Property1": "Value1",
            },
            "DeletionPolicy": "Retain",
            "DependsOn": [
                "AnotherChildStack",
            ],
            "UpdateReplacePolicy": "Retain",
        }, assert_1.ResourcePart.CompleteDefinition);
        cfnTemplate.getNestedStack('AnotherChildStack');
    });
    test('correctly parses NotificationsARNs, Timeout', () => {
        new inc.CfnInclude(stack, 'ParentStack', {
            templateFile: testTemplateFilePath('parent-with-attributes.json'),
        });
        expect(stack).toHaveResourceLike('AWS::CloudFormation::Stack', {
            "TemplateURL": "https://cfn-templates-set.s3.amazonaws.com/child-import-stack.json",
            "NotificationARNs": ["arn1"],
            "TimeoutInMinutes": 5,
        });
    });
    test('can lazily include a single child nested stack', () => {
        const parentTemplate = new inc.CfnInclude(stack, 'ParentStack', {
            templateFile: testTemplateFilePath('parent-one-child.json'),
        });
        const includedChild = parentTemplate.loadNestedStack('ChildStack', {
            templateFile: testTemplateFilePath('child-no-bucket.json'),
        });
        expect(includedChild.stack).toMatchTemplate(loadTestFileToJsObject('child-no-bucket.json'));
        expect(includedChild.includedTemplate.getResource('GrandChildStack')).toBeDefined();
    });
    describe('for a parent stack with children and grandchildren', () => {
        let assetStack;
        let parentTemplate;
        let child;
        let grandChild;
        let parentBucketParam;
        let parentKeyParam;
        let grandChildBucketParam;
        let grandChildKeyParam;
        let childBucketParam;
        let childKeyParam;
        beforeAll(() => {
            assetStack = new core.Stack();
            parentTemplate = new inc.CfnInclude(assetStack, 'ParentStack', {
                templateFile: testTemplateFilePath('parent-one-child.json'),
                loadNestedStacks: {
                    'ChildStack': {
                        templateFile: testTemplateFilePath('child-no-bucket.json'),
                        loadNestedStacks: {
                            'GrandChildStack': {
                                templateFile: testTemplateFilePath('grandchild-import-stack.json'),
                            },
                        },
                    },
                },
            });
            child = parentTemplate.getNestedStack('ChildStack');
            grandChild = child.includedTemplate.getNestedStack('GrandChildStack');
            parentBucketParam = 'AssetParameters5dc7d4a99cfe2979687dc74f2db9fd75f253b5505a1912b5ceecf70c9aefba50S3BucketEAA24F0C';
            parentKeyParam = 'AssetParameters5dc7d4a99cfe2979687dc74f2db9fd75f253b5505a1912b5ceecf70c9aefba50S3VersionKey1194CAB2';
            grandChildBucketParam = 'referencetoAssetParameters5dc7d4a99cfe2979687dc74f2db9fd75f253b5505a1912b5ceecf70c9aefba50S3BucketEAA24F0CRef';
            grandChildKeyParam = 'referencetoAssetParameters5dc7d4a99cfe2979687dc74f2db9fd75f253b5505a1912b5ceecf70c9aefba50S3VersionKey1194CAB2Ref';
            childBucketParam = 'AssetParameters891fd3ec75dc881b0fe40dc9fd1b433672637585c015265a5f0dab6bf79818d5S3Bucket23278F13';
            childKeyParam = 'AssetParameters891fd3ec75dc881b0fe40dc9fd1b433672637585c015265a5f0dab6bf79818d5S3VersionKey7316205A';
        });
        test('correctly creates parameters in the parent stack, and passes them to the child stack', () => {
            expect(assetStack).toMatchTemplate({
                "Parameters": {
                    [parentBucketParam]: {
                        "Type": "String",
                        "Description": "S3 bucket for asset \"5dc7d4a99cfe2979687dc74f2db9fd75f253b5505a1912b5ceecf70c9aefba50\"",
                    },
                    [parentKeyParam]: {
                        "Type": "String",
                        "Description": "S3 key for asset version \"5dc7d4a99cfe2979687dc74f2db9fd75f253b5505a1912b5ceecf70c9aefba50\"",
                    },
                    "AssetParameters5dc7d4a99cfe2979687dc74f2db9fd75f253b5505a1912b5ceecf70c9aefba50ArtifactHash9C417847": {
                        "Type": "String",
                        "Description": "Artifact hash for asset \"5dc7d4a99cfe2979687dc74f2db9fd75f253b5505a1912b5ceecf70c9aefba50\"",
                    },
                    [childBucketParam]: {
                        "Type": "String",
                        "Description": "S3 bucket for asset \"891fd3ec75dc881b0fe40dc9fd1b433672637585c015265a5f0dab6bf79818d5\"",
                    },
                    [childKeyParam]: {
                        "Type": "String",
                        "Description": "S3 key for asset version \"891fd3ec75dc881b0fe40dc9fd1b433672637585c015265a5f0dab6bf79818d5\"",
                    },
                    "AssetParameters891fd3ec75dc881b0fe40dc9fd1b433672637585c015265a5f0dab6bf79818d5ArtifactHashA1DE5198": {
                        "Type": "String",
                        "Description": "Artifact hash for asset \"891fd3ec75dc881b0fe40dc9fd1b433672637585c015265a5f0dab6bf79818d5\"",
                    },
                },
                "Resources": {
                    "ChildStack": {
                        "Type": "AWS::CloudFormation::Stack",
                        "Properties": {
                            "TemplateURL": {
                                "Fn::Join": ["", [
                                        "https://s3.",
                                        { "Ref": "AWS::Region" },
                                        ".",
                                        { "Ref": "AWS::URLSuffix" },
                                        "/",
                                        { "Ref": childBucketParam },
                                        "/",
                                        {
                                            "Fn::Select": [
                                                0,
                                                {
                                                    "Fn::Split": [
                                                        "||",
                                                        { "Ref": childKeyParam },
                                                    ],
                                                },
                                            ],
                                        },
                                        {
                                            "Fn::Select": [
                                                1,
                                                {
                                                    "Fn::Split": [
                                                        "||",
                                                        { "Ref": childKeyParam },
                                                    ],
                                                },
                                            ],
                                        },
                                    ]],
                            },
                            "Parameters": {
                                "MyBucketParameter": "some-magic-bucket-name",
                                [grandChildBucketParam]: {
                                    "Ref": parentBucketParam,
                                },
                                [grandChildKeyParam]: {
                                    "Ref": parentKeyParam,
                                },
                            },
                        },
                    },
                },
            });
        });
        test('correctly creates parameters in the child stack, and passes them to the grandchild stack', () => {
            expect(child.stack).toMatchTemplate({
                "Parameters": {
                    "MyBucketParameter": {
                        "Type": "String",
                        "Default": "default-bucket-param-name",
                    },
                    [grandChildBucketParam]: {
                        "Type": "String",
                    },
                    [grandChildKeyParam]: {
                        "Type": "String",
                    },
                },
                "Resources": {
                    "GrandChildStack": {
                        "Type": "AWS::CloudFormation::Stack",
                        "Properties": {
                            "TemplateURL": {
                                "Fn::Join": ["", [
                                        "https://s3.",
                                        { "Ref": "AWS::Region" },
                                        ".",
                                        { "Ref": "AWS::URLSuffix" },
                                        "/",
                                        { "Ref": grandChildBucketParam },
                                        "/",
                                        {
                                            "Fn::Select": [
                                                0,
                                                {
                                                    "Fn::Split": [
                                                        "||",
                                                        { "Ref": grandChildKeyParam },
                                                    ],
                                                },
                                            ],
                                        },
                                        {
                                            "Fn::Select": [
                                                1,
                                                {
                                                    "Fn::Split": [
                                                        "||",
                                                        { "Ref": grandChildKeyParam },
                                                    ],
                                                },
                                            ],
                                        },
                                    ]],
                            },
                            "Parameters": {
                                "MyBucketParameter": "some-other-bucket-name",
                            },
                        },
                    },
                },
            });
        });
        test('leaves grandchild stack unmodified', () => {
            expect(grandChild.stack).toMatchTemplate(loadTestFileToJsObject('grandchild-import-stack.json'));
        });
    });
    describe('for a parameter passed to the included child stack', () => {
        let parentStack;
        let childStack;
        beforeAll(() => {
            parentStack = new core.Stack();
            const parentTemplate = new inc.CfnInclude(parentStack, 'ParentStack', {
                templateFile: testTemplateFilePath('parent-two-parameters.json'),
                loadNestedStacks: {
                    'ChildStack': {
                        templateFile: testTemplateFilePath('child-two-parameters.json'),
                        parameters: {
                            'FirstParameter': 'test-value',
                        },
                    },
                },
            });
            childStack = parentTemplate.getNestedStack('ChildStack').stack;
        });
        test('correctly removes the parameter from the child stack', () => {
            expect(childStack).toMatchTemplate({
                "Parameters": {
                    "SecondParameter": {
                        "Type": "String",
                    },
                },
                "Resources": {
                    "BucketImport": {
                        "Type": "AWS::S3::Bucket",
                        "Properties": {
                            "BucketName": "test-value",
                            "AccessControl": {
                                "Ref": "SecondParameter",
                            },
                        },
                    },
                    "GrandChildStack": {
                        "Type": "AWS::CloudFormation::Stack",
                        "Properties": {
                            "TemplateURL": "https://cfn-templates-set.s3.amazonaws.com/grandchild-import-stack.json",
                            "Parameters": {
                                "FirstParameter": "test-value",
                            },
                        },
                    },
                },
            });
        });
        test('correctly removes the parameter from the parent stack', () => {
            expect(parentStack).toHaveResourceLike('AWS::CloudFormation::Stack', {
                "Parameters": {
                    "FirstParameter": assert_1.ABSENT,
                    "SecondParameter": "second-value",
                },
            });
        });
    });
});
function loadTestFileToJsObject(testTemplate) {
    return futils.readJsonSync(testTemplateFilePath(testTemplate));
}
function testTemplateFilePath(testTemplate) {
    return path.join(__dirname, 'test-templates/nested', testTemplate);
}
//# sourceMappingURL=data:application/json;base64,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