"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const path = require("path");
const assert_1 = require("@aws-cdk/assert");
require("@aws-cdk/assert/jest");
const iam = require("@aws-cdk/aws-iam");
const s3 = require("@aws-cdk/aws-s3");
const core = require("@aws-cdk/core");
const inc = require("../lib");
const futils = require("../lib/file-utils");
/* eslint-disable quote-props */
/* eslint-disable quotes */
describe('CDK Include', () => {
    let stack;
    beforeEach(() => {
        stack = new core.Stack();
    });
    test('can ingest a template with only an empty S3 Bucket, and output it unchanged', () => {
        includeTestTemplate(stack, 'only-empty-bucket.json');
        expect(stack).toMatchTemplate(loadTestFileToJsObject('only-empty-bucket.json'));
    });
    test('throws an exception if asked for resource with a logical ID not present in the template', () => {
        const cfnTemplate = includeTestTemplate(stack, 'only-empty-bucket.json');
        expect(() => {
            cfnTemplate.getResource('LogicalIdThatDoesNotExist');
        }).toThrow(/Resource with logical ID 'LogicalIdThatDoesNotExist' was not found in the template/);
    });
    test('can ingest a template with only an empty S3 Bucket, and change its property', () => {
        const cfnTemplate = includeTestTemplate(stack, 'only-empty-bucket.json');
        const cfnBucket = cfnTemplate.getResource('Bucket');
        cfnBucket.bucketName = 'my-bucket-name';
        expect(stack).toMatchTemplate({
            "Resources": {
                "Bucket": {
                    "Type": "AWS::S3::Bucket",
                    "Properties": {
                        "BucketName": "my-bucket-name",
                    },
                },
            },
        });
    });
    test('can ingest a template with only an S3 Bucket with complex properties, and output it unchanged', () => {
        const cfnTemplate = includeTestTemplate(stack, 'only-bucket-complex-props.json');
        const cfnBucket = cfnTemplate.getResource('Bucket');
        expect(cfnBucket.corsConfiguration.corsRules).toHaveLength(1);
        expect(stack).toMatchTemplate(loadTestFileToJsObject('only-bucket-complex-props.json'));
    });
    test('allows referring to a bucket defined in the template in your CDK code', () => {
        const cfnTemplate = includeTestTemplate(stack, 'only-empty-bucket.json');
        const cfnBucket = cfnTemplate.getResource('Bucket');
        const role = new iam.Role(stack, 'Role', {
            assumedBy: new iam.AnyPrincipal(),
        });
        role.addToPolicy(new iam.PolicyStatement({
            actions: ['s3:*'],
            resources: [cfnBucket.attrArn],
        }));
        expect(stack).toHaveResourceLike('AWS::IAM::Policy', {
            "PolicyDocument": {
                "Statement": [
                    {
                        "Action": "s3:*",
                        "Resource": {
                            "Fn::GetAtt": [
                                "Bucket",
                                "Arn",
                            ],
                        },
                    },
                ],
            },
        });
    });
    test('can ingest a template with a Bucket Ref-erencing a KMS Key, and output it unchanged', () => {
        includeTestTemplate(stack, 'bucket-with-encryption-key.json');
        expect(stack).toMatchTemplate(loadTestFileToJsObject('bucket-with-encryption-key.json'));
    });
    test('accepts strings for properties with type number', () => {
        includeTestTemplate(stack, 'string-for-number.json');
        expect(stack).toMatchTemplate(loadTestFileToJsObject('string-for-number.json'));
    });
    test('accepts numbers for properties with type string', () => {
        includeTestTemplate(stack, 'number-for-string.json');
        expect(stack).toMatchTemplate(loadTestFileToJsObject('number-for-string.json'));
    });
    test('correctly changes the logical IDs, including references, if imported with preserveLogicalIds=false', () => {
        const cfnTemplate = includeTestTemplate(stack, 'bucket-with-encryption-key.json', {
            preserveLogicalIds: false,
        });
        // even though the logical IDs in the resulting template are different than in the input template,
        // the L1s can still be retrieved using their original logical IDs from the template file,
        // and any modifications to them will be reflected in the resulting template
        const cfnBucket = cfnTemplate.getResource('Bucket');
        cfnBucket.bucketName = 'my-bucket-name';
        expect(stack).toMatchTemplate({
            "Resources": {
                "MyScopeKey7673692F": {
                    "Type": "AWS::KMS::Key",
                    "Properties": {
                        "KeyPolicy": {
                            "Statement": [
                                {
                                    "Action": [
                                        "kms:*",
                                    ],
                                    "Effect": "Allow",
                                    "Principal": {
                                        "AWS": {
                                            "Fn::Join": ["", [
                                                    "arn:",
                                                    { "Ref": "AWS::Partition" },
                                                    ":iam::",
                                                    { "Ref": "AWS::AccountId" },
                                                    ":root",
                                                ]],
                                        },
                                    },
                                    "Resource": "*",
                                },
                            ],
                            "Version": "2012-10-17",
                        },
                    },
                    "DeletionPolicy": "Delete",
                    "UpdateReplacePolicy": "Delete",
                },
                "MyScopeBucket02C1313B": {
                    "Type": "AWS::S3::Bucket",
                    "Properties": {
                        "BucketName": "my-bucket-name",
                        "BucketEncryption": {
                            "ServerSideEncryptionConfiguration": [
                                {
                                    "ServerSideEncryptionByDefault": {
                                        "KMSMasterKeyID": {
                                            "Fn::GetAtt": [
                                                "MyScopeKey7673692F",
                                                "Arn",
                                            ],
                                        },
                                        "SSEAlgorithm": "aws:kms",
                                    },
                                },
                            ],
                        },
                    },
                    "Metadata": {
                        "Object1": "Location1",
                        "KeyRef": { "Ref": "MyScopeKey7673692F" },
                        "KeyArn": { "Fn::GetAtt": ["MyScopeKey7673692F", "Arn"] },
                    },
                    "DeletionPolicy": "Retain",
                    "UpdateReplacePolicy": "Retain",
                },
            },
        });
    });
    test('can ingest a template with an Fn::If expression for simple values, and output it unchanged', () => {
        includeTestTemplate(stack, 'if-simple-property.json');
        expect(stack).toMatchTemplate(loadTestFileToJsObject('if-simple-property.json'));
    });
    test('can ingest a template with an Fn::If expression for complex values, and output it unchanged', () => {
        includeTestTemplate(stack, 'if-complex-property.json');
        expect(stack).toMatchTemplate(loadTestFileToJsObject('if-complex-property.json'));
    });
    test('can ingest a UserData script, and output it unchanged', () => {
        includeTestTemplate(stack, 'user-data.json');
        expect(stack).toMatchTemplate(loadTestFileToJsObject('user-data.json'));
    });
    test('can correctly ingest a resource with a property of type: Map of Lists of primitive types', () => {
        const cfnTemplate = includeTestTemplate(stack, 'ssm-association.json');
        expect(stack).toMatchTemplate(loadTestFileToJsObject('ssm-association.json'));
        const association = cfnTemplate.getResource('Association');
        expect(Object.keys(association.parameters)).toHaveLength(2);
    });
    test('can ingest a template with intrinsic functions and conditions, and output it unchanged', () => {
        includeTestTemplate(stack, 'functions-and-conditions.json');
        expect(stack).toMatchTemplate(loadTestFileToJsObject('functions-and-conditions.json'));
    });
    test('can ingest a template with Fn::Sub in string form with escaped and unescaped references and output it unchanged', () => {
        includeTestTemplate(stack, 'fn-sub-string.json');
        expect(stack).toMatchTemplate(loadTestFileToJsObject('fn-sub-string.json'));
    });
    test('can parse the string argument Fn::Sub with escaped references that contain whitespace', () => {
        includeTestTemplate(stack, 'fn-sub-escaping.json');
        expect(stack).toMatchTemplate(loadTestFileToJsObject('fn-sub-escaping.json'));
    });
    test('can ingest a template with Fn::Sub in map form and output it unchanged', () => {
        includeTestTemplate(stack, 'fn-sub-map-dotted-attributes.json');
        expect(stack).toMatchTemplate(loadTestFileToJsObject('fn-sub-map-dotted-attributes.json'));
    });
    test('can ingest a template with Fn::Sub shadowing a logical ID from the template and output it unchanged', () => {
        includeTestTemplate(stack, 'fn-sub-shadow.json');
        expect(stack).toMatchTemplate(loadTestFileToJsObject('fn-sub-shadow.json'));
    });
    test('can ingest a template with Fn::Sub attribute expression shadowing a logical ID from the template, and output it unchanged', () => {
        includeTestTemplate(stack, 'fn-sub-shadow-attribute.json');
        expect(stack).toMatchTemplate(loadTestFileToJsObject('fn-sub-shadow-attribute.json'));
    });
    test('can modify resources used in Fn::Sub in map form references and see the changes in the template', () => {
        const cfnTemplate = includeTestTemplate(stack, 'fn-sub-shadow.json');
        cfnTemplate.getResource('AnotherBucket').overrideLogicalId('NewBucket');
        expect(stack).toHaveResourceLike('AWS::S3::Bucket', {
            "BucketName": {
                "Fn::Sub": [
                    "${AnotherBucket}",
                    {
                        "AnotherBucket": { "Ref": "NewBucket" },
                    },
                ],
            },
        });
    });
    test('can modify resources used in Fn::Sub in string form and see the changes in the template', () => {
        const cfnTemplate = includeTestTemplate(stack, 'fn-sub-override.json');
        cfnTemplate.getResource('Bucket').overrideLogicalId('NewBucket');
        expect(stack).toHaveResourceLike('AWS::S3::Bucket', {
            "BucketName": {
                "Fn::Sub": "${NewBucket}-${!Bucket}-${NewBucket.DomainName}",
            },
        });
    });
    test('can ingest a template with Fn::Sub with brace edge cases and output it unchanged', () => {
        includeTestTemplate(stack, 'fn-sub-brace-edges.json');
        expect(stack).toMatchTemplate(loadTestFileToJsObject('fn-sub-brace-edges.json'));
    });
    test('can ingest a template with a Ref expression for an array value, and output it unchanged', () => {
        includeTestTemplate(stack, 'ref-array-property.json');
        expect(stack).toMatchTemplate(loadTestFileToJsObject('ref-array-property.json'));
    });
    test('renders non-Resources sections unchanged', () => {
        includeTestTemplate(stack, 'only-empty-bucket-with-parameters.json');
        expect(stack).toMatchTemplate(loadTestFileToJsObject('only-empty-bucket-with-parameters.json'));
    });
    test('resolves DependsOn with a single String value to the actual L1 class instance', () => {
        const cfnTemplate = includeTestTemplate(stack, 'resource-attribute-depends-on.json');
        const cfnBucket2 = cfnTemplate.getResource('Bucket2');
        expect(cfnBucket2.node.dependencies).toHaveLength(1);
        // we always render dependsOn as an array, even if it's a single string
        expect(stack).toHaveResourceLike('AWS::S3::Bucket', {
            "Properties": {
                "BucketName": "bucket2",
            },
            "DependsOn": [
                "Bucket1",
            ],
        }, assert_1.ResourcePart.CompleteDefinition);
    });
    test('resolves DependsOn with an array of String values to the actual L1 class instances', () => {
        const cfnTemplate = includeTestTemplate(stack, 'resource-attribute-depends-on-array.json');
        const cfnBucket2 = cfnTemplate.getResource('Bucket2');
        expect(cfnBucket2.node.dependencies).toHaveLength(2);
        expect(stack).toHaveResourceLike('AWS::S3::Bucket', {
            "Properties": {
                "BucketName": "bucket2",
            },
            "DependsOn": [
                "Bucket0",
                "Bucket1",
            ],
        }, assert_1.ResourcePart.CompleteDefinition);
    });
    test('correctly parses Conditions and the Condition resource attribute', () => {
        const cfnTemplate = includeTestTemplate(stack, 'resource-attribute-condition.json');
        const alwaysFalseCondition = cfnTemplate.getCondition('AlwaysFalseCond');
        const cfnBucket = cfnTemplate.getResource('Bucket');
        expect(cfnBucket.cfnOptions.condition).toBe(alwaysFalseCondition);
        expect(stack).toMatchTemplate(loadTestFileToJsObject('resource-attribute-condition.json'));
    });
    test('allows Conditions to reference Mappings', () => {
        includeTestTemplate(stack, 'condition-using-mapping.json');
        expect(stack).toMatchTemplate(loadTestFileToJsObject('condition-using-mapping.json'));
    });
    test('correctly change references to Conditions when renaming them', () => {
        const cfnTemplate = includeTestTemplate(stack, 'condition-same-name-as-resource.json');
        const alwaysFalse = cfnTemplate.getCondition('AlwaysFalse');
        alwaysFalse.overrideLogicalId('TotallyFalse');
        expect(stack).toMatchTemplate({
            "Parameters": {
                "Param": {
                    "Type": "String",
                },
            },
            "Conditions": {
                "AlwaysTrue": {
                    "Fn::Not": [{ "Condition": "TotallyFalse" }],
                },
                "TotallyFalse": {
                    "Fn::Equals": [{ "Ref": "Param" }, 2],
                },
            },
            "Resources": {
                "AlwaysTrue": {
                    "Type": "AWS::S3::Bucket",
                    "Properties": {
                        "BucketName": {
                            "Fn::If": ["TotallyFalse",
                                { "Ref": "Param" },
                                { "Ref": "AWS::NoValue" }],
                        },
                    },
                },
            },
        });
    });
    test('correctly parses templates with parameters', () => {
        const cfnTemplate = includeTestTemplate(stack, 'bucket-with-parameters.json');
        const param = cfnTemplate.getParameter('BucketName');
        new s3.CfnBucket(stack, 'NewBucket', {
            bucketName: param.valueAsString,
        });
        const originalTemplate = loadTestFileToJsObject('bucket-with-parameters.json');
        expect(stack).toMatchTemplate({
            "Resources": {
                ...originalTemplate.Resources,
                "NewBucket": {
                    "Type": "AWS::S3::Bucket",
                    "Properties": {
                        "BucketName": {
                            "Ref": "BucketName",
                        },
                    },
                },
            },
            "Parameters": {
                ...originalTemplate.Parameters,
            },
        });
    });
    test('getParameter() throws an exception if asked for a Parameter with a name that is not present in the template', () => {
        const cfnTemplate = includeTestTemplate(stack, 'bucket-with-parameters.json');
        expect(() => {
            cfnTemplate.getParameter('FakeBucketNameThatDoesNotExist');
        }).toThrow(/Parameter with name 'FakeBucketNameThatDoesNotExist' was not found in the template/);
    });
    test('reflects changes to a retrieved CfnParameter object in the resulting template', () => {
        const cfnTemplate = includeTestTemplate(stack, 'bucket-with-parameters.json');
        const stringParam = cfnTemplate.getParameter('BucketName');
        const numberParam = cfnTemplate.getParameter('CorsMaxAge');
        stringParam.default = 'MyDefault';
        stringParam.allowedPattern = '[0-9]*$';
        stringParam.allowedValues = ['123123', '456789'];
        stringParam.constraintDescription = 'MyNewConstraint';
        stringParam.description = 'a string of numeric characters';
        stringParam.maxLength = 6;
        stringParam.minLength = 2;
        numberParam.maxValue = 100;
        numberParam.minValue = 4;
        numberParam.noEcho = false;
        numberParam.type = "NewType";
        const originalTemplate = loadTestFileToJsObject('bucket-with-parameters.json');
        expect(stack).toMatchTemplate({
            "Resources": {
                ...originalTemplate.Resources,
            },
            "Parameters": {
                ...originalTemplate.Parameters,
                "BucketName": {
                    ...originalTemplate.Parameters.BucketName,
                    "Default": "MyDefault",
                    "AllowedPattern": "[0-9]*$",
                    "AllowedValues": ["123123", "456789"],
                    "ConstraintDescription": "MyNewConstraint",
                    "Description": "a string of numeric characters",
                    "MaxLength": 6,
                    "MinLength": 2,
                },
                "CorsMaxAge": {
                    ...originalTemplate.Parameters.CorsMaxAge,
                    "MaxValue": 100,
                    "MinValue": 4,
                    "NoEcho": false,
                    "Type": "NewType",
                },
            },
        });
    });
    test('reflects changes to a retrieved CfnCondition object in the resulting template', () => {
        const cfnTemplate = includeTestTemplate(stack, 'resource-attribute-condition.json');
        const alwaysFalseCondition = cfnTemplate.getCondition('AlwaysFalseCond');
        alwaysFalseCondition.expression = core.Fn.conditionEquals(1, 2);
        expect(stack).toMatchTemplate({
            "Conditions": {
                "AlwaysFalseCond": {
                    "Fn::Equals": [1, 2],
                },
            },
            "Resources": {
                "Bucket": {
                    "Type": "AWS::S3::Bucket",
                    "Condition": "AlwaysFalseCond",
                },
            },
        });
    });
    test('correctly handles the CreationPolicy resource attribute', () => {
        const cfnTemplate = includeTestTemplate(stack, 'resource-attribute-creation-policy.json');
        const cfnBucket = cfnTemplate.getResource('Bucket');
        expect(cfnBucket.cfnOptions.creationPolicy).toBeDefined();
        expect(stack).toMatchTemplate(loadTestFileToJsObject('resource-attribute-creation-policy.json'));
    });
    test('correctly handles the UpdatePolicy resource attribute', () => {
        const cfnTemplate = includeTestTemplate(stack, 'resource-attribute-update-policy.json');
        const cfnBucket = cfnTemplate.getResource('Bucket');
        expect(cfnBucket.cfnOptions.updatePolicy).toBeDefined();
        expect(stack).toMatchTemplate(loadTestFileToJsObject('resource-attribute-update-policy.json'));
    });
    test("correctly handles referencing the ingested template's resources across Stacks", () => {
        // for cross-stack sharing to work, we need an App
        const app = new core.App();
        stack = new core.Stack(app, 'MyStack');
        const cfnTemplate = includeTestTemplate(stack, 'only-empty-bucket.json');
        const cfnBucket = cfnTemplate.getResource('Bucket');
        const otherStack = new core.Stack(app, 'OtherStack');
        const role = new iam.Role(otherStack, 'Role', {
            assumedBy: new iam.AnyPrincipal(),
        });
        role.addToPolicy(new iam.PolicyStatement({
            actions: ['s3:*'],
            resources: [cfnBucket.attrArn],
        }));
        expect(stack).toMatchTemplate({
            ...loadTestFileToJsObject('only-empty-bucket.json'),
            "Outputs": {
                "ExportsOutputFnGetAttBucketArn436138FE": {
                    "Value": {
                        "Fn::GetAtt": ["Bucket", "Arn"],
                    },
                    "Export": {
                        "Name": "MyStack:ExportsOutputFnGetAttBucketArn436138FE",
                    },
                },
            },
        });
        expect(otherStack).toHaveResourceLike('AWS::IAM::Policy', {
            "PolicyDocument": {
                "Statement": [
                    {
                        "Action": "s3:*",
                        "Resource": {
                            "Fn::ImportValue": "MyStack:ExportsOutputFnGetAttBucketArn436138FE",
                        },
                    },
                ],
            },
        });
    });
    test('correctly re-names references to resources in the template if their logical IDs have been changed', () => {
        const cfnTemplate = includeTestTemplate(stack, 'bucket-with-encryption-key.json');
        const cfnKey = cfnTemplate.getResource('Key');
        cfnKey.overrideLogicalId('TotallyDifferentKey');
        const originalTemplate = loadTestFileToJsObject('bucket-with-encryption-key.json');
        expect(stack).toMatchTemplate({
            "Resources": {
                "Bucket": {
                    "Type": "AWS::S3::Bucket",
                    "Properties": {
                        "BucketEncryption": {
                            "ServerSideEncryptionConfiguration": [
                                {
                                    "ServerSideEncryptionByDefault": {
                                        "KMSMasterKeyID": {
                                            "Fn::GetAtt": ["TotallyDifferentKey", "Arn"],
                                        },
                                        "SSEAlgorithm": "aws:kms",
                                    },
                                },
                            ],
                        },
                    },
                    "Metadata": {
                        "Object1": "Location1",
                        "KeyRef": { "Ref": "TotallyDifferentKey" },
                        "KeyArn": { "Fn::GetAtt": ["TotallyDifferentKey", "Arn"] },
                    },
                    "DeletionPolicy": "Retain",
                    "UpdateReplacePolicy": "Retain",
                },
                "TotallyDifferentKey": originalTemplate.Resources.Key,
            },
        });
    });
    test('can include a template with a custom resource that uses attributes', () => {
        const cfnTemplate = includeTestTemplate(stack, 'custom-resource-with-attributes.json');
        expect(stack).toMatchTemplate(loadTestFileToJsObject('custom-resource-with-attributes.json'));
        const alwaysFalseCondition = cfnTemplate.getCondition('AlwaysFalseCond');
        expect(cfnTemplate.getResource('CustomBucket').cfnOptions.condition).toBe(alwaysFalseCondition);
    });
    test("throws an exception when a custom resource uses a Condition attribute that doesn't exist in the template", () => {
        expect(() => {
            includeTestTemplate(stack, 'custom-resource-with-bad-condition.json');
        }).toThrow(/Resource 'CustomResource' uses Condition 'AlwaysFalseCond' that doesn't exist/);
    });
    test('can ingest a template that contains outputs and modify them', () => {
        const cfnTemplate = includeTestTemplate(stack, 'outputs-with-references.json');
        const output = cfnTemplate.getOutput('Output1');
        output.value = 'a mutated value';
        output.description = undefined;
        output.exportName = 'an export';
        output.condition = new core.CfnCondition(stack, 'MyCondition', {
            expression: core.Fn.conditionIf('AlwaysFalseCond', core.Aws.NO_VALUE, true),
        });
        const originalTemplate = loadTestFileToJsObject('outputs-with-references.json');
        expect(stack).toMatchTemplate({
            "Conditions": {
                ...originalTemplate.Conditions,
                "MyCondition": {
                    "Fn::If": [
                        "AlwaysFalseCond",
                        { "Ref": "AWS::NoValue" },
                        true,
                    ],
                },
            },
            "Parameters": {
                ...originalTemplate.Parameters,
            },
            "Resources": {
                ...originalTemplate.Resources,
            },
            "Outputs": {
                "Output1": {
                    "Value": "a mutated value",
                    "Export": {
                        "Name": "an export",
                    },
                    "Condition": "MyCondition",
                },
                "OutputWithNoCondition": {
                    "Value": "some-value",
                },
            },
        });
    });
    test('can ingest a template that contains outputs and get those outputs', () => {
        const cfnTemplate = includeTestTemplate(stack, 'outputs-with-references.json');
        const output = cfnTemplate.getOutput('Output1');
        expect(output.condition).toBe(cfnTemplate.getCondition('AlwaysFalseCond'));
        expect(output.description).toBeDefined();
        expect(output.value).toBeDefined();
        expect(output.exportName).toBeDefined();
        expect(stack).toMatchTemplate(loadTestFileToJsObject('outputs-with-references.json'));
    });
    test("throws an exception when attempting to retrieve an Output that doesn't exist", () => {
        const cfnTemplate = includeTestTemplate(stack, 'outputs-with-references.json');
        expect(() => {
            cfnTemplate.getOutput('FakeOutput');
        }).toThrow(/Output with logical ID 'FakeOutput' was not found in the template/);
    });
    test('can ingest a template that contains Mappings, and retrieve those Mappings', () => {
        const cfnTemplate = includeTestTemplate(stack, 'only-mapping-and-bucket.json');
        const someMapping = cfnTemplate.getMapping('SomeMapping');
        someMapping.setValue('region', 'key2', 'value2');
        expect(stack).toMatchTemplate({
            "Mappings": {
                "SomeMapping": {
                    "region": {
                        "key1": "value1",
                        "key2": "value2",
                    },
                },
            },
            "Resources": {
                "Bucket": {
                    "Type": "AWS::S3::Bucket",
                    "Properties": {
                        "BucketName": {
                            "Fn::FindInMap": [
                                "SomeMapping",
                                { "Ref": "AWS::Region" },
                                "key1",
                            ],
                        },
                    },
                },
            },
        });
    });
    test("throws an exception when attempting to retrieve a Mapping that doesn't exist in the template", () => {
        const cfnTemplate = includeTestTemplate(stack, 'only-mapping-and-bucket.json');
        expect(() => {
            cfnTemplate.getMapping('NonExistentMapping');
        }).toThrow(/Mapping with name 'NonExistentMapping' was not found in the template/);
    });
    test('handles renaming Mapping references', () => {
        const cfnTemplate = includeTestTemplate(stack, 'only-mapping-and-bucket.json');
        const someMapping = cfnTemplate.getMapping('SomeMapping');
        someMapping.overrideLogicalId('DifferentMapping');
        expect(stack).toMatchTemplate({
            "Mappings": {
                "DifferentMapping": {
                    "region": {
                        "key1": "value1",
                    },
                },
            },
            "Resources": {
                "Bucket": {
                    "Type": "AWS::S3::Bucket",
                    "Properties": {
                        "BucketName": {
                            "Fn::FindInMap": [
                                "DifferentMapping",
                                { "Ref": "AWS::Region" },
                                "key1",
                            ],
                        },
                    },
                },
            },
        });
    });
    test('can ingest a template that contains Rules, and allows retrieving those Rules', () => {
        const cfnTemplate = includeTestTemplate(stack, 'only-parameters-and-rule.json');
        const rule = cfnTemplate.getRule('TestVpcRule');
        expect(rule).toBeDefined();
        expect(stack).toMatchTemplate(loadTestFileToJsObject('only-parameters-and-rule.json'));
    });
    test('fails when trying to replace Parameters referenced in Fn::ValueOf expressions with user-provided values', () => {
        expect(() => {
            includeTestTemplate(stack, 'only-parameters-and-rule.json', {
                parameters: {
                    'Subnets': ['subnet-1234abcd'],
                },
            });
        }).toThrow(/Cannot substitute parameter 'Subnets' used in Fn::ValueOf expression with attribute 'VpcId'/);
    });
    test("throws an exception when attempting to retrieve a Rule that doesn't exist in the template", () => {
        const cfnTemplate = includeTestTemplate(stack, 'only-parameters-and-rule.json');
        expect(() => {
            cfnTemplate.getRule('DoesNotExist');
        }).toThrow(/Rule with name 'DoesNotExist' was not found in the template/);
    });
    test('can ingest a template that contains Hooks, and allows retrieving those Hooks', () => {
        const cfnTemplate = includeTestTemplate(stack, 'hook-code-deploy-blue-green-ecs.json');
        const hook = cfnTemplate.getHook('EcsBlueGreenCodeDeployHook');
        expect(hook).toBeDefined();
        expect(stack).toMatchTemplate(loadTestFileToJsObject('hook-code-deploy-blue-green-ecs.json'));
    });
    test("throws an exception when attempting to retrieve a Hook that doesn't exist in the template", () => {
        const cfnTemplate = includeTestTemplate(stack, 'hook-code-deploy-blue-green-ecs.json');
        expect(() => {
            cfnTemplate.getHook('DoesNotExist');
        }).toThrow(/Hook with logical ID 'DoesNotExist' was not found in the template/);
    });
    test('replaces references to parameters with the user-specified values in Resources, Conditions, Metadata, and Options sections', () => {
        includeTestTemplate(stack, 'parameter-references.json', {
            parameters: {
                'MyParam': 'my-s3-bucket',
            },
        });
        expect(stack).toMatchTemplate({
            "Transform": {
                "Name": "AWS::Include",
                "Parameters": {
                    "Location": "my-s3-bucket",
                },
            },
            "Metadata": {
                "Field": {
                    "Fn::If": [
                        "AlwaysFalse",
                        "AWS::NoValue",
                        "my-s3-bucket",
                    ],
                },
            },
            "Conditions": {
                "AlwaysFalse": {
                    "Fn::Equals": ["my-s3-bucket", "Invalid?BucketName"],
                },
            },
            "Resources": {
                "Bucket": {
                    "Type": "AWS::S3::Bucket",
                    "Metadata": {
                        "Field": "my-s3-bucket",
                    },
                    "Properties": {
                        "BucketName": "my-s3-bucket",
                    },
                },
            },
            "Outputs": {
                "MyOutput": {
                    "Value": "my-s3-bucket",
                },
            },
        });
    });
    test('replaces parameters with falsey values in Ref expressions', () => {
        includeTestTemplate(stack, 'resource-attribute-creation-policy.json', {
            parameters: {
                'CountParameter': 0,
            },
        });
        expect(stack).toMatchTemplate({
            "Resources": {
                "Bucket": {
                    "Type": "AWS::S3::Bucket",
                    "CreationPolicy": {
                        "AutoScalingCreationPolicy": {
                            "MinSuccessfulInstancesPercent": 50,
                        },
                        "ResourceSignal": {
                            "Count": 0,
                            "Timeout": "PT5H4M3S",
                        },
                    },
                },
            },
        });
    });
    test('replaces parameters in Fn::Sub expressions', () => {
        includeTestTemplate(stack, 'fn-sub-parameters.json', {
            parameters: {
                'MyParam': 'my-s3-bucket',
            },
        });
        expect(stack).toMatchTemplate({
            "Resources": {
                "Bucket": {
                    "Type": "AWS::S3::Bucket",
                    "Properties": {
                        "BucketName": {
                            "Fn::Sub": "my-s3-bucket",
                        },
                    },
                },
            },
        });
    });
    test('does not modify Fn::Sub variables shadowing a replaced parameter', () => {
        includeTestTemplate(stack, 'fn-sub-shadow-parameter.json', {
            parameters: {
                'MyParam': 'MyValue',
            },
        });
        expect(stack).toMatchTemplate({
            "Resources": {
                "Bucket": {
                    "Type": "AWS::S3::Bucket",
                    "Properties": {
                        "BucketName": {
                            "Fn::Sub": [
                                "${MyParam}",
                                {
                                    "MyParam": "MyValue",
                                },
                            ],
                        },
                    },
                },
            },
        });
    });
    test('replaces parameters with falsey values in Fn::Sub expressions', () => {
        includeTestTemplate(stack, 'fn-sub-parameters.json', {
            parameters: {
                'MyParam': '',
            },
        });
        expect(stack).toMatchTemplate({
            "Resources": {
                "Bucket": {
                    "Type": "AWS::S3::Bucket",
                    "Properties": {
                        "BucketName": { "Fn::Sub": "" },
                    },
                },
            },
        });
    });
    test('throws an exception when parameters are passed a resource name', () => {
        expect(() => {
            includeTestTemplate(stack, 'bucket-with-parameters.json', {
                parameters: {
                    'Bucket': 'noChange',
                },
            });
        }).toThrow(/Parameter with logical ID 'Bucket' was not found in the template/);
    });
    test('throws an exception when provided a parameter to replace that is not in the template with parameters', () => {
        expect(() => {
            includeTestTemplate(stack, 'bucket-with-parameters.json', {
                parameters: {
                    'FakeParameter': 'DoesNotExist',
                },
            });
        }).toThrow(/Parameter with logical ID 'FakeParameter' was not found in the template/);
    });
    test('throws an exception when provided a parameter to replace in a template with no parameters', () => {
        expect(() => {
            includeTestTemplate(stack, 'only-empty-bucket.json', {
                parameters: {
                    'FakeParameter': 'DoesNotExist',
                },
            });
        }).toThrow(/Parameter with logical ID 'FakeParameter' was not found in the template/);
    });
});
function includeTestTemplate(scope, testTemplate, props = {}) {
    return new inc.CfnInclude(scope, 'MyScope', {
        templateFile: _testTemplateFilePath(testTemplate),
        parameters: props.parameters,
        preserveLogicalIds: props.preserveLogicalIds,
    });
}
function loadTestFileToJsObject(testTemplate) {
    return futils.readJsonSync(_testTemplateFilePath(testTemplate));
}
function _testTemplateFilePath(testTemplate) {
    return path.join(__dirname, 'test-templates', testTemplate);
}
//# sourceMappingURL=data:application/json;base64,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