"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const path = require("path");
require("@aws-cdk/assert/jest");
const core = require("@aws-cdk/core");
const inc = require("../lib");
const futils = require("../lib/file-utils");
/* eslint-disable quote-props */
/* eslint-disable quotes */
describe('CDK Include', () => {
    let stack;
    beforeEach(() => {
        stack = new core.Stack();
    });
    test('can ingest a template with all long-form CloudFormation functions and output it unchanged', () => {
        includeTestTemplate(stack, 'long-form-vpc.yaml');
        expect(stack).toMatchTemplate(loadTestFileToJsObject('long-form-vpc.yaml'));
    });
    test('can ingest a template with the short form Base64 function', () => {
        includeTestTemplate(stack, 'short-form-base64.yaml');
        expect(stack).toMatchTemplate({
            "Resources": {
                "Base64Bucket": {
                    "Type": "AWS::S3::Bucket",
                    "Properties": {
                        "BucketName": {
                            "Fn::Base64": "NonBase64BucketName",
                        },
                    },
                },
            },
        });
    });
    test('can ingest a template with the short form !Cidr function', () => {
        includeTestTemplate(stack, 'short-form-cidr.yaml');
        expect(stack).toMatchTemplate({
            "Resources": {
                "CidrVpc1": {
                    "Type": "AWS::EC2::VPC",
                    "Properties": {
                        "CidrBlock": {
                            "Fn::Cidr": [
                                "192.168.1.1/24",
                                2,
                                5,
                            ],
                        },
                    },
                },
                "CidrVpc2": {
                    "Type": "AWS::EC2::VPC",
                    "Properties": {
                        "CidrBlock": {
                            "Fn::Cidr": [
                                "192.168.1.1/24",
                                "2",
                                "5",
                            ],
                        },
                    },
                },
            },
        });
    });
    test('can ingest a template with the short form !FindInMap function, in both hyphen and bracket notation', () => {
        includeTestTemplate(stack, 'short-form-find-in-map.yaml');
        expect(stack).toMatchTemplate({
            "Mappings": {
                "RegionMap": {
                    "region-1": {
                        "HVM64": "name1",
                        "HVMG2": "name2",
                    },
                },
            },
            "Resources": {
                "Bucket1": {
                    "Type": "AWS::S3::Bucket",
                    "Properties": {
                        "BucketName": {
                            "Fn::FindInMap": [
                                "RegionMap",
                                "region-1",
                                "HVM64",
                            ],
                        },
                    },
                },
                "Bucket2": {
                    "Type": "AWS::S3::Bucket",
                    "Properties": {
                        "BucketName": {
                            "Fn::FindInMap": [
                                "RegionMap",
                                "region-1",
                                "HVMG2",
                            ],
                        },
                    },
                },
            },
        });
    });
    test('can ingest a template with the short form !GetAtt function', () => {
        includeTestTemplate(stack, 'short-form-get-att.yaml');
        expect(stack).toMatchTemplate({
            "Resources": {
                "ELB": {
                    "Type": "AWS::ElasticLoadBalancing::LoadBalancer",
                    "Properties": {
                        "AvailabilityZones": [
                            "us-east-1a",
                        ],
                        "Listeners": [
                            {
                                "LoadBalancerPort": "80",
                                "InstancePort": "80",
                                "Protocol": "HTTP",
                            },
                        ],
                    },
                },
                "Bucket0": {
                    "Type": "AWS::S3::Bucket",
                    "Properties": { "BucketName": "some-bucket" },
                },
                "Bucket1": {
                    "Type": "AWS::S3::Bucket",
                    "Properties": {
                        "BucketName": { "Fn::GetAtt": ["Bucket0", "Arn"] },
                        "AccessControl": { "Fn::GetAtt": ["ELB", "SourceSecurityGroup.GroupName"] },
                    },
                },
                "Bucket2": {
                    "Type": "AWS::S3::Bucket",
                    "Properties": {
                        "BucketName": { "Fn::GetAtt": ["Bucket1", "Arn"] },
                        "AccessControl": { "Fn::GetAtt": ["ELB", "SourceSecurityGroup.GroupName"] },
                    },
                },
            },
        });
    });
    test('can ingest a template with short form Select, GetAZs, and Ref functions', () => {
        includeTestTemplate(stack, 'short-form-select.yaml');
        expect(stack).toMatchTemplate({
            "Resources": {
                "Subnet1": {
                    "Type": "AWS::EC2::Subnet",
                    "Properties": {
                        "VpcId": {
                            "Fn::Select": [0, { "Fn::GetAZs": "" }],
                        },
                        "CidrBlock": "10.0.0.0/24",
                        "AvailabilityZone": {
                            "Fn::Select": ["0", { "Fn::GetAZs": "eu-west-2" }],
                        },
                    },
                },
                "Subnet2": {
                    "Type": "AWS::EC2::Subnet",
                    "Properties": {
                        "VpcId": {
                            "Ref": "Subnet1",
                        },
                        "CidrBlock": "10.0.0.0/24",
                        "AvailabilityZone": {
                            "Fn::Select": [0, { "Fn::GetAZs": "eu-west-2" }],
                        },
                    },
                },
            },
        });
    });
    test('can ingest a template with the short form !ImportValue function', () => {
        includeTestTemplate(stack, 'short-form-import-value.yaml');
        expect(stack).toMatchTemplate({
            "Resources": {
                "Bucket1": {
                    "Type": "AWS::S3::Bucket",
                    "Properties": {
                        "BucketName": {
                            "Fn::ImportValue": "SomeSharedValue",
                        },
                    },
                },
            },
        });
    });
    test('can ingest a template with the short form !Join function', () => {
        includeTestTemplate(stack, 'short-form-join.yaml');
        expect(stack).toMatchTemplate({
            "Resources": {
                "Bucket": {
                    "Type": "AWS::S3::Bucket",
                    "Properties": {
                        "BucketName": {
                            "Fn::Join": [' ', [
                                    "NamePart1 ",
                                    { "Fn::ImportValue": "SomeSharedValue" },
                                ]],
                        },
                    },
                },
            },
        });
    });
    test('can ingest a template with the short form !Split function that uses both brackets and hyphens', () => {
        includeTestTemplate(stack, 'short-form-split.yaml');
        expect(stack).toMatchTemplate({
            "Resources": {
                "Bucket1": {
                    "Type": "AWS::S3::Bucket",
                    "Properties": {
                        "BucketName": {
                            "Fn::Split": [' ', {
                                    "Fn::ImportValue": "SomeSharedBucketName",
                                }],
                        },
                    },
                },
                "Bucket2": {
                    "Type": "AWS::S3::Bucket",
                    "Properties": {
                        "BucketName": {
                            "Fn::Split": [' ', {
                                    "Fn::ImportValue": "SomeSharedBucketName",
                                }],
                        },
                    },
                },
            },
        });
    });
    test('can ingest a template with the short form !Transform function', () => {
        // Note that this yaml template fails validation. It is unclear how to invoke !Transform.
        includeTestTemplate(stack, 'invalid/short-form-transform.yaml');
        expect(stack).toMatchTemplate({
            "Resources": {
                "Bucket": {
                    "Type": "AWS::S3::Bucket",
                    "Properties": {
                        "BucketName": {
                            "Fn::Transform": {
                                "Name": "SomeMacroName",
                                "Parameters": {
                                    "key1": "value1",
                                    "key2": "value2",
                                },
                            },
                        },
                    },
                },
            },
        });
    });
    test('can ingest a template with the short form conditionals', () => {
        includeTestTemplate(stack, 'short-form-conditionals.yaml');
        expect(stack).toMatchTemplate({
            "Conditions": {
                "AlwaysTrueCond": {
                    "Fn::And": [
                        {
                            "Fn::Not": [
                                { "Fn::Equals": [{ "Ref": "AWS::Region" }, "completely-made-up-region"] },
                            ],
                        },
                        {
                            "Fn::Or": [
                                { "Fn::Equals": [{ "Ref": "AWS::Region" }, "completely-made-up-region"] },
                                { "Fn::Equals": [{ "Ref": "AWS::Region" }, "completely-made-up-region"] },
                            ],
                        },
                    ],
                },
            },
            "Resources": {
                "Bucket": {
                    "Type": "AWS::S3::Bucket",
                    "Properties": {
                        "BucketName": {
                            "Fn::If": [
                                "AlwaysTrueCond",
                                "MyBucketName",
                                { "Ref": "AWS::NoValue" },
                            ],
                        },
                    },
                },
            },
        });
    });
    test('can ingest a template with the short form Conditions', () => {
        includeTestTemplate(stack, 'short-form-conditions.yaml');
        expect(stack).toMatchTemplate({
            "Conditions": {
                "AlwaysTrueCond": {
                    "Fn::Not": [
                        { "Fn::Equals": [{ "Ref": "AWS::Region" }, "completely-made-up-region1"] },
                    ],
                },
                "AnotherAlwaysTrueCond": {
                    "Fn::Not": [
                        { "Fn::Equals": [{ "Ref": "AWS::Region" }, "completely-made-up-region2"] },
                    ],
                },
                "ThirdAlwaysTrueCond": {
                    "Fn::Not": [
                        { "Fn::Equals": [{ "Ref": "AWS::Region" }, "completely-made-up-region3"] },
                    ],
                },
                "CombinedCond": {
                    "Fn::Or": [
                        { "Condition": "AlwaysTrueCond" },
                        { "Condition": "AnotherAlwaysTrueCond" },
                        { "Condition": "ThirdAlwaysTrueCond" },
                    ],
                },
            },
            "Resources": {
                "Bucket": {
                    "Type": "AWS::S3::Bucket",
                    "Properties": {
                        "BucketName": {
                            "Fn::If": [
                                "CombinedCond",
                                "MyBucketName",
                                { "Ref": "AWS::NoValue" },
                            ],
                        },
                    },
                },
            },
        });
    });
    test('can ingest a yaml with long-form functions and output it unchanged', () => {
        includeTestTemplate(stack, 'long-form-subnet.yaml');
        expect(stack).toMatchTemplate(loadTestFileToJsObject('long-form-subnet.yaml'));
    });
    test('can ingest a YAML tempalte with Fn::Sub in string form and output it unchanged', () => {
        includeTestTemplate(stack, 'short-form-fnsub-string.yaml');
        expect(stack).toMatchTemplate(loadTestFileToJsObject('short-form-fnsub-string.yaml'));
    });
    test('can ingest a YAML tmeplate with Fn::Sub in map form and output it unchanged', () => {
        includeTestTemplate(stack, 'short-form-sub-map.yaml');
        expect(stack).toMatchTemplate(loadTestFileToJsObject('short-form-sub-map.yaml'));
    });
    test('the parser throws an error on a YAML tmeplate with short form import value that uses short form sub', () => {
        expect(() => {
            includeTestTemplate(stack, 'invalid/short-form-import-sub.yaml');
        }).toThrow(/A node can have at most one tag/);
    });
});
function includeTestTemplate(scope, testTemplate) {
    return new inc.CfnInclude(scope, 'MyScope', {
        templateFile: _testTemplateFilePath(testTemplate),
    });
}
function loadTestFileToJsObject(testTemplate) {
    return futils.readYamlSync(_testTemplateFilePath(testTemplate));
}
function _testTemplateFilePath(testTemplate) {
    return path.join(__dirname, 'test-templates/yaml', testTemplate);
}
//# sourceMappingURL=data:application/json;base64,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