use std::str::FromStr;

use fastobo::ast::HeaderClause;
use fastobo::ast::HeaderFrame;
use fastobo::ast::Ident;
use fastobo::ast::LiteralPropertyValue;
use fastobo::ast::NaiveDateTime;
use fastobo::ast::NamespaceIdent;
use fastobo::ast::PrefixedIdent;
use fastobo::ast::PropertyValue;
use fastobo::ast::QuotedString;
use fastobo::ast::RelationIdent;
use fastobo::ast::ResourcePropertyValue;
use fastobo::ast::UnquotedString;

use super::FromGraph;
use crate::constants::property::dc;
use crate::constants::property::obo_in_owl;
use crate::constants::property::rdfs;
use crate::error::Result;
use crate::model::BasicPropertyValue;
use crate::model::Meta;

impl FromGraph<Meta> for HeaderFrame {
    fn from_graph(meta: Meta) -> Result<Self> {
        let mut frame = Self::new();

        if let Some(ref def) = meta.definition {}

        if let Some(ref iri) = meta.version {}

        // QUESTION: is this semantically correct?
        for comment in meta.comments {
            frame.push(HeaderClause::Remark(Box::new(UnquotedString::new(comment))));
        }
        for pv in meta.basic_property_values {
            frame.push(HeaderClause::from_graph(pv)?);
        }

        // ... TODO ... //

        Ok(frame)
    }
}

impl FromGraph<BasicPropertyValue> for HeaderClause {
    fn from_graph(pv: BasicPropertyValue) -> Result<Self> {
        match pv.pred.as_str() {
            obo_in_owl::AUTO_GENERATED_BY => Ok(HeaderClause::AutoGeneratedBy(Box::new(
                UnquotedString::new(pv.val),
            ))),
            obo_in_owl::CREATION_DATE | dc::DATE | obo_in_owl::HAS_DATE => {
                let dt = NaiveDateTime::from_str(&pv.val)?;
                Ok(HeaderClause::Date(Box::new(dt)))
            }
            obo_in_owl::HAS_DEFAULT_NAMESPACE => {
                let ns_id = NamespaceIdent::from_str(&pv.val)?;
                Ok(HeaderClause::DefaultNamespace(Box::new(ns_id)))
            }
            obo_in_owl::HAS_OBO_FORMAT_VERSION => Ok(HeaderClause::FormatVersion(Box::new(
                UnquotedString::new(pv.val),
            ))),
            obo_in_owl::NAMESPACE_ID_RULE => Ok(HeaderClause::NamespaceIdRule(Box::new(
                UnquotedString::new(pv.val),
            ))),
            obo_in_owl::SAVED_BY => {
                Ok(HeaderClause::SavedBy(Box::new(UnquotedString::new(pv.val))))
            }
            rdfs::COMMENT => Ok(HeaderClause::Remark(Box::new(UnquotedString::new(pv.val)))),
            other => {
                let rel = RelationIdent::from_str(&other)?;
                let pv = match Ident::from_str(&pv.val) {
                    Ok(id) => PropertyValue::from(ResourcePropertyValue::new(rel, id)),
                    Err(_) => PropertyValue::from(LiteralPropertyValue::new(
                        rel,
                        QuotedString::new(pv.val),
                        Ident::from(PrefixedIdent::new("xsd", "string")),
                    )),
                };
                Ok(HeaderClause::PropertyValue(Box::new(pv)))
            }
        }
    }
}
