use fastobo::ast::HeaderFrame;

use super::Context;
use super::IntoGraphCtx;
use crate::constants::property::obo_in_owl;
use crate::error::Result;
use crate::model::BasicPropertyValue;
use crate::model::Meta;

impl IntoGraphCtx<Meta> for HeaderFrame {
    fn into_graph_ctx(self, ctx: &mut Context) -> Result<Meta> {
        use fastobo::ast::HeaderClause::*;

        let definition = None;
        let mut comments = Vec::new();
        let mut subsets = Vec::new();
        let xrefs = Vec::new();
        let synonyms = Vec::new();
        let mut basic_property_values = Vec::new();
        let mut version = None;
        let deprecated = false;

        // extract the ontology identifier
        let id = self.iter().find_map(|clause| match clause {
            Ontology(id) => Some(id.clone()),
            _ => None,
        });

        for clause in self.into_iter() {
            match clause {
                FormatVersion(v) => {
                    basic_property_values.push(BasicPropertyValue::new(
                        obo_in_owl::HAS_OBO_FORMAT_VERSION.to_string(),
                        v.into_string(),
                    ));
                }
                DataVersion(v) => {
                    if let Some(ref ont) = id {
                        version = Some(format!(
                            "{}{}/{}/{}.owl",
                            crate::constants::uri::OBO,
                            ont.as_str(),
                            v.as_str(),
                            ont.as_str()
                        ));
                    } else {
                        // FIXME: should this be an error instead
                        version = Some(v.into_string());
                    }
                }
                Date(dt) => {
                    basic_property_values.push(BasicPropertyValue::new(
                        obo_in_owl::HAS_DATE.to_string(),
                        dt.to_string(),
                    ));
                }
                SavedBy(name) => {
                    basic_property_values.push(BasicPropertyValue::new(
                        obo_in_owl::SAVED_BY.to_string(),
                        name.into_string(),
                    ));
                }
                AutoGeneratedBy(name) => {
                    basic_property_values.push(BasicPropertyValue::new(
                        obo_in_owl::AUTO_GENERATED_BY.to_string(),
                        name.into_string(),
                    ));
                }
                // NB: Import processed in the header: all imports are merged
                // together as separate graphs into the same graph document.
                Import(import) => (),
                Subsetdef(id, def) => {
                    subsets.push(id.to_string());
                }
                // FIXME: declare a new entity.
                SynonymTypedef(ty, def, optscope) => (),
                DefaultNamespace(ns) => {
                    basic_property_values.push(BasicPropertyValue::new(
                        obo_in_owl::HAS_DEFAULT_NAMESPACE.to_string(),
                        ns.to_string(),
                    ));
                }
                NamespaceIdRule(idrule) => {
                    basic_property_values.push(BasicPropertyValue::new(
                        obo_in_owl::NAMESPACE_ID_RULE.to_string(),
                        idrule.into_string(),
                    ));
                }
                Idspace(prefix, url, optdef) => (),
                TreatXrefsAsEquivalent(prefix) => (),
                TreatXrefsAsGenusDifferentia(prefix, rid, cid) => (),
                TreatXrefsAsReverseGenusDifferentia(prefix, rid, cid) => (),
                TreatXrefsAsRelationship(prefix, rid) => (),
                TreatXrefsAsIsA(prefix) => (),
                TreatXrefsAsHasSubclass(prefix) => (),
                PropertyValue(pv) => {
                    basic_property_values.push(pv.into_graph_ctx(ctx)?);
                }
                Remark(remark) => {
                    comments.push(remark.into_string());
                }
                Ontology(ontology) => (),
                OwlAxioms(axioms) => (),
                Unreserved(key, value) => (),
            }
        }

        Ok(Meta {
            definition,
            comments,
            subsets,
            xrefs,
            synonyms,
            basic_property_values,
            version,
            deprecated,
        })
    }
}
