"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.LowCostECS = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const lib = require("aws-cdk-lib");
const ec2 = require("aws-cdk-lib/aws-ec2");
const ecs = require("aws-cdk-lib/aws-ecs");
const aws_efs_1 = require("aws-cdk-lib/aws-efs");
const aws_events_1 = require("aws-cdk-lib/aws-events");
const aws_events_targets_1 = require("aws-cdk-lib/aws-events-targets");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_logs_1 = require("aws-cdk-lib/aws-logs");
const route53 = require("aws-cdk-lib/aws-route53");
const aws_sns_1 = require("aws-cdk-lib/aws-sns");
const sfn = require("aws-cdk-lib/aws-stepfunctions");
const sfn_tasks = require("aws-cdk-lib/aws-stepfunctions-tasks");
const constructs_1 = require("constructs");
class LowCostECS extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const vpc = props.vpc ??
            new ec2.Vpc(scope, 'Vpc', {
                natGateways: 0,
                subnetConfiguration: [
                    {
                        name: 'PublicSubnet',
                        subnetType: ec2.SubnetType.PUBLIC,
                    },
                ],
            });
        this.cluster = new ecs.Cluster(scope, 'Cluster', {
            vpc: vpc,
            containerInsights: props.containerInsights,
        });
        this.hostAutoScalingGroup = this.cluster.addCapacity('HostInstanceCapacity', {
            machineImage: ecs.EcsOptimizedImage.amazonLinux2(ecs.AmiHardwareType.STANDARD, {
                cachedInContext: true,
            }),
            instanceType: new ec2.InstanceType(props.hostInstanceType ?? 't2.micro'),
            spotPrice: props.hostInstanceSpotPrice,
            vpcSubnets: { subnetType: ec2.SubnetType.PUBLIC },
            associatePublicIpAddress: true,
            minCapacity: 1,
            maxCapacity: 1,
        });
        if (props.securityGroups) {
            this.hostAutoScalingGroup.node.tryRemoveChild('InstanceSecurityGroup');
            props.securityGroups.forEach((sg) => this.hostAutoScalingGroup.addSecurityGroup(sg));
        }
        else {
            this.hostAutoScalingGroup.connections.allowFromAnyIpv4(ec2.Port.tcp(80));
            this.hostAutoScalingGroup.connections.allowFromAnyIpv4(ec2.Port.tcp(443));
            this.hostAutoScalingGroup.connections.allowFrom(ec2.Peer.anyIpv6(), ec2.Port.tcp(80));
            this.hostAutoScalingGroup.connections.allowFrom(ec2.Peer.anyIpv6(), ec2.Port.tcp(443));
        }
        /**
         * Add managed policy to allow ssh through ssm manager
         */
        this.hostAutoScalingGroup.role.addManagedPolicy(aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AmazonSSMManagedInstanceCore'));
        /**
         * Add policy to associate elastic ip on startup
         */
        this.hostAutoScalingGroup.role.addToPrincipalPolicy(new aws_iam_1.PolicyStatement({
            effect: aws_iam_1.Effect.ALLOW,
            actions: ['ec2:DescribeAddresses', 'ec2:AssociateAddress'],
            resources: ['*'],
        }));
        const hostInstanceIp = new ec2.CfnEIP(this, 'HostInstanceIp');
        const tagUniqueId = lib.Names.uniqueId(hostInstanceIp);
        hostInstanceIp.tags.setTag('Name', tagUniqueId);
        const awsCliTag = props.awsCliDockerTag ?? 'latest';
        this.hostAutoScalingGroup.addUserData('INSTANCE_ID=$(curl --silent http://169.254.169.254/latest/meta-data/instance-id)', `ALLOCATION_ID=$(docker run --net=host amazon/aws-cli:${awsCliTag} ec2 describe-addresses --region ${this.hostAutoScalingGroup.env.region} --filter Name=tag:Name,Values=${tagUniqueId} --query 'Addresses[].AllocationId' --output text | head)`, `docker run --net=host amazon/aws-cli:${awsCliTag} ec2 associate-address --region ${this.hostAutoScalingGroup.env.region} --instance-id "$INSTANCE_ID" --allocation-id "$ALLOCATION_ID" --allow-reassociation`);
        this.certFileSystem = new aws_efs_1.FileSystem(this, 'FileSystem', {
            vpc,
            encrypted: true,
            securityGroup: new ec2.SecurityGroup(this, 'FileSystemSecurityGroup', {
                vpc: vpc,
                allowAllOutbound: false,
            }),
            removalPolicy: props.removalPolicy ?? lib.RemovalPolicy.DESTROY,
        });
        this.certFileSystem.connections.allowDefaultPortTo(this.hostAutoScalingGroup);
        this.certFileSystem.connections.allowDefaultPortFrom(this.hostAutoScalingGroup);
        /**
         * ARecord to Elastic ip
         */
        const hostedZone = route53.HostedZone.fromLookup(this, 'HostedZone', {
            domainName: props.hostedZoneDomain,
        });
        const records = props.recordDomainNames ?? [hostedZone.zoneName];
        records.forEach((record) => new route53.ARecord(this, `ARecord${record}`, {
            zone: hostedZone,
            recordName: record,
            target: route53.RecordTarget.fromIpAddresses(hostInstanceIp.ref),
        }));
        /**
         * Certbot Task Definition
         * Mounts generated certificate to EFS
         */
        const logGroup = props.logGroup ??
            new aws_logs_1.LogGroup(this, 'LogGroup', {
                retention: aws_logs_1.RetentionDays.TWO_YEARS,
                removalPolicy: props.removalPolicy ?? lib.RemovalPolicy.DESTROY,
            });
        const certbotTaskDefinition = new ecs.Ec2TaskDefinition(this, 'CertbotTaskDefinition');
        certbotTaskDefinition.addToTaskRolePolicy(new aws_iam_1.PolicyStatement({
            effect: aws_iam_1.Effect.ALLOW,
            actions: ['route53:ListHostedZones', 'route53:GetChange'],
            resources: ['*'],
        }));
        certbotTaskDefinition.addToTaskRolePolicy(new aws_iam_1.PolicyStatement({
            effect: aws_iam_1.Effect.ALLOW,
            actions: ['route53:ChangeResourceRecordSets'],
            resources: [hostedZone.hostedZoneArn],
        }));
        const certbotTag = props.certbotDockerTag ?? 'v1.29.0';
        const certbotContainer = certbotTaskDefinition.addContainer('CertbotContainer', {
            image: ecs.ContainerImage.fromRegistry(`certbot/dns-route53:${certbotTag}`),
            containerName: 'certbot',
            memoryReservationMiB: 64,
            command: [
                'certonly',
                '--verbose',
                '--preferred-challenges=dns-01',
                '--dns-route53',
                '--dns-route53-propagation-seconds=300',
                '--non-interactive',
                '--agree-tos',
                '--expand',
                '-m',
                props.email,
                '--cert-name',
                records[0],
                ...records.flatMap((domain) => ['-d', domain]),
            ],
            logging: ecs.LogDriver.awsLogs({
                logGroup,
                streamPrefix: certbotTag,
            }),
        });
        this.certFileSystem.grant(certbotTaskDefinition.taskRole, 'elasticfilesystem:ClientWrite');
        certbotTaskDefinition.addVolume({
            name: 'certVolume',
            efsVolumeConfiguration: {
                fileSystemId: this.certFileSystem.fileSystemId,
            },
        });
        certbotContainer.addMountPoints({
            sourceVolume: 'certVolume',
            containerPath: '/etc/letsencrypt',
            readOnly: false,
        });
        /**
         * Schedule Certbot certificate create/renew on Step Functions
         * Sends email notification on certbot failure
         */
        this.topic = new aws_sns_1.Topic(this, 'Topic');
        new aws_sns_1.Subscription(this, 'EmailSubscription', {
            topic: this.topic,
            protocol: aws_sns_1.SubscriptionProtocol.EMAIL,
            endpoint: props.email,
        });
        const certbotRunTask = new sfn_tasks.EcsRunTask(this, 'CreateCertificate', {
            cluster: this.cluster,
            taskDefinition: certbotTaskDefinition,
            launchTarget: new sfn_tasks.EcsEc2LaunchTarget(),
            integrationPattern: sfn.IntegrationPattern.RUN_JOB,
        });
        certbotRunTask.addCatch(new sfn_tasks.SnsPublish(this, 'SendEmailOnFailure', {
            topic: this.topic,
            message: sfn.TaskInput.fromJsonPathAt('$'),
        }).next(new sfn.Fail(this, 'Fail')));
        certbotRunTask.addRetry({
            interval: lib.Duration.seconds(20),
        });
        const certbotStateMachine = new sfn.StateMachine(this, 'StateMachine', {
            definition: certbotRunTask,
        });
        new aws_events_1.Rule(this, 'CertbotScheduleRule', {
            schedule: aws_events_1.Schedule.rate(lib.Duration.days(props.certbotScheduleInterval ?? 60)),
            targets: [new aws_events_targets_1.SfnStateMachine(certbotStateMachine)],
        });
        /**
         * Server ECS task
         */
        this.serverTaskDefinition = this.createTaskDefinition(props.serverTaskDefinition ?? this.sampleTaskDefinition(records, logGroup));
        if (!this.serverTaskDefinition.defaultContainer) {
            throw new Error('defaultContainer is required for serverTaskDefinition. Add at least one essential container.');
        }
        this.certFileSystem.grant(this.serverTaskDefinition.taskRole, 'elasticfilesystem:ClientMount');
        this.serverTaskDefinition.addVolume({
            name: 'certVolume',
            efsVolumeConfiguration: {
                fileSystemId: this.certFileSystem.fileSystemId,
            },
        });
        this.serverTaskDefinition.defaultContainer.addMountPoints({
            sourceVolume: 'certVolume',
            containerPath: '/etc/letsencrypt',
            readOnly: true,
        });
        /**
         * AWS cli container to execute certbot sfn before the default container startup.
         */
        this.serverTaskDefinition.defaultContainer.addContainerDependencies({
            container: this.serverTaskDefinition.addContainer('AWSCliContainer', {
                image: ecs.ContainerImage.fromRegistry(`amazon/aws-cli:${awsCliTag}`),
                containerName: 'aws-cli',
                memoryReservationMiB: 64,
                entryPoint: ['/bin/bash', '-c'],
                command: [
                    `set -eux
          aws configure set region ${certbotStateMachine.env.region} && \\
          aws configure set output text && \\
          EXECUTION_ARN=$(aws stepfunctions start-execution --state-machine-arn ${certbotStateMachine.stateMachineArn} --query executionArn) && \\
          until [ $(aws stepfunctions describe-execution --execution-arn "$EXECUTION_ARN" --query status) != RUNNING ];
          do
            echo "Waiting for $EXECUTION_ARN"
            sleep 10
          done`,
                ],
                essential: false,
                logging: ecs.LogDriver.awsLogs({
                    logGroup: logGroup,
                    streamPrefix: awsCliTag,
                }),
            }),
            condition: ecs.ContainerDependencyCondition.COMPLETE,
        });
        certbotStateMachine.grantExecution(this.serverTaskDefinition.taskRole, 'states:DescribeExecution');
        certbotStateMachine.grantStartExecution(this.serverTaskDefinition.taskRole);
        this.service = new ecs.Ec2Service(this, 'Service', {
            cluster: this.cluster,
            taskDefinition: this.serverTaskDefinition,
            desiredCount: 1,
            minHealthyPercent: 0,
            maxHealthyPercent: 100,
            circuitBreaker: {
                rollback: true,
            },
            enableExecuteCommand: true,
        });
        new lib.CfnOutput(this, 'PublicIpAddress', { value: hostInstanceIp.ref });
        new lib.CfnOutput(this, 'StateMachineName', { value: certbotStateMachine.stateMachineName });
        new lib.CfnOutput(this, 'ClusterName', { value: this.cluster.clusterName });
        new lib.CfnOutput(this, 'ServiceName', { value: this.service.serviceName });
    }
    createTaskDefinition(taskDefinitionOptions) {
        const serverTaskDefinition = new ecs.Ec2TaskDefinition(this, 'ServerTaskDefinition', taskDefinitionOptions.taskDefinition);
        taskDefinitionOptions.containers.forEach((containerDefinition, index) => {
            serverTaskDefinition.addContainer(containerDefinition.containerName ?? `container${index}`, containerDefinition);
        });
        taskDefinitionOptions.volumes?.forEach((volume) => serverTaskDefinition.addVolume(volume));
        return serverTaskDefinition;
    }
    sampleTaskDefinition(records, logGroup) {
        return {
            containers: [
                {
                    image: ecs.ContainerImage.fromAsset(path.join(__dirname, '../examples/containers/nginx')),
                    containerName: 'nginx',
                    memoryReservationMiB: 64,
                    essential: true,
                    environment: {
                        SERVER_NAME: records.join(' '),
                        CERT_NAME: records[0],
                    },
                    logging: ecs.LogDrivers.awsLogs({
                        logGroup: logGroup,
                        streamPrefix: 'sample',
                    }),
                    portMappings: [
                        {
                            hostPort: 80,
                            containerPort: 80,
                            protocol: ecs.Protocol.TCP,
                        },
                        {
                            hostPort: 443,
                            containerPort: 443,
                            protocol: ecs.Protocol.TCP,
                        },
                    ],
                },
            ],
        };
    }
}
exports.LowCostECS = LowCostECS;
_a = JSII_RTTI_SYMBOL_1;
LowCostECS[_a] = { fqn: "low-cost-ecs.LowCostECS", version: "0.0.39" };
//# sourceMappingURL=data:application/json;base64,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