"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.CloudfrontWebAcl = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*********************************************************************************************************************
 Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.

 Licensed under the Apache License, Version 2.0 (the "License").
 You may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.
 ******************************************************************************************************************** */
const path = require("path");
const pdk_nag_1 = require("@aws-prototyping-sdk/pdk-nag");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_lambda_1 = require("aws-cdk-lib/aws-lambda");
const custom_resources_1 = require("aws-cdk-lib/custom-resources");
const cdk_nag_1 = require("cdk-nag");
const constructs_1 = require("constructs");
/**
 * This construct creates a WAFv2 Web ACL for cloudfront in the us-east-1 region (required for cloudfront) no matter the
 * region of the parent cdk stack.
 */
class CloudfrontWebAcl extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const stack = aws_cdk_lib_1.Stack.of(this);
        const aclName = `${stack.stackName}-${id}`; // Unique per stack
        const onEventHandler = this.createOnEventHandler(stack, aclName);
        const customResource = this.createAclCustomResource(stack, aclName, onEventHandler, props);
        this.webAclId = customResource.getAttString("WebAclId");
        this.webAclArn = customResource.getAttString("WebAclArn");
    }
    /**
     * Creates an event handler for managing an ACL in us-east-1.
     *
     * @param stack containing Stack instance.
     * @param aclName name of the ACL to manage.
     * @private
     */
    createOnEventHandler(stack, aclName) {
        const onEventHandlerName = `${pdk_nag_1.PDKNag.getStackPrefix(stack)
            .split("/")
            .join("-")}OnEventHandler`;
        const onEventHandlerRole = new aws_iam_1.Role(this, "OnEventHandlerRole", {
            assumedBy: new aws_iam_1.ServicePrincipal("lambda.amazonaws.com"),
            inlinePolicies: {
                logs: new aws_iam_1.PolicyDocument({
                    statements: [
                        new aws_iam_1.PolicyStatement({
                            effect: aws_iam_1.Effect.ALLOW,
                            actions: [
                                "logs:CreateLogGroup",
                                "logs:CreateLogStream",
                                "logs:PutLogEvents",
                            ],
                            resources: [
                                `arn:aws:logs:${stack.region}:${stack.account}:log-group:/aws/lambda/${onEventHandlerName}`,
                                `arn:aws:logs:${stack.region}:${stack.account}:log-group:/aws/lambda/${onEventHandlerName}:*`,
                            ],
                        }),
                    ],
                }),
                wafv2: new aws_iam_1.PolicyDocument({
                    statements: [
                        new aws_iam_1.PolicyStatement({
                            effect: aws_iam_1.Effect.ALLOW,
                            actions: [
                                "wafv2:CreateWebACL",
                                "wafv2:DeleteWebACL",
                                "wafv2:UpdateWebACL",
                                "wafv2:GetWebACL",
                            ],
                            resources: [
                                `arn:aws:wafv2:us-east-1:${stack.account}:global/ipset/${aclName}-IPSet/*`,
                                `arn:aws:wafv2:us-east-1:${stack.account}:global/webacl/${aclName}/*`,
                                `arn:aws:wafv2:us-east-1:${stack.account}:global/managedruleset/*/*`,
                            ],
                        }),
                        new aws_iam_1.PolicyStatement({
                            effect: aws_iam_1.Effect.ALLOW,
                            actions: [
                                "wafv2:CreateIPSet",
                                "wafv2:DeleteIPSet",
                                "wafv2:UpdateIPSet",
                                "wafv2:GetIPSet",
                            ],
                            resources: [
                                `arn:aws:wafv2:us-east-1:${stack.account}:global/ipset/${aclName}-IPSet/*`,
                            ],
                        }),
                    ],
                }),
            },
        });
        const onEventHandler = new aws_lambda_1.Function(this, "CloudfrontWebAclOnEventHandler", {
            code: aws_lambda_1.Code.fromAsset(path.join(__dirname, "../lib/webacl_event_handler")),
            role: onEventHandlerRole,
            functionName: onEventHandlerName,
            handler: "index.onEvent",
            runtime: aws_lambda_1.Runtime.NODEJS_16_X,
            timeout: aws_cdk_lib_1.Duration.seconds(300),
        });
        cdk_nag_1.NagSuppressions.addResourceSuppressions(onEventHandlerRole, [
            {
                id: "AwsSolutions-IAM5",
                reason: "WafV2 resources have been scoped down to the ACL/IPSet level, however * is still needed as resource id's are created just in time.",
                appliesTo: [
                    {
                        regex: `/^Resource::arn:aws:wafv2:us-east-1:${pdk_nag_1.PDKNag.getStackAccountRegex(stack)}:global/(.*)$/g`,
                    },
                ],
            },
            {
                id: "AwsSolutions-IAM5",
                reason: "Cloudwatch resources have been scoped down to the LogGroup level, however * is still needed as stream names are created just in time.",
                appliesTo: [
                    {
                        regex: `/^Resource::arn:aws:logs:${pdk_nag_1.PDKNag.getStackRegionRegex(stack)}:${pdk_nag_1.PDKNag.getStackAccountRegex(stack)}:log-group:/aws/lambda/${onEventHandlerName}:\*/g`,
                    },
                ],
            },
        ], true);
        return onEventHandler;
    }
    /**
     * Creates a Custom resource to manage the deployment of the ACL.
     *
     * @param stack containing Stack instance.
     * @param aclName name of the ACL to manage.
     * @param onEventHandler event handler to use for deployment.
     * @param props user provided properties for configuring the ACL.
     * @private
     */
    createAclCustomResource(stack, aclName, onEventHandler, props) {
        const providerFunctionName = `${onEventHandler.functionName}-Provider`;
        const providerRole = new aws_iam_1.Role(this, "CloudfrontWebAclProviderRole", {
            assumedBy: new aws_iam_1.ServicePrincipal("lambda.amazonaws.com"),
            inlinePolicies: {
                logs: new aws_iam_1.PolicyDocument({
                    statements: [
                        new aws_iam_1.PolicyStatement({
                            effect: aws_iam_1.Effect.ALLOW,
                            actions: [
                                "logs:CreateLogGroup",
                                "logs:CreateLogStream",
                                "logs:PutLogEvents",
                            ],
                            resources: [
                                `arn:aws:logs:${stack.region}:${stack.account}:log-group:/aws/lambda/${providerFunctionName}`,
                                `arn:aws:logs:${stack.region}:${stack.account}:log-group:/aws/lambda/${providerFunctionName}:*`,
                            ],
                        }),
                    ],
                }),
            },
        });
        const provider = new custom_resources_1.Provider(this, "CloudfrontWebAclProvider", {
            onEventHandler,
            role: providerRole,
            providerFunctionName,
        });
        cdk_nag_1.NagSuppressions.addResourceSuppressions(providerRole, [
            {
                id: "AwsSolutions-IAM5",
                reason: "Cloudwatch resources have been scoped down to the LogGroup level, however * is still needed as stream names are created just in time.",
            },
        ], true);
        cdk_nag_1.NagSuppressions.addResourceSuppressions(provider, [
            {
                id: "AwsSolutions-L1",
                reason: "Latest runtime cannot be configured. CDK will need to upgrade the Provider construct accordingly.",
            },
        ], true);
        return new aws_cdk_lib_1.CustomResource(this, "CFWebAclCustomResource", {
            serviceToken: provider.serviceToken,
            properties: {
                ID: aclName,
                MANAGED_RULES: props?.managedRules ?? [
                    { vendor: "AWS", name: "AWSManagedRulesCommonRuleSet" },
                ],
                CIDR_ALLOW_LIST: props?.cidrAllowList,
            },
        });
    }
}
exports.CloudfrontWebAcl = CloudfrontWebAcl;
_a = JSII_RTTI_SYMBOL_1;
CloudfrontWebAcl[_a] = { fqn: "@aws-prototyping-sdk/static-website.CloudfrontWebAcl", version: "0.12.7" };
//# sourceMappingURL=data:application/json;base64,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