import { Construct, Duration, IConstruct, SecretValue } from '@aws-cdk/core';
import { CfnListener } from '../elasticloadbalancingv2.generated';
import { IListenerAction } from '../shared/listener-action';
import { IApplicationListener } from './application-listener';
import { IApplicationTargetGroup } from './application-target-group';
/**
 * What to do when a client makes a request to a listener.
 *
 * Some actions can be combined with other ones (specifically,
 * you can perform authentication before serving the request).
 *
 * Multiple actions form a linked chain; the chain must always terminate in a
 * *(weighted)forward*, *fixedResponse* or *redirect* action.
 *
 * If an action supports chaining, the next action can be indicated
 * by passing it in the `next` property.
 *
 * (Called `ListenerAction` instead of the more strictly correct
 * `ListenerAction` because this is the class most users interact
 * with, and we want to make it not too visually overwhelming).
 *
 * @stability stable
 */
export declare class ListenerAction implements IListenerAction {
    private readonly actionJson;
    protected readonly next?: ListenerAction | undefined;
    /**
     * Authenticate using an identity provider (IdP) that is compliant with OpenID Connect (OIDC).
     *
     * @see https://docs.aws.amazon.com/elasticloadbalancing/latest/application/listener-authenticate-users.html#oidc-requirements
     * @stability stable
     */
    static authenticateOidc(options: AuthenticateOidcOptions): ListenerAction;
    /**
     * Forward to one or more Target Groups.
     *
     * @see https://docs.aws.amazon.com/elasticloadbalancing/latest/application/load-balancer-listeners.html#forward-actions
     * @stability stable
     */
    static forward(targetGroups: IApplicationTargetGroup[], options?: ForwardOptions): ListenerAction;
    /**
     * Forward to one or more Target Groups which are weighted differently.
     *
     * @see https://docs.aws.amazon.com/elasticloadbalancing/latest/application/load-balancer-listeners.html#forward-actions
     * @stability stable
     */
    static weightedForward(targetGroups: WeightedTargetGroup[], options?: ForwardOptions): ListenerAction;
    /**
     * Return a fixed response.
     *
     * @see https://docs.aws.amazon.com/elasticloadbalancing/latest/application/load-balancer-listeners.html#fixed-response-actions
     * @stability stable
     */
    static fixedResponse(statusCode: number, options?: FixedResponseOptions): ListenerAction;
    /**
     * Redirect to a different URI.
     *
     * A URI consists of the following components:
     * protocol://hostname:port/path?query. You must modify at least one of the
     * following components to avoid a redirect loop: protocol, hostname, port, or
     * path. Any components that you do not modify retain their original values.
     *
     * You can reuse URI components using the following reserved keywords:
     *
     * - `#{protocol}`
     * - `#{host}`
     * - `#{port}`
     * - `#{path}` (the leading "/" is removed)
     * - `#{query}`
     *
     * For example, you can change the path to "/new/#{path}", the hostname to
     * "example.#{host}", or the query to "#{query}&value=xyz".
     *
     * @see https://docs.aws.amazon.com/elasticloadbalancing/latest/application/load-balancer-listeners.html#redirect-actions
     * @stability stable
     */
    static redirect(options: RedirectOptions): ListenerAction;
    /**
     * Create an instance of ListenerAction.
     *
     * The default class should be good enough for most cases and
     * should be created by using one of the static factory functions,
     * but allow overriding to make sure we allow flexibility for the future.
     *
     * @stability stable
     */
    protected constructor(actionJson: CfnListener.ActionProperty, next?: ListenerAction | undefined);
    /**
     * Render the actions in this chain.
     *
     * @stability stable
     */
    renderActions(): CfnListener.ActionProperty[];
    /**
     * Called when the action is being used in a listener.
     *
     * @stability stable
     */
    bind(scope: Construct, listener: IApplicationListener, associatingConstruct?: IConstruct): void;
    /**
     * Renumber the "order" fields in the actions array.
     *
     * We don't number for 0 or 1 elements, but otherwise number them 1...#actions
     * so ELB knows about the right order.
     *
     * Do this in `ListenerAction` instead of in `Listener` so that we give
     * users the opportunity to override by subclassing and overriding `renderActions`.
     *
     * @stability stable
     */
    protected renumber(actions: CfnListener.ActionProperty[]): CfnListener.ActionProperty[];
}
/**
 * Options for `ListenerAction.forward()`.
 *
 * @stability stable
 */
export interface ForwardOptions {
    /**
     * For how long clients should be directed to the same target group.
     *
     * Range between 1 second and 7 days.
     *
     * @default - No stickiness
     * @stability stable
     */
    readonly stickinessDuration?: Duration;
}
/**
 * A Target Group and weight combination.
 *
 * @stability stable
 */
export interface WeightedTargetGroup {
    /**
     * The target group.
     *
     * @stability stable
     */
    readonly targetGroup: IApplicationTargetGroup;
    /**
     * The target group's weight.
     *
     * Range is [0..1000).
     *
     * @default 1
     * @stability stable
     */
    readonly weight?: number;
}
/**
 * Options for `ListenerAction.fixedResponse()`.
 *
 * @stability stable
 */
export interface FixedResponseOptions {
    /**
     * Content Type of the response.
     *
     * Valid Values: text/plain | text/css | text/html | application/javascript | application/json
     *
     * @default - Automatically determined
     * @stability stable
     */
    readonly contentType?: string;
    /**
     * The response body.
     *
     * @default - No body
     * @stability stable
     */
    readonly messageBody?: string;
}
/**
 * Options for `ListenerAction.redirect()`.
 *
 * A URI consists of the following components:
 * protocol://hostname:port/path?query. You must modify at least one of the
 * following components to avoid a redirect loop: protocol, hostname, port, or
 * path. Any components that you do not modify retain their original values.
 *
 * You can reuse URI components using the following reserved keywords:
 *
 * - `#{protocol}`
 * - `#{host}`
 * - `#{port}`
 * - `#{path}` (the leading "/" is removed)
 * - `#{query}`
 *
 * For example, you can change the path to "/new/#{path}", the hostname to
 * "example.#{host}", or the query to "#{query}&value=xyz".
 *
 * @stability stable
 */
export interface RedirectOptions {
    /**
     * The hostname.
     *
     * This component is not percent-encoded. The hostname can contain #{host}.
     *
     * @default - No change
     * @stability stable
     */
    readonly host?: string;
    /**
     * The absolute path, starting with the leading "/".
     *
     * This component is not percent-encoded. The path can contain #{host}, #{path}, and #{port}.
     *
     * @default - No change
     * @stability stable
     */
    readonly path?: string;
    /**
     * The port.
     *
     * You can specify a value from 1 to 65535 or #{port}.
     *
     * @default - No change
     * @stability stable
     */
    readonly port?: string;
    /**
     * The protocol.
     *
     * You can specify HTTP, HTTPS, or #{protocol}. You can redirect HTTP to HTTP, HTTP to HTTPS, and HTTPS to HTTPS. You cannot redirect HTTPS to HTTP.
     *
     * @default - No change
     * @stability stable
     */
    readonly protocol?: string;
    /**
     * The query parameters, URL-encoded when necessary, but not percent-encoded.
     *
     * Do not include the leading "?", as it is automatically added. You can specify any of the reserved keywords.
     *
     * @default - No change
     * @stability stable
     */
    readonly query?: string;
    /**
     * The HTTP redirect code.
     *
     * The redirect is either permanent (HTTP 301) or temporary (HTTP 302).
     *
     * @default false
     * @stability stable
     */
    readonly permanent?: boolean;
}
/**
 * Options for `ListenerAction.authenciateOidc()`.
 *
 * @stability stable
 */
export interface AuthenticateOidcOptions {
    /**
     * What action to execute next.
     *
     * @stability stable
     */
    readonly next: ListenerAction;
    /**
     * The query parameters (up to 10) to include in the redirect request to the authorization endpoint.
     *
     * @default - No extra parameters
     * @stability stable
     */
    readonly authenticationRequestExtraParams?: Record<string, string>;
    /**
     * The authorization endpoint of the IdP.
     *
     * This must be a full URL, including the HTTPS protocol, the domain, and the path.
     *
     * @stability stable
     */
    readonly authorizationEndpoint: string;
    /**
     * The OAuth 2.0 client identifier.
     *
     * @stability stable
     */
    readonly clientId: string;
    /**
     * The OAuth 2.0 client secret.
     *
     * @stability stable
     */
    readonly clientSecret: SecretValue;
    /**
     * The OIDC issuer identifier of the IdP.
     *
     * This must be a full URL, including the HTTPS protocol, the domain, and the path.
     *
     * @stability stable
     */
    readonly issuer: string;
    /**
     * The behavior if the user is not authenticated.
     *
     * @default UnauthenticatedAction.AUTHENTICATE
     * @stability stable
     */
    readonly onUnauthenticatedRequest?: UnauthenticatedAction;
    /**
     * The set of user claims to be requested from the IdP.
     *
     * To verify which scope values your IdP supports and how to separate multiple values, see the documentation for your IdP.
     *
     * @default "openid"
     * @stability stable
     */
    readonly scope?: string;
    /**
     * The name of the cookie used to maintain session information.
     *
     * @default "AWSELBAuthSessionCookie"
     * @stability stable
     */
    readonly sessionCookieName?: string;
    /**
     * The maximum duration of the authentication session.
     *
     * @default Duration.days(7)
     * @stability stable
     */
    readonly sessionTimeout?: Duration;
    /**
     * The token endpoint of the IdP.
     *
     * This must be a full URL, including the HTTPS protocol, the domain, and the path.
     *
     * @stability stable
     */
    readonly tokenEndpoint: string;
    /**
     * The user info endpoint of the IdP.
     *
     * This must be a full URL, including the HTTPS protocol, the domain, and the path.
     *
     * @stability stable
     */
    readonly userInfoEndpoint: string;
}
/**
 * What to do with unauthenticated requests.
 *
 * @stability stable
 */
export declare enum UnauthenticatedAction {
    /**
     * Return an HTTP 401 Unauthorized error.
     *
     * @stability stable
     */
    DENY = "deny",
    /**
     * Allow the request to be forwarded to the target.
     *
     * @stability stable
     */
    ALLOW = "allow",
    /**
     * Redirect the request to the IdP authorization endpoint.
     *
     * @stability stable
     */
    AUTHENTICATE = "authenticate"
}
