import * as cdk from '@aws-cdk/core';
import { Construct } from 'constructs';
import { IApplicationListener } from './application-listener';
import { ListenerAction } from './application-listener-action';
import { IApplicationTargetGroup } from './application-target-group';
import { ListenerCondition } from './conditions';
/**
 * Basic properties for defining a rule on a listener.
 *
 * @stability stable
 */
export interface BaseApplicationListenerRuleProps {
    /**
     * Priority of the rule.
     *
     * The rule with the lowest priority will be used for every request.
     *
     * Priorities must be unique.
     *
     * @stability stable
     */
    readonly priority: number;
    /**
     * Target groups to forward requests to.
     *
     * Only one of `action`, `fixedResponse`, `redirectResponse` or `targetGroups` can be specified.
     *
     * Implies a `forward` action.
     *
     * @default - No target groups.
     * @stability stable
     */
    readonly targetGroups?: IApplicationTargetGroup[];
    /**
     * Action to perform when requests are received.
     *
     * Only one of `action`, `fixedResponse`, `redirectResponse` or `targetGroups` can be specified.
     *
     * @default - No action
     * @stability stable
     */
    readonly action?: ListenerAction;
    /**
     * (deprecated) Fixed response to return.
     *
     * Only one of `action`, `fixedResponse`, `redirectResponse` or `targetGroups` can be specified.
     *
     * @default - No fixed response.
     * @deprecated Use `action` instead.
     */
    readonly fixedResponse?: FixedResponse;
    /**
     * (deprecated) Redirect response to return.
     *
     * Only one of `action`, `fixedResponse`, `redirectResponse` or `targetGroups` can be specified.
     *
     * @default - No redirect response.
     * @deprecated Use `action` instead.
     */
    readonly redirectResponse?: RedirectResponse;
    /**
     * Rule applies if matches the conditions.
     *
     * @default - No conditions.
     * @see https://docs.aws.amazon.com/elasticloadbalancing/latest/application/load-balancer-listeners.html
     * @stability stable
     */
    readonly conditions?: ListenerCondition[];
    /**
     * (deprecated) Rule applies if the requested host matches the indicated host.
     *
     * May contain up to three '*' wildcards.
     *
     * @default - No host condition.
     * @see https://docs.aws.amazon.com/elasticloadbalancing/latest/application/load-balancer-listeners.html#host-conditions
     * @deprecated Use `conditions` instead.
     */
    readonly hostHeader?: string;
    /**
     * (deprecated) Rule applies if the requested path matches the given path pattern.
     *
     * @default - No path condition.
     * @see https://docs.aws.amazon.com/elasticloadbalancing/latest/application/load-balancer-listeners.html#path-conditions
     * @deprecated Use `conditions` instead.
     */
    readonly pathPattern?: string;
    /**
     * (deprecated) Rule applies if the requested path matches any of the given patterns.
     *
     * Paths may contain up to three '*' wildcards.
     *
     * @default - No path conditions.
     * @see https://docs.aws.amazon.com/elasticloadbalancing/latest/application/load-balancer-listeners.html#path-conditions
     * @deprecated Use `conditions` instead.
     */
    readonly pathPatterns?: string[];
}
/**
 * Properties for defining a listener rule.
 *
 * @stability stable
 */
export interface ApplicationListenerRuleProps extends BaseApplicationListenerRuleProps {
    /**
     * The listener to attach the rule to.
     *
     * @stability stable
     */
    readonly listener: IApplicationListener;
}
/**
 * The content type for a fixed response.
 *
 * @stability stable
 */
export declare enum ContentType {
    /**
     * @stability stable
     */
    TEXT_PLAIN = "text/plain",
    /**
     * @stability stable
     */
    TEXT_CSS = "text/css",
    /**
     * @stability stable
     */
    TEXT_HTML = "text/html",
    /**
     * @stability stable
     */
    APPLICATION_JAVASCRIPT = "application/javascript",
    /**
     * @stability stable
     */
    APPLICATION_JSON = "application/json"
}
/**
 * A fixed response.
 *
 * @stability stable
 */
export interface FixedResponse {
    /**
     * The HTTP response code (2XX, 4XX or 5XX).
     *
     * @stability stable
     */
    readonly statusCode: string;
    /**
     * The content type.
     *
     * @default text/plain
     * @stability stable
     */
    readonly contentType?: ContentType;
    /**
     * The message.
     *
     * @default no message
     * @stability stable
     */
    readonly messageBody?: string;
}
/**
 * A redirect response.
 *
 * @stability stable
 */
export interface RedirectResponse {
    /**
     * The hostname.
     *
     * This component is not percent-encoded. The hostname can contain #{host}.
     *
     * @default origin host of request
     * @stability stable
     */
    readonly host?: string;
    /**
     * The absolute path, starting with the leading "/".
     *
     * This component is not percent-encoded.
     * The path can contain #{host}, #{path}, and #{port}.
     *
     * @default origin path of request
     * @stability stable
     */
    readonly path?: string;
    /**
     * The port.
     *
     * You can specify a value from 1 to 65535 or #{port}.
     *
     * @default origin port of request
     * @stability stable
     */
    readonly port?: string;
    /**
     * The protocol.
     *
     * You can specify HTTP, HTTPS, or #{protocol}. You can redirect HTTP to HTTP,
     * HTTP to HTTPS, and HTTPS to HTTPS. You cannot redirect HTTPS to HTTP.
     *
     * @default origin protocol of request
     * @stability stable
     */
    readonly protocol?: string;
    /**
     * The query parameters, URL-encoded when necessary, but not percent-encoded.
     *
     * Do not include the leading "?", as it is automatically added.
     * You can specify any of the reserved keywords.
     *
     * @default origin query string of request
     * @stability stable
     */
    readonly query?: string;
    /**
     * The HTTP redirect code (HTTP_301 or HTTP_302).
     *
     * @stability stable
     */
    readonly statusCode: string;
}
/**
 * Define a new listener rule.
 *
 * @stability stable
 */
export declare class ApplicationListenerRule extends cdk.Construct {
    /**
     * The ARN of this rule.
     *
     * @stability stable
     */
    readonly listenerRuleArn: string;
    private readonly conditions;
    private readonly legacyConditions;
    private readonly listener;
    private action?;
    /**
     * @stability stable
     */
    constructor(scope: Construct, id: string, props: ApplicationListenerRuleProps);
    /**
     * (deprecated) Add a non-standard condition to this rule.
     *
     * If the condition conflicts with an already set condition, it will be overwritten by the one you specified.
     *
     * @deprecated use `addCondition` instead.
     */
    setCondition(field: string, values: string[] | undefined): void;
    /**
     * Add a non-standard condition to this rule.
     *
     * @stability stable
     */
    addCondition(condition: ListenerCondition): void;
    /**
     * Configure the action to perform for this rule.
     *
     * @stability stable
     */
    configureAction(action: ListenerAction): void;
    /**
     * (deprecated) Add a TargetGroup to load balance to.
     *
     * @deprecated Use configureAction instead
     */
    addTargetGroup(targetGroup: IApplicationTargetGroup): void;
    /**
     * (deprecated) Add a fixed response.
     *
     * @deprecated Use configureAction instead
     */
    addFixedResponse(fixedResponse: FixedResponse): void;
    /**
     * (deprecated) Add a redirect response.
     *
     * @deprecated Use configureAction instead
     */
    addRedirectResponse(redirectResponse: RedirectResponse): void;
    /**
     * Validate the rule.
     *
     * @stability stable
     */
    protected validate(): string[];
    /**
     * Render the conditions for this rule
     */
    private renderConditions;
}
/**
 * Validate the status code and message body of a fixed response
 *
 * @internal
 */
export declare function validateFixedResponse(fixedResponse: FixedResponse): void;
/**
 * Validate the status code and message body of a redirect response
 *
 * @internal
 */
export declare function validateRedirectResponse(redirectResponse: RedirectResponse): void;
