"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.validateRedirectResponse = exports.validateFixedResponse = exports.ApplicationListenerRule = exports.ContentType = void 0;
const cdk = require("@aws-cdk/core");
const elasticloadbalancingv2_generated_1 = require("../elasticloadbalancingv2.generated");
const application_listener_action_1 = require("./application-listener-action");
/**
 * The content type for a fixed response.
 *
 * @stability stable
 */
var ContentType;
(function (ContentType) {
    ContentType["TEXT_PLAIN"] = "text/plain";
    ContentType["TEXT_CSS"] = "text/css";
    ContentType["TEXT_HTML"] = "text/html";
    ContentType["APPLICATION_JAVASCRIPT"] = "application/javascript";
    ContentType["APPLICATION_JSON"] = "application/json";
})(ContentType = exports.ContentType || (exports.ContentType = {}));
/**
 * Define a new listener rule.
 *
 * @stability stable
 */
class ApplicationListenerRule extends cdk.Construct {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        super(scope, id);
        this.legacyConditions = {};
        this.conditions = props.conditions || [];
        const hasPathPatterns = props.pathPatterns || props.pathPattern;
        if (this.conditions.length === 0 && !props.hostHeader && !hasPathPatterns) {
            throw new Error('At least one of \'conditions\', \'hostHeader\', \'pathPattern\' or \'pathPatterns\' is required when defining a load balancing rule.');
        }
        const possibleActions = ['action', 'targetGroups', 'fixedResponse', 'redirectResponse'];
        const providedActions = possibleActions.filter(action => props[action] !== undefined);
        if (providedActions.length > 1) {
            throw new Error(`'${providedActions}' specified together, specify only one`);
        }
        if (props.priority <= 0) {
            throw new Error('Priority must have value greater than or equal to 1');
        }
        this.listener = props.listener;
        const resource = new elasticloadbalancingv2_generated_1.CfnListenerRule(this, 'Resource', {
            listenerArn: props.listener.listenerArn,
            priority: props.priority,
            conditions: cdk.Lazy.any({ produce: () => this.renderConditions() }),
            actions: cdk.Lazy.any({ produce: () => this.action ? this.action.renderActions() : [] }),
        });
        if (props.hostHeader) {
            this.setCondition('host-header', [props.hostHeader]);
        }
        if (hasPathPatterns) {
            if (props.pathPattern && props.pathPatterns) {
                throw new Error('Both `pathPatterns` and `pathPattern` are specified, specify only one');
            }
            const pathPattern = props.pathPattern ? [props.pathPattern] : props.pathPatterns;
            this.setCondition('path-pattern', pathPattern);
        }
        if (props.action) {
            this.configureAction(props.action);
        }
        (props.targetGroups || []).forEach(this.addTargetGroup.bind(this));
        if (props.fixedResponse) {
            this.addFixedResponse(props.fixedResponse);
        }
        else if (props.redirectResponse) {
            this.addRedirectResponse(props.redirectResponse);
        }
        this.listenerRuleArn = resource.ref;
    }
    /**
     * (deprecated) Add a non-standard condition to this rule.
     *
     * If the condition conflicts with an already set condition, it will be overwritten by the one you specified.
     *
     * @deprecated use `addCondition` instead.
     */
    setCondition(field, values) {
        if (values === undefined) {
            delete this.legacyConditions[field];
            return;
        }
        this.legacyConditions[field] = values;
    }
    /**
     * Add a non-standard condition to this rule.
     *
     * @stability stable
     */
    addCondition(condition) {
        this.conditions.push(condition);
    }
    /**
     * Configure the action to perform for this rule.
     *
     * @stability stable
     */
    configureAction(action) {
        // It might make sense to 'throw' here.
        //
        // However, programs may already exist out there which configured an action twice,
        // in which case the second action accidentally overwrite the initial action, and in some
        // way ended up with a program that did what the author intended. If we were to add throw now,
        // the previously working program would be broken.
        //
        // Instead, signal this through a warning.
        // @deprecate: upon the next major version bump, replace this with a `throw`
        if (this.action) {
            cdk.Annotations.of(this).addWarning('An Action already existed on this ListenerRule and was replaced. Configure exactly one default Action.');
        }
        action.bind(this, this.listener, this);
        this.action = action;
    }
    /**
     * (deprecated) Add a TargetGroup to load balance to.
     *
     * @deprecated Use configureAction instead
     */
    addTargetGroup(targetGroup) {
        this.configureAction(application_listener_action_1.ListenerAction.forward([targetGroup]));
    }
    /**
     * (deprecated) Add a fixed response.
     *
     * @deprecated Use configureAction instead
     */
    addFixedResponse(fixedResponse) {
        validateFixedResponse(fixedResponse);
        this.configureAction(application_listener_action_1.ListenerAction.fixedResponse(cdk.Token.asNumber(fixedResponse.statusCode), {
            contentType: fixedResponse.contentType,
            messageBody: fixedResponse.messageBody,
        }));
    }
    /**
     * (deprecated) Add a redirect response.
     *
     * @deprecated Use configureAction instead
     */
    addRedirectResponse(redirectResponse) {
        validateRedirectResponse(redirectResponse);
        this.configureAction(application_listener_action_1.ListenerAction.redirect({
            host: redirectResponse.host,
            path: redirectResponse.path,
            permanent: redirectResponse.statusCode === 'HTTP_301',
            port: redirectResponse.port,
            protocol: redirectResponse.protocol,
            query: redirectResponse.query,
        }));
    }
    /**
     * Validate the rule.
     *
     * @stability stable
     */
    validate() {
        if (this.action === undefined) {
            return ['Listener rule needs at least one action'];
        }
        const legacyConditionFields = Object.keys(this.legacyConditions);
        if (legacyConditionFields.length === 0 && this.conditions.length === 0) {
            return ['Listener rule needs at least one condition'];
        }
        return [];
    }
    /**
     * Render the conditions for this rule
     */
    renderConditions() {
        const legacyConditions = Object.entries(this.legacyConditions).map(([field, values]) => {
            return { field, values };
        });
        const conditions = this.conditions.map(condition => condition.renderRawCondition());
        return [
            ...legacyConditions,
            ...conditions,
        ];
    }
}
exports.ApplicationListenerRule = ApplicationListenerRule;
/**
 * Validate the status code and message body of a fixed response
 *
 * @internal
 */
function validateFixedResponse(fixedResponse) {
    if (fixedResponse.statusCode && !/^(2|4|5)\d\d$/.test(fixedResponse.statusCode)) {
        throw new Error('`statusCode` must be 2XX, 4XX or 5XX.');
    }
    if (fixedResponse.messageBody && fixedResponse.messageBody.length > 1024) {
        throw new Error('`messageBody` cannot have more than 1024 characters.');
    }
}
exports.validateFixedResponse = validateFixedResponse;
/**
 * Validate the status code and message body of a redirect response
 *
 * @internal
 */
function validateRedirectResponse(redirectResponse) {
    if (redirectResponse.protocol && !/^(HTTPS?|#\{protocol\})$/i.test(redirectResponse.protocol)) {
        throw new Error('`protocol` must be HTTP, HTTPS, or #{protocol}.');
    }
    if (!redirectResponse.statusCode || !/^HTTP_30[12]$/.test(redirectResponse.statusCode)) {
        throw new Error('`statusCode` must be HTTP_301 or HTTP_302.');
    }
}
exports.validateRedirectResponse = validateRedirectResponse;
//# sourceMappingURL=data:application/json;base64,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