"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.NetworkListenerAction = void 0;
/**
 * What to do when a client makes a request to a listener.
 *
 * Some actions can be combined with other ones (specifically,
 * you can perform authentication before serving the request).
 *
 * Multiple actions form a linked chain; the chain must always terminate in a
 * *(weighted)forward*, *fixedResponse* or *redirect* action.
 *
 * If an action supports chaining, the next action can be indicated
 * by passing it in the `next` property.
 *
 * @stability stable
 */
class NetworkListenerAction {
    /**
     * Create an instance of NetworkListenerAction.
     *
     * The default class should be good enough for most cases and
     * should be created by using one of the static factory functions,
     * but allow overriding to make sure we allow flexibility for the future.
     *
     * @stability stable
     */
    constructor(actionJson, next) {
        this.actionJson = actionJson;
        this.next = next;
    }
    /**
     * Forward to one or more Target Groups.
     *
     * @stability stable
     */
    static forward(targetGroups, options = {}) {
        if (targetGroups.length === 0) {
            throw new Error('Need at least one targetGroup in a NetworkListenerAction.forward()');
        }
        if (targetGroups.length === 1 && options.stickinessDuration === undefined) {
            // Render a "simple" action for backwards compatibility with old templates
            return new TargetGroupListenerAction(targetGroups, {
                type: 'forward',
                targetGroupArn: targetGroups[0].targetGroupArn,
            });
        }
        return new TargetGroupListenerAction(targetGroups, {
            type: 'forward',
            forwardConfig: {
                targetGroups: targetGroups.map(g => ({ targetGroupArn: g.targetGroupArn })),
                targetGroupStickinessConfig: options.stickinessDuration ? {
                    durationSeconds: options.stickinessDuration.toSeconds(),
                    enabled: true,
                } : undefined,
            },
        });
    }
    /**
     * Forward to one or more Target Groups which are weighted differently.
     *
     * @stability stable
     */
    static weightedForward(targetGroups, options = {}) {
        if (targetGroups.length === 0) {
            throw new Error('Need at least one targetGroup in a NetworkListenerAction.weightedForward()');
        }
        return new TargetGroupListenerAction(targetGroups.map(g => g.targetGroup), {
            type: 'forward',
            forwardConfig: {
                targetGroups: targetGroups.map(g => ({ targetGroupArn: g.targetGroup.targetGroupArn, weight: g.weight })),
                targetGroupStickinessConfig: options.stickinessDuration ? {
                    durationSeconds: options.stickinessDuration.toSeconds(),
                    enabled: true,
                } : undefined,
            },
        });
    }
    /**
     * Render the actions in this chain.
     *
     * @stability stable
     */
    renderActions() {
        var _a, _b;
        return this.renumber([this.actionJson, ...(_b = (_a = this.next) === null || _a === void 0 ? void 0 : _a.renderActions()) !== null && _b !== void 0 ? _b : []]);
    }
    /**
     * Called when the action is being used in a listener.
     *
     * @stability stable
     */
    bind(scope, listener) {
        // Empty on purpose
        Array.isArray(scope);
        Array.isArray(listener);
    }
    /**
     * Renumber the "order" fields in the actions array.
     *
     * We don't number for 0 or 1 elements, but otherwise number them 1...#actions
     * so ELB knows about the right order.
     *
     * Do this in `NetworkListenerAction` instead of in `Listener` so that we give
     * users the opportunity to override by subclassing and overriding `renderActions`.
     *
     * @stability stable
     */
    renumber(actions) {
        if (actions.length < 2) {
            return actions;
        }
        return actions.map((action, i) => ({ ...action, order: i + 1 }));
    }
}
exports.NetworkListenerAction = NetworkListenerAction;
/**
 * Listener Action that calls "registerListener" on TargetGroups
 */
class TargetGroupListenerAction extends NetworkListenerAction {
    constructor(targetGroups, actionJson) {
        super(actionJson);
        this.targetGroups = targetGroups;
    }
    bind(_scope, listener) {
        for (const tg of this.targetGroups) {
            tg.registerListener(listener);
        }
    }
}
//# sourceMappingURL=data:application/json;base64,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