"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.NetworkListener = void 0;
const cxschema = require("@aws-cdk/cloud-assembly-schema");
const core_1 = require("@aws-cdk/core");
const base_listener_1 = require("../shared/base-listener");
const enums_1 = require("../shared/enums");
const util_1 = require("../shared/util");
const network_listener_action_1 = require("./network-listener-action");
const network_target_group_1 = require("./network-target-group");
/**
 * Define a Network Listener.
 *
 * @stability stable
 * @resource AWS::ElasticLoadBalancingV2::Listener
 */
class NetworkListener extends base_listener_1.BaseListener {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        const certs = props.certificates || [];
        const proto = props.protocol || (certs.length > 0 ? enums_1.Protocol.TLS : enums_1.Protocol.TCP);
        util_1.validateNetworkProtocol(proto);
        if (proto === enums_1.Protocol.TLS && certs.filter(v => v != null).length === 0) {
            throw new Error('When the protocol is set to TLS, you must specify certificates');
        }
        if (proto !== enums_1.Protocol.TLS && certs.length > 0) {
            throw new Error('Protocol must be TLS when certificates have been specified');
        }
        super(scope, id, {
            loadBalancerArn: props.loadBalancer.loadBalancerArn,
            protocol: proto,
            port: props.port,
            sslPolicy: props.sslPolicy,
            certificates: props.certificates,
        });
        this.loadBalancer = props.loadBalancer;
        this.protocol = proto;
        if (props.defaultAction && props.defaultTargetGroups) {
            throw new Error('Specify at most one of \'defaultAction\' and \'defaultTargetGroups\'');
        }
        if (props.defaultAction) {
            this.setDefaultAction(props.defaultAction);
        }
        if (props.defaultTargetGroups) {
            this.setDefaultAction(network_listener_action_1.NetworkListenerAction.forward(props.defaultTargetGroups));
        }
    }
    /**
     * Looks up a network listener.
     *
     * @stability stable
     */
    static fromLookup(scope, id, options) {
        let listenerProtocol;
        if (options.listenerProtocol) {
            util_1.validateNetworkProtocol(options.listenerProtocol);
            switch (options.listenerProtocol) {
                case enums_1.Protocol.TCP:
                    listenerProtocol = cxschema.LoadBalancerListenerProtocol.TCP;
                    break;
                case enums_1.Protocol.UDP:
                    listenerProtocol = cxschema.LoadBalancerListenerProtocol.UDP;
                    break;
                case enums_1.Protocol.TCP_UDP:
                    listenerProtocol = cxschema.LoadBalancerListenerProtocol.TCP_UDP;
                    break;
                case enums_1.Protocol.TLS:
                    listenerProtocol = cxschema.LoadBalancerListenerProtocol.TLS;
                    break;
            }
        }
        const props = base_listener_1.BaseListener._queryContextProvider(scope, {
            userOptions: options,
            listenerProtocol: listenerProtocol,
            loadBalancerType: cxschema.LoadBalancerType.NETWORK,
        });
        class LookedUp extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.listenerArn = props.listenerArn;
            }
        }
        return new LookedUp(scope, id);
    }
    /**
     * Import an existing listener.
     *
     * @stability stable
     */
    static fromNetworkListenerArn(scope, id, networkListenerArn) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.listenerArn = networkListenerArn;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Load balance incoming requests to the given target groups.
     *
     * All target groups will be load balanced to with equal weight and without
     * stickiness. For a more complex configuration than that, use `addAction()`.
     *
     * @stability stable
     */
    addTargetGroups(_id, ...targetGroups) {
        this.setDefaultAction(network_listener_action_1.NetworkListenerAction.forward(targetGroups));
    }
    /**
     * Perform the given Action on incoming requests.
     *
     * This allows full control of the default Action of the load balancer,
     * including weighted forwarding. See the `NetworkListenerAction` class for
     * all options.
     *
     * @stability stable
     */
    addAction(_id, props) {
        this.setDefaultAction(props.action);
    }
    /**
     * Load balance incoming requests to the given load balancing targets.
     *
     * This method implicitly creates a NetworkTargetGroup for the targets
     * involved, and a 'forward' action to route traffic to the given TargetGroup.
     *
     * If you want more control over the precise setup, create the TargetGroup
     * and use `addAction` yourself.
     *
     * It's possible to add conditions to the targets added in this way. At least
     * one set of targets must be added without conditions.
     *
     * @returns The newly created target group
     * @stability stable
     */
    addTargets(id, props) {
        var _a;
        if (!this.loadBalancer.vpc) {
            // eslint-disable-next-line max-len
            throw new Error('Can only call addTargets() when using a constructed Load Balancer or imported Load Balancer with specified VPC; construct a new TargetGroup and use addTargetGroup');
        }
        const group = new network_target_group_1.NetworkTargetGroup(this, id + 'Group', {
            deregistrationDelay: props.deregistrationDelay,
            healthCheck: props.healthCheck,
            port: props.port,
            protocol: (_a = props.protocol) !== null && _a !== void 0 ? _a : this.protocol,
            proxyProtocolV2: props.proxyProtocolV2,
            targetGroupName: props.targetGroupName,
            targets: props.targets,
            vpc: this.loadBalancer.vpc,
        });
        this.addTargetGroups(id, group);
        return group;
    }
    /**
     * Wrapper for _setDefaultAction which does a type-safe bind
     */
    setDefaultAction(action) {
        action.bind(this, this);
        this._setDefaultAction(action);
    }
}
exports.NetworkListener = NetworkListener;
//# sourceMappingURL=data:application/json;base64,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