"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.NetworkTargetGroup = void 0;
const cloudwatch = require("@aws-cdk/aws-cloudwatch");
const cdk = require("@aws-cdk/core");
const base_target_group_1 = require("../shared/base-target-group");
const enums_1 = require("../shared/enums");
const imported_1 = require("../shared/imported");
const util_1 = require("../shared/util");
/**
 * Define a Network Target Group.
 *
 * @stability stable
 */
class NetworkTargetGroup extends base_target_group_1.TargetGroupBase {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        const proto = props.protocol || enums_1.Protocol.TCP;
        util_1.validateNetworkProtocol(proto);
        super(scope, id, props, {
            protocol: proto,
            port: props.port,
        });
        this.listeners = [];
        if (props.proxyProtocolV2 != null) {
            this.setAttribute('proxy_protocol_v2.enabled', props.proxyProtocolV2 ? 'true' : 'false');
        }
        this.addTarget(...(props.targets || []));
    }
    /**
     * Import an existing target group.
     *
     * @stability stable
     */
    static fromTargetGroupAttributes(scope, id, attrs) {
        return new ImportedNetworkTargetGroup(scope, id, attrs);
    }
    /**
     * (deprecated) Import an existing listener.
     *
     * @deprecated Use `fromTargetGroupAttributes` instead
     */
    static import(scope, id, props) {
        return NetworkTargetGroup.fromTargetGroupAttributes(scope, id, props);
    }
    /**
     * Add a load balancing target to this target group.
     *
     * @stability stable
     */
    addTarget(...targets) {
        for (const target of targets) {
            const result = target.attachToNetworkTargetGroup(this);
            this.addLoadBalancerTarget(result);
        }
    }
    /**
     * Register a listener that is load balancing to this target group.
     *
     * Don't call this directly. It will be called by listeners.
     *
     * @stability stable
     */
    registerListener(listener) {
        this.loadBalancerAttachedDependencies.add(listener);
        this.listeners.push(listener);
    }
    /**
     * The number of targets that are considered healthy.
     *
     * @default Average over 5 minutes
     * @stability stable
     */
    metricHealthyHostCount(props) {
        return this.metric('HealthyHostCount', {
            statistic: 'Average',
            ...props,
        });
    }
    /**
     * The number of targets that are considered unhealthy.
     *
     * @default Average over 5 minutes
     * @stability stable
     */
    metricUnHealthyHostCount(props) {
        return this.metric('UnHealthyHostCount', {
            statistic: 'Average',
            ...props,
        });
    }
    /**
     * Full name of first load balancer.
     *
     * @stability stable
     */
    get firstLoadBalancerFullName() {
        if (this.listeners.length === 0) {
            throw new Error('The TargetGroup needs to be attached to a LoadBalancer before you can call this method');
        }
        return base_target_group_1.loadBalancerNameFromListenerArn(this.listeners[0].listenerArn);
    }
    /**
     * Validate the current construct.
     *
     * This method can be implemented by derived constructs in order to perform
     * validation logic. It is called on all constructs before synthesis.
     *
     * @stability stable
     */
    validate() {
        const ret = super.validate();
        const healthCheck = this.healthCheck || {};
        const allowedIntervals = [10, 30];
        if (healthCheck.interval) {
            const seconds = healthCheck.interval.toSeconds();
            if (!cdk.Token.isUnresolved(seconds) && !allowedIntervals.includes(seconds)) {
                ret.push(`Health check interval '${seconds}' not supported. Must be one of the following values '${allowedIntervals.join(',')}'.`);
            }
        }
        if (healthCheck.healthyThresholdCount) {
            const thresholdCount = healthCheck.healthyThresholdCount;
            if (thresholdCount < 2 || thresholdCount > 10) {
                ret.push(`Healthy Threshold Count '${thresholdCount}' not supported. Must be a number between 2 and 10.`);
            }
        }
        if (healthCheck.unhealthyThresholdCount) {
            const thresholdCount = healthCheck.unhealthyThresholdCount;
            if (thresholdCount < 2 || thresholdCount > 10) {
                ret.push(`Unhealthy Threshold Count '${thresholdCount}' not supported. Must be a number between 2 and 10.`);
            }
        }
        if (healthCheck.healthyThresholdCount && healthCheck.unhealthyThresholdCount &&
            healthCheck.healthyThresholdCount !== healthCheck.unhealthyThresholdCount) {
            ret.push([
                `Healthy and Unhealthy Threshold Counts must be the same: ${healthCheck.healthyThresholdCount}`,
                `is not equal to ${healthCheck.unhealthyThresholdCount}.`,
            ].join(' '));
        }
        if (!healthCheck.protocol) {
            return ret;
        }
        if (!NLB_HEALTH_CHECK_PROTOCOLS.includes(healthCheck.protocol)) {
            ret.push(`Health check protocol '${healthCheck.protocol}' is not supported. Must be one of [${NLB_HEALTH_CHECK_PROTOCOLS.join(', ')}]`);
        }
        if (healthCheck.path && !NLB_PATH_HEALTH_CHECK_PROTOCOLS.includes(healthCheck.protocol)) {
            ret.push([
                `'${healthCheck.protocol}' health checks do not support the path property.`,
                `Must be one of [${NLB_PATH_HEALTH_CHECK_PROTOCOLS.join(', ')}]`,
            ].join(' '));
        }
        if (healthCheck.timeout && healthCheck.timeout.toSeconds() !== NLB_HEALTH_CHECK_TIMEOUTS[healthCheck.protocol]) {
            ret.push([
                'Custom health check timeouts are not supported for Network Load Balancer health checks.',
                `Expected ${NLB_HEALTH_CHECK_TIMEOUTS[healthCheck.protocol]} seconds for ${healthCheck.protocol}, got ${healthCheck.timeout.toSeconds()}`,
            ].join(' '));
        }
        return ret;
    }
    metric(metricName, props) {
        return new cloudwatch.Metric({
            namespace: 'AWS/NetworkELB',
            metricName,
            dimensions: { LoadBalancer: this.firstLoadBalancerFullName, TargetGroup: this.targetGroupFullName },
            ...props,
        }).attachTo(this);
    }
}
exports.NetworkTargetGroup = NetworkTargetGroup;
/**
 * An imported network target group
 */
class ImportedNetworkTargetGroup extends imported_1.ImportedTargetGroupBase {
    registerListener(_listener) {
        // Nothing to do, we know nothing of our members
    }
    addTarget(...targets) {
        for (const target of targets) {
            const result = target.attachToNetworkTargetGroup(this);
            if (result.targetJson !== undefined) {
                throw new Error('Cannot add a non-self registering target to an imported TargetGroup. Create a new TargetGroup instead.');
            }
        }
    }
}
const NLB_HEALTH_CHECK_PROTOCOLS = [enums_1.Protocol.HTTP, enums_1.Protocol.HTTPS, enums_1.Protocol.TCP];
const NLB_PATH_HEALTH_CHECK_PROTOCOLS = [enums_1.Protocol.HTTP, enums_1.Protocol.HTTPS];
const NLB_HEALTH_CHECK_TIMEOUTS = {
    [enums_1.Protocol.HTTP]: 6,
    [enums_1.Protocol.HTTPS]: 10,
    [enums_1.Protocol.TCP]: 10,
};
//# sourceMappingURL=data:application/json;base64,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