"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.BaseListener = void 0;
const cxschema = require("@aws-cdk/cloud-assembly-schema");
const core_1 = require("@aws-cdk/core");
const elasticloadbalancingv2_generated_1 = require("../elasticloadbalancingv2.generated");
const util_1 = require("./util");
/**
 * Base class for listeners.
 *
 * @stability stable
 */
class BaseListener extends core_1.Resource {
    /**
     * @stability stable
     */
    constructor(scope, id, additionalProps) {
        super(scope, id);
        const resource = new elasticloadbalancingv2_generated_1.CfnListener(this, 'Resource', {
            ...additionalProps,
            defaultActions: core_1.Lazy.any({ produce: () => this.defaultAction ? this.defaultAction.renderActions() : [] }),
        });
        this.listenerArn = resource.ref;
    }
    /**
     * Queries the load balancer listener context provider for load balancer
     * listener info.
     * @internal
     */
    static _queryContextProvider(scope, options) {
        var _a;
        if (core_1.Token.isUnresolved(options.userOptions.loadBalancerArn)
            || Object.values((_a = options.userOptions.loadBalancerTags) !== null && _a !== void 0 ? _a : {}).some(core_1.Token.isUnresolved)
            || core_1.Token.isUnresolved(options.userOptions.listenerPort)) {
            throw new Error('All arguments to look up a load balancer listener must be concrete (no Tokens)');
        }
        let cxschemaTags;
        if (options.userOptions.loadBalancerTags) {
            cxschemaTags = util_1.mapTagMapToCxschema(options.userOptions.loadBalancerTags);
        }
        const props = core_1.ContextProvider.getValue(scope, {
            provider: cxschema.ContextProvider.LOAD_BALANCER_LISTENER_PROVIDER,
            props: {
                listenerArn: options.listenerArn,
                listenerPort: options.userOptions.listenerPort,
                listenerProtocol: options.listenerProtocol,
                loadBalancerArn: options.userOptions.loadBalancerArn,
                loadBalancerTags: cxschemaTags,
                loadBalancerType: options.loadBalancerType,
            },
            dummyValue: {
                listenerArn: `arn:aws:elasticloadbalancing:us-west-2:123456789012:listener/${options.loadBalancerType}/my-load-balancer/50dc6c495c0c9188/f2f7dc8efc522ab2`,
                listenerPort: 80,
                securityGroupIds: ['sg-123456789012'],
            },
        }).value;
        return props;
    }
    /**
     * Validate this listener.
     *
     * @stability stable
     */
    validate() {
        if (!this.defaultAction) {
            return ['Listener needs at least one default action or target group (call addTargetGroups or addAction)'];
        }
        return [];
    }
    /**
     * Configure the default action
     *
     * @internal
     */
    _setDefaultAction(action) {
        // It might make sense to 'throw' here.
        //
        // However, programs may already exist out there which configured an action twice,
        // in which case the second action accidentally overwrite the initial action, and in some
        // way ended up with a program that did what the author intended. If we were to add throw now,
        // the previously working program would be broken.
        //
        // Instead, signal this through a warning.
        // @deprecate: upon the next major version bump, replace this with a `throw`
        if (this.defaultAction) {
            core_1.Annotations.of(this).addWarning('A default Action already existed on this Listener and was replaced. Configure exactly one default Action.');
        }
        this.defaultAction = action;
    }
}
exports.BaseListener = BaseListener;
//# sourceMappingURL=data:application/json;base64,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