"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.BaseLoadBalancer = void 0;
const ec2 = require("@aws-cdk/aws-ec2");
const iam = require("@aws-cdk/aws-iam");
const cxschema = require("@aws-cdk/cloud-assembly-schema");
const core_1 = require("@aws-cdk/core");
const cxapi = require("@aws-cdk/cx-api");
const region_info_1 = require("@aws-cdk/region-info");
const elasticloadbalancingv2_generated_1 = require("../elasticloadbalancingv2.generated");
const util_1 = require("./util");
/**
 * Base class for both Application and Network Load Balancers.
 *
 * @stability stable
 */
class BaseLoadBalancer extends core_1.Resource {
    /**
     * @stability stable
     */
    constructor(scope, id, baseProps, additionalProps) {
        super(scope, id, {
            physicalName: baseProps.loadBalancerName,
        });
        /**
         * Attributes set on this load balancer
         */
        this.attributes = {};
        const internetFacing = util_1.ifUndefined(baseProps.internetFacing, false);
        const vpcSubnets = util_1.ifUndefined(baseProps.vpcSubnets, (internetFacing ? { subnetType: ec2.SubnetType.PUBLIC } : {}));
        const { subnetIds, internetConnectivityEstablished } = baseProps.vpc.selectSubnets(vpcSubnets);
        this.vpc = baseProps.vpc;
        const resource = new elasticloadbalancingv2_generated_1.CfnLoadBalancer(this, 'Resource', {
            name: this.physicalName,
            subnets: subnetIds,
            scheme: internetFacing ? 'internet-facing' : 'internal',
            loadBalancerAttributes: core_1.Lazy.any({ produce: () => util_1.renderAttributes(this.attributes) }, { omitEmptyArray: true }),
            ...additionalProps,
        });
        if (internetFacing) {
            resource.node.addDependency(internetConnectivityEstablished);
        }
        this.setAttribute('deletion_protection.enabled', baseProps.deletionProtection ? 'true' : 'false');
        this.loadBalancerCanonicalHostedZoneId = resource.attrCanonicalHostedZoneId;
        this.loadBalancerDnsName = resource.attrDnsName;
        this.loadBalancerFullName = resource.attrLoadBalancerFullName;
        this.loadBalancerName = resource.attrLoadBalancerName;
        this.loadBalancerArn = resource.ref;
        this.loadBalancerSecurityGroups = resource.attrSecurityGroups;
    }
    /**
     * Queries the load balancer context provider for load balancer info.
     * @internal
     */
    static _queryContextProvider(scope, options) {
        var _a;
        if (core_1.Token.isUnresolved(options.userOptions.loadBalancerArn)
            || Object.values((_a = options.userOptions.loadBalancerTags) !== null && _a !== void 0 ? _a : {}).some(core_1.Token.isUnresolved)) {
            throw new Error('All arguments to look up a load balancer must be concrete (no Tokens)');
        }
        let cxschemaTags;
        if (options.userOptions.loadBalancerTags) {
            cxschemaTags = util_1.mapTagMapToCxschema(options.userOptions.loadBalancerTags);
        }
        const props = core_1.ContextProvider.getValue(scope, {
            provider: cxschema.ContextProvider.LOAD_BALANCER_PROVIDER,
            props: {
                loadBalancerArn: options.userOptions.loadBalancerArn,
                loadBalancerTags: cxschemaTags,
                loadBalancerType: options.loadBalancerType,
            },
            dummyValue: {
                ipAddressType: cxapi.LoadBalancerIpAddressType.DUAL_STACK,
                loadBalancerArn: `arn:aws:elasticloadbalancing:us-west-2:123456789012:loadbalancer/${options.loadBalancerType}/my-load-balancer/50dc6c495c0c9188`,
                loadBalancerCanonicalHostedZoneId: 'Z3DZXE0EXAMPLE',
                loadBalancerDnsName: 'my-load-balancer-1234567890.us-west-2.elb.amazonaws.com',
                securityGroupIds: ['sg-1234'],
                vpcId: 'vpc-12345',
            },
        }).value;
        return props;
    }
    /**
     * Enable access logging for this load balancer.
     *
     * A region must be specified on the stack containing the load balancer; you cannot enable logging on
     * environment-agnostic stacks. See https://docs.aws.amazon.com/cdk/latest/guide/environments.html
     *
     * @stability stable
     */
    logAccessLogs(bucket, prefix) {
        this.setAttribute('access_logs.s3.enabled', 'true');
        this.setAttribute('access_logs.s3.bucket', bucket.bucketName.toString());
        this.setAttribute('access_logs.s3.prefix', prefix);
        const region = core_1.Stack.of(this).region;
        if (core_1.Token.isUnresolved(region)) {
            throw new Error('Region is required to enable ELBv2 access logging');
        }
        const account = region_info_1.RegionInfo.get(region).elbv2Account;
        if (!account) {
            throw new Error(`Cannot enable access logging; don't know ELBv2 account for region ${region}`);
        }
        prefix = prefix || '';
        bucket.grantPut(new iam.AccountPrincipal(account), `${(prefix ? prefix + '/' : '')}AWSLogs/${core_1.Stack.of(this).account}/*`);
        // make sure the bucket's policy is created before the ALB (see https://github.com/aws/aws-cdk/issues/1633)
        this.node.addDependency(bucket);
    }
    /**
     * Set a non-standard attribute on the load balancer.
     *
     * @see https://docs.aws.amazon.com/elasticloadbalancing/latest/application/application-load-balancers.html#load-balancer-attributes
     * @stability stable
     */
    setAttribute(key, value) {
        this.attributes[key] = value;
    }
    /**
     * Remove an attribute from the load balancer.
     *
     * @stability stable
     */
    removeAttribute(key) {
        this.setAttribute(key, undefined);
    }
}
exports.BaseLoadBalancer = BaseLoadBalancer;
//# sourceMappingURL=data:application/json;base64,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