"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.loadBalancerNameFromListenerArn = exports.TargetGroupBase = void 0;
const cdk = require("@aws-cdk/core");
const elasticloadbalancingv2_generated_1 = require("../elasticloadbalancingv2.generated");
const enums_1 = require("./enums");
const util_1 = require("./util");
/**
 * Define the target of a load balancer.
 *
 * @stability stable
 */
class TargetGroupBase extends cdk.Construct {
    /**
     * @stability stable
     */
    constructor(scope, id, baseProps, additionalProps) {
        super(scope, id);
        /**
         * Configurable dependable with all resources that lead to load balancer attachment.
         *
         * @stability stable
         */
        this.loadBalancerAttachedDependencies = new cdk.ConcreteDependable();
        /**
         * Attributes of this target group
         */
        this.attributes = {};
        /**
         * The JSON objects returned by the directly registered members of this target group
         */
        this.targetsJson = new Array();
        if (baseProps.deregistrationDelay !== undefined) {
            this.setAttribute('deregistration_delay.timeout_seconds', baseProps.deregistrationDelay.toSeconds().toString());
        }
        this.healthCheck = baseProps.healthCheck || {};
        this.vpc = baseProps.vpc;
        this.targetType = baseProps.targetType;
        this.resource = new elasticloadbalancingv2_generated_1.CfnTargetGroup(this, 'Resource', {
            name: baseProps.targetGroupName,
            targetGroupAttributes: cdk.Lazy.any({ produce: () => util_1.renderAttributes(this.attributes) }, { omitEmptyArray: true }),
            targetType: cdk.Lazy.string({ produce: () => this.targetType }),
            targets: cdk.Lazy.any({ produce: () => this.targetsJson }, { omitEmptyArray: true }),
            vpcId: cdk.Lazy.string({ produce: () => this.vpc && this.targetType !== enums_1.TargetType.LAMBDA ? this.vpc.vpcId : undefined }),
            // HEALTH CHECK
            healthCheckEnabled: cdk.Lazy.any({ produce: () => this.healthCheck && this.healthCheck.enabled }),
            healthCheckIntervalSeconds: cdk.Lazy.number({
                produce: () => this.healthCheck && this.healthCheck.interval && this.healthCheck.interval.toSeconds(),
            }),
            healthCheckPath: cdk.Lazy.string({ produce: () => this.healthCheck && this.healthCheck.path }),
            healthCheckPort: cdk.Lazy.string({ produce: () => this.healthCheck && this.healthCheck.port }),
            healthCheckProtocol: cdk.Lazy.string({ produce: () => this.healthCheck && this.healthCheck.protocol }),
            healthCheckTimeoutSeconds: cdk.Lazy.number({
                produce: () => this.healthCheck && this.healthCheck.timeout && this.healthCheck.timeout.toSeconds(),
            }),
            healthyThresholdCount: cdk.Lazy.number({ produce: () => this.healthCheck && this.healthCheck.healthyThresholdCount }),
            unhealthyThresholdCount: cdk.Lazy.number({ produce: () => this.healthCheck && this.healthCheck.unhealthyThresholdCount }),
            matcher: cdk.Lazy.any({
                produce: () => this.healthCheck && this.healthCheck.healthyHttpCodes !== undefined ? {
                    httpCode: this.healthCheck.healthyHttpCodes,
                } : undefined,
            }),
            ...additionalProps,
        });
        this.targetGroupLoadBalancerArns = this.resource.attrLoadBalancerArns;
        this.targetGroupArn = this.resource.ref;
        this.targetGroupFullName = this.resource.attrTargetGroupFullName;
        this.loadBalancerArns = this.resource.attrLoadBalancerArns.toString();
        this.targetGroupName = this.resource.attrTargetGroupName;
        this.defaultPort = additionalProps.port;
    }
    /**
     * List of constructs that need to be depended on to ensure the TargetGroup is associated to a load balancer.
     *
     * @stability stable
     */
    get loadBalancerAttached() {
        return this.loadBalancerAttachedDependencies;
    }
    /**
     * Set/replace the target group's health check.
     *
     * @stability stable
     */
    configureHealthCheck(healthCheck) {
        this.healthCheck = healthCheck;
    }
    /**
     * Set a non-standard attribute on the target group.
     *
     * @see https://docs.aws.amazon.com/elasticloadbalancing/latest/application/load-balancer-target-groups.html#target-group-attributes
     * @stability stable
     */
    setAttribute(key, value) {
        this.attributes[key] = value;
    }
    /**
     * Register the given load balancing target as part of this group.
     *
     * @stability stable
     */
    addLoadBalancerTarget(props) {
        if (this.targetType !== undefined && this.targetType !== props.targetType) {
            throw new Error(`Already have a of type '${this.targetType}', adding '${props.targetType}'; make all targets the same type.`);
        }
        this.targetType = props.targetType;
        if (this.targetType === enums_1.TargetType.LAMBDA && this.targetsJson.length >= 1) {
            throw new Error('TargetGroup can only contain one LAMBDA target. Create a new TargetGroup.');
        }
        if (props.targetJson) {
            this.targetsJson.push(props.targetJson);
        }
    }
    /**
     * Validate the current construct.
     *
     * This method can be implemented by derived constructs in order to perform
     * validation logic. It is called on all constructs before synthesis.
     *
     * @stability stable
     */
    validate() {
        const ret = super.validate();
        if (this.targetType === undefined && this.targetsJson.length === 0) {
            cdk.Annotations.of(this).addWarning("When creating an empty TargetGroup, you should specify a 'targetType' (this warning may become an error in the future).");
        }
        if (this.targetType !== enums_1.TargetType.LAMBDA && this.vpc === undefined) {
            ret.push("'vpc' is required for a non-Lambda TargetGroup");
        }
        return ret;
    }
}
exports.TargetGroupBase = TargetGroupBase;
/**
 * Extract the full load balancer name (used for metrics) from the listener ARN:
 *
 * Turns
 *
 *     arn:aws:elasticloadbalancing:us-west-2:123456789012:listener/app/my-load-balancer/50dc6c495c0c9188/f2f7dc8efc522ab2
 *
 * Into
 *
 *     app/my-load-balancer/50dc6c495c0c9188
 */
function loadBalancerNameFromListenerArn(listenerArn) {
    const arnParts = cdk.Fn.split('/', listenerArn);
    return `${cdk.Fn.select(1, arnParts)}/${cdk.Fn.select(2, arnParts)}/${cdk.Fn.select(3, arnParts)}`;
}
exports.loadBalancerNameFromListenerArn = loadBalancerNameFromListenerArn;
//# sourceMappingURL=data:application/json;base64,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