"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.loadBalancerNameFromListenerArn = exports.TargetGroupBase = void 0;
const jsiiDeprecationWarnings = require("../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk = require("@aws-cdk/core");
const elasticloadbalancingv2_generated_1 = require("../elasticloadbalancingv2.generated");
const enums_1 = require("./enums");
const util_1 = require("./util");
// keep this import separate from other imports to reduce chance for merge conflicts with v2-main
// eslint-disable-next-line no-duplicate-imports, import/order
const core_1 = require("@aws-cdk/core");
/**
 * Define the target of a load balancer
 */
class TargetGroupBase extends core_1.Construct {
    constructor(scope, id, baseProps, additionalProps) {
        super(scope, id);
        /**
         * Configurable dependable with all resources that lead to load balancer attachment
         */
        this.loadBalancerAttachedDependencies = new cdk.ConcreteDependable();
        /**
         * Attributes of this target group
         */
        this.attributes = {};
        /**
         * The JSON objects returned by the directly registered members of this target group
         */
        this.targetsJson = new Array();
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_elasticloadbalancingv2_BaseTargetGroupProps(baseProps);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, TargetGroupBase);
            }
            throw error;
        }
        if (baseProps.deregistrationDelay !== undefined) {
            this.setAttribute('deregistration_delay.timeout_seconds', baseProps.deregistrationDelay.toSeconds().toString());
        }
        this.healthCheck = baseProps.healthCheck || {};
        this.vpc = baseProps.vpc;
        this.targetType = baseProps.targetType;
        this.resource = new elasticloadbalancingv2_generated_1.CfnTargetGroup(this, 'Resource', {
            name: baseProps.targetGroupName,
            targetGroupAttributes: cdk.Lazy.any({ produce: () => util_1.renderAttributes(this.attributes) }, { omitEmptyArray: true }),
            targetType: cdk.Lazy.string({ produce: () => this.targetType }),
            targets: cdk.Lazy.any({ produce: () => this.targetsJson }, { omitEmptyArray: true }),
            vpcId: cdk.Lazy.string({ produce: () => this.vpc && this.targetType !== enums_1.TargetType.LAMBDA ? this.vpc.vpcId : undefined }),
            // HEALTH CHECK
            healthCheckEnabled: cdk.Lazy.any({ produce: () => this.healthCheck?.enabled }),
            healthCheckIntervalSeconds: cdk.Lazy.number({
                produce: () => this.healthCheck?.interval?.toSeconds(),
            }),
            healthCheckPath: cdk.Lazy.string({ produce: () => this.healthCheck?.path }),
            healthCheckPort: cdk.Lazy.string({ produce: () => this.healthCheck?.port }),
            healthCheckProtocol: cdk.Lazy.string({ produce: () => this.healthCheck?.protocol }),
            healthCheckTimeoutSeconds: cdk.Lazy.number({
                produce: () => this.healthCheck?.timeout?.toSeconds(),
            }),
            healthyThresholdCount: cdk.Lazy.number({ produce: () => this.healthCheck?.healthyThresholdCount }),
            unhealthyThresholdCount: cdk.Lazy.number({ produce: () => this.healthCheck?.unhealthyThresholdCount }),
            matcher: cdk.Lazy.any({
                produce: () => this.healthCheck?.healthyHttpCodes !== undefined || this.healthCheck?.healthyGrpcCodes !== undefined ? {
                    grpcCode: this.healthCheck.healthyGrpcCodes,
                    httpCode: this.healthCheck.healthyHttpCodes,
                } : undefined,
            }),
            ...additionalProps,
        });
        this.targetGroupLoadBalancerArns = this.resource.attrLoadBalancerArns;
        this.targetGroupArn = this.resource.ref;
        this.targetGroupFullName = this.resource.attrTargetGroupFullName;
        this.loadBalancerArns = this.resource.attrLoadBalancerArns.toString();
        this.targetGroupName = this.resource.attrTargetGroupName;
        this.defaultPort = additionalProps.port;
    }
    /**
     * List of constructs that need to be depended on to ensure the TargetGroup is associated to a load balancer
     */
    get loadBalancerAttached() {
        return this.loadBalancerAttachedDependencies;
    }
    /**
     * Set/replace the target group's health check
     */
    configureHealthCheck(healthCheck) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_elasticloadbalancingv2_HealthCheck(healthCheck);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.configureHealthCheck);
            }
            throw error;
        }
        this.healthCheck = healthCheck;
    }
    /**
     * Set a non-standard attribute on the target group
     *
     * @see https://docs.aws.amazon.com/elasticloadbalancing/latest/application/load-balancer-target-groups.html#target-group-attributes
     */
    setAttribute(key, value) {
        this.attributes[key] = value;
    }
    /**
     * Register the given load balancing target as part of this group
     */
    addLoadBalancerTarget(props) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_elasticloadbalancingv2_LoadBalancerTargetProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addLoadBalancerTarget);
            }
            throw error;
        }
        if (this.targetType !== undefined && this.targetType !== props.targetType) {
            throw new Error(`Already have a of type '${this.targetType}', adding '${props.targetType}'; make all targets the same type.`);
        }
        this.targetType = props.targetType;
        if (this.targetType === enums_1.TargetType.LAMBDA && this.targetsJson.length >= 1) {
            throw new Error('TargetGroup can only contain one LAMBDA target. Create a new TargetGroup.');
        }
        if (props.targetJson) {
            this.targetsJson.push(props.targetJson);
        }
    }
    validate() {
        const ret = super.validate();
        if (this.targetType === undefined && this.targetsJson.length === 0) {
            cdk.Annotations.of(this).addWarning("When creating an empty TargetGroup, you should specify a 'targetType' (this warning may become an error in the future).");
        }
        if (this.targetType !== enums_1.TargetType.LAMBDA && this.vpc === undefined) {
            ret.push("'vpc' is required for a non-Lambda TargetGroup");
        }
        // https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-elasticloadbalancingv2-targetgroup.html#cfn-elasticloadbalancingv2-targetgroup-name
        const targetGroupName = this.resource.name;
        if (!cdk.Token.isUnresolved(targetGroupName) && targetGroupName !== undefined) {
            if (targetGroupName.length > 32) {
                ret.push(`Target group name: "${targetGroupName}" can have a maximum of 32 characters.`);
            }
            if (targetGroupName.startsWith('-') || targetGroupName.endsWith('-')) {
                ret.push(`Target group name: "${targetGroupName}" must not begin or end with a hyphen.`);
            }
            if (!/^[0-9a-z-]+$/i.test(targetGroupName)) {
                ret.push(`Target group name: "${targetGroupName}" must contain only alphanumeric characters or hyphens.`);
            }
        }
        return ret;
    }
}
exports.TargetGroupBase = TargetGroupBase;
_a = JSII_RTTI_SYMBOL_1;
TargetGroupBase[_a] = { fqn: "@aws-cdk/aws-elasticloadbalancingv2.TargetGroupBase", version: "1.181.0" };
/**
 * Extract the full load balancer name (used for metrics) from the listener ARN:
 *
 * Turns
 *
 *     arn:aws:elasticloadbalancing:us-west-2:123456789012:listener/app/my-load-balancer/50dc6c495c0c9188/f2f7dc8efc522ab2
 *
 * Into
 *
 *     app/my-load-balancer/50dc6c495c0c9188
 */
function loadBalancerNameFromListenerArn(listenerArn) {
    const arnParts = cdk.Fn.split('/', listenerArn);
    return `${cdk.Fn.select(1, arnParts)}/${cdk.Fn.select(2, arnParts)}/${cdk.Fn.select(3, arnParts)}`;
}
exports.loadBalancerNameFromListenerArn = loadBalancerNameFromListenerArn;
//# sourceMappingURL=data:application/json;base64,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