#!/usr/bin/env python
# -*- coding: utf-8 -*-

"""
 File name: mod_in_nse.py
 Application: The NewsLookout Web Scraping Application
 Date: 2021-06-01
 Copyright 2021, The NewsLookout Web Scraping Application, Sandeep Singh Sandhu, sandeep.sandhu@gmx.com
 Country: India
 Purpose: Plugin for the National Stock Exchange Data, India
 Save the daily BhavCopy file generated by the exchange at the end of the day


 Notice:
 This software is intended for demonstration and educational purposes only. This software is
 experimental and a work in progress. Under no circumstances should these files be used in
 relation to any critical system(s). Use of these files is at your own risk.

 Before using it for web scraping any website, always consult that website's terms of use.
 Do not use this software to fetch any data from any website that has forbidden use of web
 scraping or similar mechanisms, or violates its terms of use in any other way. The author is
 not liable for such kind of inappropriate use of this software.

 THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED,
 INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE
 FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR
 OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 DEALINGS IN THE SOFTWARE.

"""

##########

# import standard python libraries:
from datetime import datetime
import os
import logging
import re

# import web retrieval and text processing python libraries:
# from bs4 import BeautifulSoup
# import lxml
import zipfile
import requests

# import this project's python libraries:
from base_plugin import basePlugin
from scraper_utils import getPreviousDaysDate, calculateCRC32
from data_structs import Types, NewsArticle

##########

logger = logging.getLogger(__name__)


class mod_in_nse(basePlugin):
    """ Web Scraping plugin: mod_in_nse
    For: National Stock Exchange
    """

    minArticleLengthInChars = 10000
    pluginType = Types.MODULE_DATA_CONTENT  # implies data content fetcher

    mainURL = 'https://www1.nseindia.com/archives/equities/bhavcopy/pr/PR'
    mainURL_suffix = ".zip"

    masterData = {'mod_in_nse_EQUITY_L.csv': 'https://archives.nseindia.com/content/equities/EQUITY_L.csv',
                  'mod_in_nse_SME_EQUITY_L.csv': 'https://archives.nseindia.com/emerge/corporates/content/SME_EQUITY_L.csv',
                  'mod_in_nse_eq_etfseclist.csv': 'https://archives.nseindia.com/content/equities/eq_etfseclist.csv',
                  'mod_in_nse_PREF.csv': 'https://archives.nseindia.com/content/equities/PREF.csv',
                  'mod_in_nse_namechange.csv': 'https://archives.nseindia.com/content/equities/namechange.csv',
                  'mod_in_nse_symbolchange.csv': 'https://archives.nseindia.com/content/equities/symbolchange.csv',
                  'mod_in_nse_WARRANT.csv': 'https://archives.nseindia.com/content/equities/WARRANT.csv',
                  'mod_in_nse_DEBT.csv': 'https://archives.nseindia.com/content/equities/DEBT.csv',
                  'mod_in_nse_Redmn_Def_Web.csv': 'https://archives.nseindia.com/content/debt/Redmn_Def_Web.csv'
                  }
    master_data_dir = None
    masterDataExtractedFlag = False

    pledgesURL1 = 'https://www.nseindia.com/companies-listing/corporate-filings-pledged-data'
    pledgesURL2 = 'https://www.nseindia.com/api/corporate-pledgedata?index=equities&csv=true'
    pledgesURL3 = 'https://www.nseindia.com/companies-listing/corporate-filings-regulation-7-promoter'
    pledgesDataExtractedFlag = False

    listOfURLS = []
    urlUniqueRegexps = ["(^https://www1.nseindia.com/archives/equities/bhavcopy/pr/PR)([0-9]+)(.zip$)"]
    urlMatchPatterns = []
    uRLdata = dict()
    authorRegexps = []
    articleDateRegexps = dict()
    validURLStringsToCheck = []
    invalidURLSubStrings = []
    nonContentURLs = []
    nonContentStrings = []
    allowedDomains = ["www1.nseindia.com", "www.nseindia.com", "archives.nseindia.com"]

    def __init__(self):
        """ Initialize the object
        """
        self.pledgesDataExtractedFlag = False
        self.count_history_to_fetch = 1
        super().__init__()

    def getURLsListForDate(self, runDate, completedURLs):
        """ Retrieve the URLs List For a given Date
        """
        self.listOfURLS = []
        try:
            # set dates for retrieval based on recursion level in configuration file:
            if self.configData['recursion_level'] == 2:
                self.count_history_to_fetch = 10
            elif self.configData['recursion_level'] == 3:
                self.count_history_to_fetch = 31
            elif self.configData['recursion_level'] > 3:
                self.count_history_to_fetch = 3650
            else:
                self.count_history_to_fetch = 1
            # begin with run date:
            businessDate = runDate
            for dayCount in range(self.count_history_to_fetch):
                # decrement dates one by one
                businessDate = getPreviousDaysDate(businessDate)
                urlForDate = self.mainURL + businessDate.strftime("%d%m%y") + self.mainURL_suffix
                self.listOfURLS.append(urlForDate)
            # remove already retrieved URLs:
            self.listOfURLS = completedURLs.removeAlreadyFetchedURLs(self.listOfURLS, self.pluginName)
            logger.info("Total count of valid articles to be retrieved = %s for business date: %s",
                        len(self.listOfURLS), businessDate.strftime("%Y-%m-%d"))
        except Exception as e:
            logger.error("Error trying to retrieve URL list at recursion level %s: %s", self.configData['recursion_level'], e)

    def fetchDataFromURL(self, uRLtoFetch, WorkerID):
        """ Fetch data From given URL
        """
        fullPathName = ""
        dirPathName = ""
        sizeOfDataDownloaded = -1
        uncompressSize = 0
        publishDateStr = ""
        # output tuple structure: (uRL, len_raw_data, len_text, publish_date)
        resultVal = (uRLtoFetch, None, None, None)
        self.master_data_dir = self.configData['master_data_dir']
        logger.debug("Fetching %s, Worker ID %s", uRLtoFetch.encode("ascii"), WorkerID)
        try:
            (publishDate, dataUniqueID) = self.extractUniqueIDFromURL(uRLtoFetch)
            rawData = self.downloadDataArchive(uRLtoFetch, type(self).__name__)
            # write data to file:
            fileNameWithOutExt = self.makeUniqueFileName(dataUniqueID)
            publishDateStr = str(publishDate.strftime("%Y-%m-%d"))
            dirPathName = os.path.join(self.configData['data_dir'], publishDateStr)
            fullPathName = os.path.join(dirPathName, fileNameWithOutExt + ".zip")
            sizeOfDataDownloaded = len(rawData)
        except Exception as e:
            logger.error("Trying to fetch data from given URL: %s", e)

        if sizeOfDataDownloaded > self.minArticleLengthInChars:
            try:
                if os.path.isdir(dirPathName) is False:
                    # since dir does not exist, so try creating it:
                    os.mkdir(dirPathName)
            except Exception as theError:
                logger.error("Error creating data directory '%s', Exception was: %s", dirPathName, theError)
            try:
                with open(fullPathName, 'wb') as fp:
                    n = fp.write(rawData)
                    logger.debug("Wrote %s bytes to file: %s", n, fullPathName)
                    fp.close()
                # save master data:
                sizeOfDataDownloaded = self.fetchMasterData(uRLtoFetch, self.master_data_dir, WorkerID, sizeOfDataDownloaded)
                # save pledges data:
                # sizeOfDataDownloaded = sizeOfDataDownloaded + self.fetchPledgesData(self.master_data_dir, publishDate)
                uncompressSize = self.parseFetchedData(str(publishDate.strftime("%Y%m%d")),
                                                       fullPathName,
                                                       dirPathName,
                                                       WorkerID,
                                                       uRLtoFetch)
            except Exception as theError:
                logger.error("Error saving downloaded data to zip file '%s': %s", fullPathName, theError)
            resultVal = (uRLtoFetch, sizeOfDataDownloaded, uncompressSize, publishDateStr)
        else:
            logger.info("Ignoring data file '%s' since its size (%s bytes) is less than the minimum of %s bytes",
                        fullPathName, len(rawData), self.minArticleLengthInChars)
        return(resultVal)

    def fetchPledgesData(self, dirPathName, publishDate):
        """ Fetch Pledges Data
        Disabled since it is blocked by NSE website
        """
        nFileWriteBytesCount = 0
        fetchTimeout = self.configData['fetch_timeout']
        pledgeURLs = ['https://www.nseindia.com/companies-listing/corporate-filings-pledged-data',
                      'https://www.nseindia.com/dist/js/blinkui/src/blinkui.js',
                      'https://www.nseindia.com/assets/js/nse-min.js',
                      'https://www.nseindia.com/dist/js/sections/corporate-filings.js',
                      'https://www.nseindia.com/assets/js/jquery-3.5.1.min.js',
                      'https://www.nseindia.com/assets/js/popper.min.js',
                      'https://www.nseindia.com/assets/js/jquery.mCustomScrollbar.min.js',
                      'https://www.nseindia.com/assets/js/jquery.responsivetabs.js',
                      'https://www.nseindia.com/assets/js/bootstrap.min.js',
                      'https://www.nseindia.com/assets/js/slick.min.js',
                      'https://www.nseindia.com/assets/js/bootstrap-multiselect.min.js',
                      'https://www.nseindia.com/assets/js/stock/highstock.js',
                      'https://www.nseindia.com/assets/js/rangeslider.min.js',
                      'https://www.nseindia.com/assets/js/excellentexport.js',
                      'https://www.nseindia.com/assets/js/gijgo.min.js',
                      'https://www.nseindia.com/assets/js/typeahead.bundle.js',
                      'https://www.nseindia.com/assets/js/metisMenu.min.js',
                      'https://www.nseindia.com/akam/11/6a433b6b',
                      'https://www.nseindia.com/dist/js/static/public/assets/js/common.js',
                      'https://www.nseindia.com/dist/js/components/quick-links.js',
                      'https://www.nseindia.com/api/holiday-master?type=trading',
                      'https://www.nseindia.com/json/CorporateFiling/CF-sastpledged.json',
                      'https://www.nseindia.com/api/merged-daily-reports?key=favDerivatives',
                      'https://www.nseindia.com/api/merged-daily-reports?key=favCapital',
                      'https://www.nseindia.com/api/circulars',
                      'https://www.nseindia.com/api/merged-daily-reports?key=favDebt',
                      'https://www.nseindia.com/api/latest-circular']
        try:
            publishDateStr = str(publishDate.strftime("%Y-%m-%d"))
            httpHeaders = {'DNT': '1', 'Connection': 'close', 'Referer':
                           'https://www.nseindia.com/companies-listing/corporate-filings-pledged-data', 'Cookie':
                           'nsit=Ds7Vl0Yj379mZde30g0nZQ5I; AKA_A2=A; ak_bmsc=6D9619B384E61EB2D5D6394450D293E5170F' +
                           '2133475800005361A160F8577D2E~plkpGpIxlunjAU8qFb5y9KN/dMhwP9D+Wep9yrJtz5I+8kiMEE5cmeKk' +
                           'jBTzD75O1T2z06zeMCKa5dacyIP/U1n4DPoxcwsbjR87QcZIHJntX0pWg1k0FlstATINQ3sl1X3F1MbffCl7j' +
                           'wP812fjoqs90HqlLbraR8JC82FmSIX8aO0T9456xD76ldIPrpka/DBGhfyH1Km//mrMpTUtiKL6ebLqP3snTV' +
                           'ev8+Vz1u4oc=; nseappid=eyJhbGciOiJIUzI1NiIsInR5cCI6IkpXVCJ9.eyJpc3MiOiJhcGkubnNlIiwiY' +
                           'XVkIjoiYXBpLm5zZSIsImlhdCI6MTYyMTE4ODk1NCwiZXhwIjoxNjIxMTkyNTU0fQ.QmW6tfAQYQuhcafxApu' +
                           'FNhGwH54b-56q6zSD2ZTrwNU; bm_mi=42A1BBB0F94249E75F3D3F1512C1A39D~AuhClqNB8L/IvVznu4rE' +
                           'D/Rwg6ehp1+4SI2zIqy58oBvMxN3km8pM0bWQfuMwnyTO4Z0vwaMqtEpgp8ba0/U23zLoLEfAUhT7i5672nse' +
                           'FwXYrqgs9qy2RoQIl/yot7l++arikkwFRvurO064WqmWRm4vpZLLj5ZFXYAXQIZWj2LvMXtDxAuarWKqrBOBX' +
                           'qdNsPXLS1EnqwH+Ui20HnyhqgJid1hzrKBpMrBUJ+E8l9U9CItcnan54EGQ1N4t6gJg+VbvfHVLeqKIaGJaoR' +
                           'exlJHLXinMFaGI+NIiktvgsk=; bm_sv=3F92FB39943E25067D8188E8CE6F14E8~sAy0A2tWyRGfKiKgjYa' +
                           'blY1CKPjwam9RQStgfNAnADghkRhVOmdvFfh3hfrcma1qGei3Rm4JtFjZvdLPb3Yt8msYi8vcXR5iRI2rUtDc' +
                           '4wf1L54aTFV95soQxA18gbCkCjbUwUALlq46r1vpOhsp5R4TIEtpHBMgR40iXxjvtYQ='
                           }
            if self.pledgesDataExtractedFlag is False:
                with requests.Session() as sess:
                    sess.proxies.update(self.configData['proxies'])
                    sess.headers.update({'sec-ch-ua': '" Not A;Brand";v="99", "Chromium";v="90", "Microsoft Edge";v="90"'})
                    sess.headers.update({'sec-ch-ua-mobile': '?0'})
                    sess.headers.update({'Sec-Fetch-Dest': 'empty'})
                    sess.headers.update({'Sec-Fetch-Site': 'none'})
                    sess.headers.update({'Sec-Fetch-Mode': 'navigate'})
                    sess.headers.update({'Sec-Fetch-User': '?1'})
                    sess.headers.update({'Sec-Fetch-Dest': 'document'})
                    sess.headers.update(httpHeaders)
                    for url in pledgeURLs:
                        try:
                            httpsResp = sess.get(url, timeout=fetchTimeout)
                            logger.debug('Got %s characters from url: %s', len(httpsResp.content), url)
                        except Exception as e:
                            logger.error('Error getting url %s: %s', url, e)
                    httpsResp = sess.get(self.pledgesURL2, timeout=fetchTimeout)
                    pledgesPathName = os.path.join(dirPathName, 'nse_pledges_' + publishDateStr + '.csv')
                    if httpsResp is not None:
                        with open(pledgesPathName, 'wb') as fp:
                            nFileWriteBytesCount = fp.write(httpsResp.content)
                            logger.debug("Wrote %s bytes data to pledges file: %s", nFileWriteBytesCount, pledgesPathName)
                            fp.close()
                    self.pledgesDataExtractedFlag = True
            else:
                logger.info("Already downloaded the pledges data as of date %s, hence, not repeating download.",
                            publishDateStr)
        except Exception as e:
            logger.error("Error saving pledges data: %s", e)
        return(nFileWriteBytesCount)

    def fetchMasterData(self, uRLtoFetch, dirPathName, WorkerID, sizeOfDataDownloaded):
        """ fetch Master Data"""
        if self.masterDataExtractedFlag is False:
            for masterFileName in self.masterData:
                try:
                    masterRawData = self.networkHelper.fetchRawDataFromURL(self.masterData[masterFileName],
                                                                           type(self).__name__)
                    sizeOfDataDownloaded = sizeOfDataDownloaded + len(masterRawData)
                    fullMasterPathName = os.path.join(dirPathName, masterFileName)
                    with open(fullMasterPathName, 'wb') as fp:
                        n = fp.write(masterRawData)
                        logger.debug("Wrote %s bytes to file: %s", n, fullMasterPathName)
                        fp.close()
                        self.masterDataExtractedFlag = True
                except Exception as e:
                    logger.error("Error saving master data: %s", e)
        return(sizeOfDataDownloaded)

    def parseFetchedData(self, publishDateStr, zipFileName, dataDirForDate, WorkerID, uRLtoFetch):
        """Parse the fetched Data
        """
        uncompressSize = 0
        try:
            zipDatafile = zipfile.ZipFile(zipFileName, mode='r')
            logger.debug("Expanding the fetched Zip archive, WorkerID = %s", WorkerID)
            for memberZipInfo in zipDatafile.infolist():
                memberFileName = memberZipInfo.filename
                if memberFileName.find('Readme.txt') < 0:
                    try:
                        if os.path.isfile(os.path.join(dataDirForDate, memberFileName)) is False:
                            logger.debug("Extracting file '%s' from zip archive.", memberFileName)
                            zipDatafile.extract(memberZipInfo, path=dataDirForDate)
                        newFileName = self.calculateFileNamePrefix(memberFileName, dataDirForDate, publishDateStr)
                        if os.path.isfile(newFileName) is False:
                            logger.debug("Renaming %s to %s",
                                         os.path.join(dataDirForDate, memberFileName),
                                         newFileName)
                            os.rename(os.path.join(dataDirForDate, memberFileName),
                                      newFileName)
                            uncompressSize = uncompressSize + memberZipInfo.file_size
                        if memberFileName.startswith('An'):
                            self.processAnnouncements(newFileName, uRLtoFetch)
                    except Exception as e:
                        logger.error("Error exptracting file %s from the downloaded zip archive: %s", memberFileName, e)
            # close the zip file
            zipDatafile.close()
            os.remove(zipFileName)  # delete zip file as its no longer required
        except Exception as e:
            logger.error("Error expanding archive file: %s", e)
        return(uncompressSize)

    def calculateFileNamePrefix(self, oldName, dataDirForDate, publishDateStr):
        """ calculate FileName Prefix """
        filename_part, file_extension = os.path.splitext(oldName)
        if oldName.startswith('An'):
            newFileNamePrefix = os.path.join(dataDirForDate,
                                             "equity_nse_announce_" + publishDateStr + file_extension)
        elif oldName.startswith('Bm'):
            newFileNamePrefix = os.path.join(dataDirForDate,
                                             "equity_nse_boardmeet_" + publishDateStr + file_extension)
        elif oldName.startswith('Bc'):
            newFileNamePrefix = os.path.join(dataDirForDate,
                                             "equity_nse_corp_action_" + publishDateStr + file_extension)
        elif oldName.startswith('Pd'):
            newFileNamePrefix = os.path.join(dataDirForDate,
                                             "equity_nse_" + publishDateStr + file_extension)
        elif oldName.startswith('corpbond'):
            newFileNamePrefix = os.path.join(dataDirForDate,
                                             "debt_nse_" + publishDateStr + file_extension)
        else:
            newFileNamePrefix = os.path.join(dataDirForDate, type(self).__name__ + "_" + oldName)
        return(newFileNamePrefix)

    def extractUniqueIDFromURL(self, URLToFetch):
        """ Get Unique ID From URL by extracting RegEx patterns matching any of urlMatchPatterns
        """
        # use today's date as default
        uniqueString = datetime.now().strftime('%d%m%y')
        date_obj = None
        if len(URLToFetch) > 6:
            for urlPattern in self.urlMatchPatterns:
                try:
                    result = urlPattern.search(URLToFetch)
                    uniqueString = result.group(2)
                    date_obj = datetime.strptime(uniqueString, '%d%m%y')
                    # if we did not encounter any error till this point, then this is the answer, so exit loop
                    break
                except Exception as e:
                    logger.debug("Error identifying unique ID of URL: %s , URL was: %s, Pattern: %s",
                                 e,
                                 URLToFetch.encode('ascii'),
                                 urlPattern)
        return((date_obj, uniqueString))

    def processAnnouncements(self, announceFileFullPath, archiveURL):
        """ Process all company Announcements to the exchange
        """
        logger.debug("Processing announcements for file: %s", announceFileFullPath)
        try:
            recordPrefixPat = re.compile(r'([a-zA-Z0-9\(\)¿ \-.^:]{3,})([ ]+[a-zA-Z0-9\-]{2,})([ ]+:)')
            with open(announceFileFullPath, 'rt') as fp:
                fileRecords = fp.readlines()
                fp.close()
            logger.debug("Read %s announcements.", len(fileRecords))
            for index, announceRecord in enumerate(fileRecords):
                try:
                    if (index > 0 and announceRecord.find(':') > -1 and
                            announceRecord.lower().find("declaration of nav ") < 0 and
                            announceRecord.lower().find("recommended final dividend") < 0 and
                            announceRecord.lower().find("about investor presentation") < 0 and
                            announceRecord.lower().find("that the record date has been fixed on") < 0 and
                            announceRecord.lower().find("the maturity date of the scheme is") < 0 and
                            announceRecord.lower().find("suspension of trading") < 0 and
                            announceRecord.lower().find("postal ballot") < 0):
                        recordPrefix = announceRecord[: announceRecord.find(':')+1]
                        announceContent = announceRecord[announceRecord.find(':')+1:]
                        if index+1 < len(fileRecords) and fileRecords[index+1].find(':') == -1:
                            announceContent = announceContent + " " + fileRecords[index+1]
                        searchRes = recordPrefixPat.search(recordPrefix)
                        if searchRes is not None:
                            entityName = searchRes.group(1)
                            companySymbol = searchRes.group(2).strip()
                            thisArticle = NewsArticle()  # make article for each announcement
                            thisArticle.setPublishDate(self.configData['rundate'])
                            thisArticle.setModuleName(self.pluginName)
                            thisArticle.setIndustries([companySymbol])
                            thisArticle.setTitle('NSE Announcement for ' + entityName.strip())
                            thisArticle.setKeyWords([entityName.strip()])
                            thisArticle.setText(announceContent)
                            thisArticle.setURL(archiveURL)
                            # save each article to its unique filename: crc32 of line no, company symbol and date
                            articleUniqueID = str(calculateCRC32(companySymbol.strip()))
                            thisArticle.setArticleID(articleUniqueID)
                            thisArticle.setSource('NSE')
                            thisArticle.identifyTriggerWordFlags(self.configData['configReader'])
                            filename = self.makeUniqueFileName(articleUniqueID)
                            thisArticle.writeFiles(filename, self.configData['data_dir'], announceContent)
                        else:
                            logger.debug("Skipping record %s as it is not properly formatted.", index)
                except Exception as e:
                    logger.error("Error processing NSE announcement no %s: %s", index, e)
        except Exception as e:
            logger.error("Error processing NSE announcements: %s", e)

# # end of file ##
