import ape

from ape_vyper.exceptions import FallbackNotDefinedError, NonPayableError


def test_immutable_number(contract, account, START_NUM):
    """
    Ensure that immutable members work in coverage.
    This is needed for full function coverage.
    """
    assert contract._immutable_number() == START_NUM


def test_happy_path(contract, account, chain):
    """
    Covers some implicit statements as well two source statements.
    """
    num1 = 5
    num2 = 6
    receipt = contract.foo_method(num1, num2, sender=account)

    # In order for full coverage, we need to also call `_number()`.
    assert contract._number() == num1 + num2

    if chain.provider.supports_tracing:
        assert receipt.return_value is True


def test_sad_path(contract, account, chain):
    """
    Covers some implicit statements as well as one source statement.
    """
    kwargs = {"dev_message": "dev: sub-zero"} if chain.provider.supports_tracing else {}
    with ape.reverts(**kwargs):
        contract.foo_method(0, 1, sender=account)


def test_non_payable(contract, account):
    """
    Coverage (at least) the implicit statement for non-payable check.
    """
    with ape.reverts(NonPayableError):
        contract.foo_method(5, 1, sender=account, value=1)


def test_no_default_method(contract, account):
    """
    Covers the implicit check for fallback not defined.
    """
    with ape.reverts(FallbackNotDefinedError):
        contract(sender=account)


def test_foo_method_no_args(contract, account):
    """
    Covers the statement from the autogenerated method.
    """
    receipt = contract.foo_method(sender=account)
    assert not receipt.failed


def test_foo_method_one_arg(contract, account):
    """
    Covers the statement from the autogenerated method.
    """
    receipt = contract.foo_method(1, sender=account)
    assert not receipt.failed


def test_view_method(contract, account):
    """
    Prove that view methods work in coverage.
    Without this, it would not be 100%.
    """
    assert contract.view_method() is True
