"use strict";
const assert_1 = require("@aws-cdk/assert");
const ec2 = require("@aws-cdk/aws-ec2");
const cdk = require("@aws-cdk/core");
const elbv2 = require("../../lib");
const helpers_1 = require("../helpers");
const LB_SECURITY_GROUP = { 'Fn::GetAtt': ['LBSecurityGroup8A41EA2B', 'GroupId'] };
const IMPORTED_LB_SECURITY_GROUP = { 'Fn::ImportValue': 'Stack:ExportsOutputFnGetAttLBSecurityGroup8A41EA2BGroupId851EE1F6' };
function expectSameStackSGRules(stack) {
    expectSGRules(stack, LB_SECURITY_GROUP);
}
function expectedImportedSGRules(stack) {
    expectSGRules(stack, IMPORTED_LB_SECURITY_GROUP);
}
function expectSGRules(stack, lbGroup) {
    assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::SecurityGroupEgress', {
        GroupId: lbGroup,
        IpProtocol: 'tcp',
        Description: 'Load balancer to target',
        DestinationSecurityGroupId: { 'Fn::GetAtt': ['TargetSGDB98152D', 'GroupId'] },
        FromPort: 8008,
        ToPort: 8008,
    }));
    assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::SecurityGroupIngress', {
        IpProtocol: 'tcp',
        Description: 'Load balancer to target',
        FromPort: 8008,
        GroupId: { 'Fn::GetAtt': ['TargetSGDB98152D', 'GroupId'] },
        SourceSecurityGroupId: lbGroup,
        ToPort: 8008,
    }));
}
class TestFixture {
    constructor(createListener) {
        this.app = new cdk.App();
        this.stack = new cdk.Stack(this.app, 'Stack');
        this.vpc = new ec2.Vpc(this.stack, 'VPC', {
            maxAzs: 2,
        });
        this.lb = new elbv2.ApplicationLoadBalancer(this.stack, 'LB', { vpc: this.vpc });
        createListener = createListener === undefined ? true : createListener;
        if (createListener) {
            this._listener = this.lb.addListener('Listener', { port: 80, open: false });
        }
    }
    get listener() {
        if (this._listener === undefined) {
            throw new Error('Did not create a listener');
        }
        return this._listener;
    }
}
module.exports = {
    'security groups are automatically opened bidi for default rule'(test) {
        // GIVEN
        const fixture = new TestFixture();
        const target = new helpers_1.FakeSelfRegisteringTarget(fixture.stack, 'Target', fixture.vpc);
        // WHEN
        fixture.listener.addTargets('TargetGroup', {
            port: 8008,
            targets: [target],
        });
        // THEN
        expectSameStackSGRules(fixture.stack);
        test.done();
    },
    'security groups are automatically opened bidi for additional rule'(test) {
        // GIVEN
        const fixture = new TestFixture();
        const target1 = new helpers_1.FakeSelfRegisteringTarget(fixture.stack, 'DefaultTarget', fixture.vpc);
        const target2 = new helpers_1.FakeSelfRegisteringTarget(fixture.stack, 'Target', fixture.vpc);
        // WHEN
        fixture.listener.addTargets('TargetGroup1', {
            port: 80,
            targets: [target1],
        });
        fixture.listener.addTargetGroups('Rule', {
            priority: 10,
            hostHeader: 'example.com',
            targetGroups: [new elbv2.ApplicationTargetGroup(fixture.stack, 'TargetGroup2', {
                    vpc: fixture.vpc,
                    port: 8008,
                    targets: [target2],
                })],
        });
        // THEN
        expectSameStackSGRules(fixture.stack);
        test.done();
    },
    'adding the same targets twice also works'(test) {
        // GIVEN
        const fixture = new TestFixture();
        const target = new helpers_1.FakeSelfRegisteringTarget(fixture.stack, 'Target', fixture.vpc);
        // WHEN
        const group = new elbv2.ApplicationTargetGroup(fixture.stack, 'TargetGroup', {
            vpc: fixture.vpc,
            port: 8008,
            targets: [target],
        });
        fixture.listener.addTargetGroups('Default', {
            targetGroups: [group],
        });
        fixture.listener.addTargetGroups('WithPath', {
            priority: 10,
            pathPattern: '/hello',
            targetGroups: [group],
        });
        // THEN
        expectSameStackSGRules(fixture.stack);
        test.done();
    },
    'same result if target is added to group after assigning to listener'(test) {
        // GIVEN
        const fixture = new TestFixture();
        const group = new elbv2.ApplicationTargetGroup(fixture.stack, 'TargetGroup', {
            vpc: fixture.vpc,
            port: 8008,
        });
        fixture.listener.addTargetGroups('Default', {
            targetGroups: [group],
        });
        // WHEN
        const target = new helpers_1.FakeSelfRegisteringTarget(fixture.stack, 'Target', fixture.vpc);
        group.addTarget(target);
        // THEN
        expectSameStackSGRules(fixture.stack);
        test.done();
    },
    'ingress is added to child stack SG instead of parent stack'(test) {
        // GIVEN
        const fixture = new TestFixture(true);
        const parentGroup = new elbv2.ApplicationTargetGroup(fixture.stack, 'TargetGroup', {
            vpc: fixture.vpc,
            port: 8008,
            targets: [new helpers_1.FakeSelfRegisteringTarget(fixture.stack, 'Target', fixture.vpc)],
        });
        // listener requires at least one rule for ParentStack to create
        fixture.listener.addTargetGroups('Default', { targetGroups: [parentGroup] });
        const childStack = new cdk.Stack(fixture.app, 'childStack');
        // WHEN
        const childGroup = new elbv2.ApplicationTargetGroup(childStack, 'TargetGroup', {
            // We're assuming the 2nd VPC is peered to the 1st, or something.
            vpc: fixture.vpc,
            port: 8008,
            targets: [new helpers_1.FakeSelfRegisteringTarget(childStack, 'Target', fixture.vpc)],
        });
        new elbv2.ApplicationListenerRule(childStack, 'ListenerRule', {
            listener: fixture.listener,
            targetGroups: [childGroup],
            priority: 100,
            hostHeader: 'www.foo.com',
        });
        // THEN
        expectSameStackSGRules(fixture.stack);
        expectedImportedSGRules(childStack);
        test.done();
    },
    'SG peering works on exported/imported load balancer'(test) {
        // GIVEN
        const fixture = new TestFixture(false);
        const stack2 = new cdk.Stack(fixture.app, 'stack2');
        const vpc2 = new ec2.Vpc(stack2, 'VPC');
        const group = new elbv2.ApplicationTargetGroup(stack2, 'TargetGroup', {
            // We're assuming the 2nd VPC is peered to the 1st, or something.
            vpc: vpc2,
            port: 8008,
            targets: [new helpers_1.FakeSelfRegisteringTarget(stack2, 'Target', vpc2)],
        });
        // WHEN
        const lb2 = elbv2.ApplicationLoadBalancer.fromApplicationLoadBalancerAttributes(stack2, 'LB', {
            loadBalancerArn: fixture.lb.loadBalancerArn,
            securityGroupId: fixture.lb.connections.securityGroups[0].securityGroupId,
            securityGroupAllowsAllOutbound: false,
        });
        const listener2 = lb2.addListener('YetAnotherListener', { port: 80 });
        listener2.addTargetGroups('Default', { targetGroups: [group] });
        // THEN
        expectedImportedSGRules(stack2);
        test.done();
    },
    'SG peering works on exported/imported listener'(test) {
        // GIVEN
        const fixture = new TestFixture();
        const stack2 = new cdk.Stack(fixture.app, 'stack2');
        const vpc2 = new ec2.Vpc(stack2, 'VPC');
        const group = new elbv2.ApplicationTargetGroup(stack2, 'TargetGroup', {
            // We're assuming the 2nd VPC is peered to the 1st, or something.
            vpc: vpc2,
            port: 8008,
            targets: [new helpers_1.FakeSelfRegisteringTarget(stack2, 'Target', vpc2)],
        });
        fixture.listener.addTargets('default', { port: 80 });
        // WHEN
        const listener2 = elbv2.ApplicationListener.fromApplicationListenerAttributes(stack2, 'YetAnotherListener', {
            defaultPort: 8008,
            securityGroupId: fixture.listener.connections.securityGroups[0].securityGroupId,
            listenerArn: fixture.listener.listenerArn,
            securityGroupAllowsAllOutbound: false,
        });
        listener2.addTargetGroups('Default', {
            // Must be a non-default target
            priority: 10,
            hostHeader: 'example.com',
            targetGroups: [group],
        });
        // THEN
        expectedImportedSGRules(stack2);
        test.done();
    },
    'default port peering works on constructed listener'(test) {
        // GIVEN
        const fixture = new TestFixture();
        fixture.listener.addTargets('Default', { port: 8080, targets: [new elbv2.InstanceTarget('i-12345')] });
        // WHEN
        fixture.listener.connections.allowDefaultPortFromAnyIpv4('Open to the world');
        // THEN
        assert_1.expect(fixture.stack).to(assert_1.haveResource('AWS::EC2::SecurityGroup', {
            SecurityGroupIngress: [
                {
                    CidrIp: '0.0.0.0/0',
                    Description: 'Open to the world',
                    FromPort: 80,
                    IpProtocol: 'tcp',
                    ToPort: 80,
                },
            ],
        }));
        test.done();
    },
    'default port peering works on imported listener'(test) {
        // GIVEN
        const stack2 = new cdk.Stack();
        // WHEN
        const listener2 = elbv2.ApplicationListener.fromApplicationListenerAttributes(stack2, 'YetAnotherListener', {
            listenerArn: 'listener-arn',
            securityGroupId: 'imported-security-group-id',
            defaultPort: 8080,
        });
        listener2.connections.allowDefaultPortFromAnyIpv4('Open to the world');
        // THEN
        assert_1.expect(stack2).to(assert_1.haveResource('AWS::EC2::SecurityGroupIngress', {
            CidrIp: '0.0.0.0/0',
            Description: 'Open to the world',
            IpProtocol: 'tcp',
            FromPort: 8080,
            ToPort: 8080,
            GroupId: 'imported-security-group-id',
        }));
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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