"use strict";
const assert_1 = require("@aws-cdk/assert");
const dynamodb = require("@aws-cdk/aws-dynamodb");
const lambda = require("@aws-cdk/aws-lambda");
const sqs = require("@aws-cdk/aws-sqs");
const cdk = require("@aws-cdk/core");
const sources = require("../lib");
const test_function_1 = require("./test-function");
module.exports = {
    'sufficiently complex example'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const fn = new test_function_1.TestFunction(stack, 'Fn');
        const table = new dynamodb.Table(stack, 'T', {
            partitionKey: {
                name: 'id',
                type: dynamodb.AttributeType.STRING
            },
            stream: dynamodb.StreamViewType.NEW_IMAGE
        });
        // WHEN
        fn.addEventSource(new sources.DynamoEventSource(table, {
            startingPosition: lambda.StartingPosition.TRIM_HORIZON
        }));
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
            "PolicyDocument": {
                "Statement": [
                    {
                        "Action": "dynamodb:ListStreams",
                        "Effect": "Allow",
                        "Resource": { "Fn::Join": ["", [{ "Fn::GetAtt": ["TD925BC7E", "Arn"] }, "/stream/*"]] }
                    },
                    {
                        "Action": [
                            "dynamodb:DescribeStream",
                            "dynamodb:GetRecords",
                            "dynamodb:GetShardIterator",
                        ],
                        "Effect": "Allow",
                        "Resource": {
                            "Fn::GetAtt": [
                                "TD925BC7E",
                                "StreamArn"
                            ]
                        }
                    }
                ],
                "Version": "2012-10-17"
            },
            "PolicyName": "FnServiceRoleDefaultPolicyC6A839BF",
            "Roles": [{
                    "Ref": "FnServiceRoleB9001A96"
                }]
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Lambda::EventSourceMapping', {
            "EventSourceArn": {
                "Fn::GetAtt": [
                    "TD925BC7E",
                    "StreamArn"
                ]
            },
            "FunctionName": {
                "Ref": "Fn9270CBC0"
            },
            "BatchSize": 100,
            "StartingPosition": "TRIM_HORIZON"
        }));
        test.done();
    },
    'specific batch size'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const fn = new test_function_1.TestFunction(stack, 'Fn');
        const table = new dynamodb.Table(stack, 'T', {
            partitionKey: {
                name: 'id',
                type: dynamodb.AttributeType.STRING
            },
            stream: dynamodb.StreamViewType.NEW_IMAGE
        });
        // WHEN
        fn.addEventSource(new sources.DynamoEventSource(table, {
            batchSize: 50,
            startingPosition: lambda.StartingPosition.LATEST
        }));
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Lambda::EventSourceMapping', {
            "EventSourceArn": {
                "Fn::GetAtt": [
                    "TD925BC7E",
                    "StreamArn"
                ]
            },
            "FunctionName": {
                "Ref": "Fn9270CBC0"
            },
            "BatchSize": 50,
            "StartingPosition": "LATEST"
        }));
        test.done();
    },
    'fails if streaming not enabled on table'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const fn = new test_function_1.TestFunction(stack, 'Fn');
        const table = new dynamodb.Table(stack, 'T', {
            partitionKey: {
                name: 'id',
                type: dynamodb.AttributeType.STRING
            }
        });
        // WHEN
        test.throws(() => fn.addEventSource(new sources.DynamoEventSource(table, {
            batchSize: 50,
            startingPosition: lambda.StartingPosition.LATEST
        })), /DynamoDB Streams must be enabled on the table T/);
        test.done();
    },
    'fails if batch size < 1'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const fn = new test_function_1.TestFunction(stack, 'Fn');
        const table = new dynamodb.Table(stack, 'T', {
            partitionKey: {
                name: 'id',
                type: dynamodb.AttributeType.STRING
            },
            stream: dynamodb.StreamViewType.NEW_IMAGE
        });
        // WHEN
        test.throws(() => fn.addEventSource(new sources.DynamoEventSource(table, {
            batchSize: 0,
            startingPosition: lambda.StartingPosition.LATEST
        })), /Maximum batch size must be between 1 and 1000 inclusive \(given 0\)/);
        test.done();
    },
    'fails if batch size > 1000'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const fn = new test_function_1.TestFunction(stack, 'Fn');
        const table = new dynamodb.Table(stack, 'T', {
            partitionKey: {
                name: 'id',
                type: dynamodb.AttributeType.STRING
            },
            stream: dynamodb.StreamViewType.NEW_IMAGE
        });
        // WHEN
        test.throws(() => fn.addEventSource(new sources.DynamoEventSource(table, {
            batchSize: 1001,
            startingPosition: lambda.StartingPosition.LATEST
        })), /Maximum batch size must be between 1 and 1000 inclusive \(given 1001\)/);
        test.done();
    },
    'specific maxBatchingWindow'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const fn = new test_function_1.TestFunction(stack, 'Fn');
        const table = new dynamodb.Table(stack, 'T', {
            partitionKey: {
                name: 'id',
                type: dynamodb.AttributeType.STRING
            },
            stream: dynamodb.StreamViewType.NEW_IMAGE
        });
        // WHEN
        fn.addEventSource(new sources.DynamoEventSource(table, {
            maxBatchingWindow: cdk.Duration.minutes(2),
            startingPosition: lambda.StartingPosition.LATEST
        }));
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Lambda::EventSourceMapping', {
            "EventSourceArn": {
                "Fn::GetAtt": [
                    "TD925BC7E",
                    "StreamArn"
                ]
            },
            "FunctionName": {
                "Ref": "Fn9270CBC0"
            },
            "MaximumBatchingWindowInSeconds": 120,
            "StartingPosition": "LATEST"
        }));
        test.done();
    },
    'throws if maxBatchingWindow > 300 seconds'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const fn = new test_function_1.TestFunction(stack, 'Fn');
        const table = new dynamodb.Table(stack, 'T', {
            partitionKey: {
                name: 'id',
                type: dynamodb.AttributeType.STRING
            },
            stream: dynamodb.StreamViewType.NEW_IMAGE
        });
        // THEN
        test.throws(() => fn.addEventSource(new sources.DynamoEventSource(table, {
            maxBatchingWindow: cdk.Duration.seconds(301),
            startingPosition: lambda.StartingPosition.LATEST
        })), /maxBatchingWindow cannot be over 300 seconds/);
        test.done();
    },
    'contains eventSourceMappingId after lambda binding'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const fn = new test_function_1.TestFunction(stack, 'Fn');
        const table = new dynamodb.Table(stack, 'T', {
            partitionKey: {
                name: 'id',
                type: dynamodb.AttributeType.STRING
            },
            stream: dynamodb.StreamViewType.NEW_IMAGE
        });
        const eventSource = new sources.DynamoEventSource(table, {
            startingPosition: lambda.StartingPosition.TRIM_HORIZON
        });
        // WHEN
        fn.addEventSource(eventSource);
        // THEN
        test.ok(eventSource.eventSourceMappingId);
        test.done();
    },
    'eventSourceMappingId throws error before binding to lambda'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const table = new dynamodb.Table(stack, 'T', {
            partitionKey: {
                name: 'id',
                type: dynamodb.AttributeType.STRING
            },
            stream: dynamodb.StreamViewType.NEW_IMAGE
        });
        const eventSource = new sources.DynamoEventSource(table, {
            startingPosition: lambda.StartingPosition.TRIM_HORIZON
        });
        // WHEN/THEN
        test.throws(() => eventSource.eventSourceMappingId, /DynamoEventSource is not yet bound to an event source mapping/);
        test.done();
    },
    'specific retryAttempts'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const fn = new test_function_1.TestFunction(stack, 'Fn');
        const table = new dynamodb.Table(stack, 'T', {
            partitionKey: {
                name: 'id',
                type: dynamodb.AttributeType.STRING
            },
            stream: dynamodb.StreamViewType.NEW_IMAGE
        });
        // WHEN
        fn.addEventSource(new sources.DynamoEventSource(table, {
            retryAttempts: 10,
            startingPosition: lambda.StartingPosition.LATEST
        }));
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Lambda::EventSourceMapping', {
            "EventSourceArn": {
                "Fn::GetAtt": [
                    "TD925BC7E",
                    "StreamArn"
                ]
            },
            "FunctionName": {
                "Ref": "Fn9270CBC0"
            },
            "MaximumRetryAttempts": 10,
            "StartingPosition": "LATEST"
        }));
        test.done();
    },
    'fails if retryAttempts < 0'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const fn = new test_function_1.TestFunction(stack, 'Fn');
        const table = new dynamodb.Table(stack, 'T', {
            partitionKey: {
                name: 'id',
                type: dynamodb.AttributeType.STRING
            },
            stream: dynamodb.StreamViewType.NEW_IMAGE
        });
        // THEN
        test.throws(() => fn.addEventSource(new sources.DynamoEventSource(table, {
            retryAttempts: -1,
            startingPosition: lambda.StartingPosition.LATEST
        })), /retryAttempts must be between 0 and 10000 inclusive, got -1/);
        test.done();
    },
    'fails if retryAttempts > 10000'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const fn = new test_function_1.TestFunction(stack, 'Fn');
        const table = new dynamodb.Table(stack, 'T', {
            partitionKey: {
                name: 'id',
                type: dynamodb.AttributeType.STRING
            },
            stream: dynamodb.StreamViewType.NEW_IMAGE
        });
        // THEN
        test.throws(() => fn.addEventSource(new sources.DynamoEventSource(table, {
            retryAttempts: 10001,
            startingPosition: lambda.StartingPosition.LATEST
        })), /retryAttempts must be between 0 and 10000 inclusive, got 10001/);
        test.done();
    },
    'specific bisectBatchOnFunctionError'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const fn = new test_function_1.TestFunction(stack, 'Fn');
        const table = new dynamodb.Table(stack, 'T', {
            partitionKey: {
                name: 'id',
                type: dynamodb.AttributeType.STRING
            },
            stream: dynamodb.StreamViewType.NEW_IMAGE
        });
        // WHEN
        fn.addEventSource(new sources.DynamoEventSource(table, {
            bisectBatchOnError: true,
            startingPosition: lambda.StartingPosition.LATEST
        }));
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Lambda::EventSourceMapping', {
            "EventSourceArn": {
                "Fn::GetAtt": [
                    "TD925BC7E",
                    "StreamArn"
                ]
            },
            "FunctionName": {
                "Ref": "Fn9270CBC0"
            },
            "BisectBatchOnFunctionError": true,
            "StartingPosition": "LATEST"
        }));
        test.done();
    },
    'specific parallelizationFactor'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const fn = new test_function_1.TestFunction(stack, 'Fn');
        const table = new dynamodb.Table(stack, 'T', {
            partitionKey: {
                name: 'id',
                type: dynamodb.AttributeType.STRING
            },
            stream: dynamodb.StreamViewType.NEW_IMAGE
        });
        // WHEN
        fn.addEventSource(new sources.DynamoEventSource(table, {
            parallelizationFactor: 5,
            startingPosition: lambda.StartingPosition.LATEST
        }));
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Lambda::EventSourceMapping', {
            "EventSourceArn": {
                "Fn::GetAtt": [
                    "TD925BC7E",
                    "StreamArn"
                ]
            },
            "FunctionName": {
                "Ref": "Fn9270CBC0"
            },
            "ParallelizationFactor": 5,
            "StartingPosition": "LATEST"
        }));
        test.done();
    },
    'fails if parallelizationFactor < 1'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const fn = new test_function_1.TestFunction(stack, 'Fn');
        const table = new dynamodb.Table(stack, 'T', {
            partitionKey: {
                name: 'id',
                type: dynamodb.AttributeType.STRING
            },
            stream: dynamodb.StreamViewType.NEW_IMAGE
        });
        // THEN
        test.throws(() => fn.addEventSource(new sources.DynamoEventSource(table, {
            parallelizationFactor: 0,
            startingPosition: lambda.StartingPosition.LATEST
        })), /parallelizationFactor must be between 1 and 10 inclusive, got 0/);
        test.done();
    },
    'fails if parallelizationFactor > 10'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const fn = new test_function_1.TestFunction(stack, 'Fn');
        const table = new dynamodb.Table(stack, 'T', {
            partitionKey: {
                name: 'id',
                type: dynamodb.AttributeType.STRING
            },
            stream: dynamodb.StreamViewType.NEW_IMAGE
        });
        // THEN
        test.throws(() => fn.addEventSource(new sources.DynamoEventSource(table, {
            parallelizationFactor: 11,
            startingPosition: lambda.StartingPosition.LATEST
        })), /parallelizationFactor must be between 1 and 10 inclusive, got 11/);
        test.done();
    },
    'specific maxRecordAge'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const fn = new test_function_1.TestFunction(stack, 'Fn');
        const table = new dynamodb.Table(stack, 'T', {
            partitionKey: {
                name: 'id',
                type: dynamodb.AttributeType.STRING
            },
            stream: dynamodb.StreamViewType.NEW_IMAGE
        });
        // WHEN
        fn.addEventSource(new sources.DynamoEventSource(table, {
            maxRecordAge: cdk.Duration.seconds(100),
            startingPosition: lambda.StartingPosition.LATEST
        }));
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Lambda::EventSourceMapping', {
            "EventSourceArn": {
                "Fn::GetAtt": [
                    "TD925BC7E",
                    "StreamArn"
                ]
            },
            "FunctionName": {
                "Ref": "Fn9270CBC0"
            },
            "MaximumRecordAgeInSeconds": 100,
            "StartingPosition": "LATEST"
        }));
        test.done();
    },
    'fails if maxRecordAge < 60 seconds'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const fn = new test_function_1.TestFunction(stack, 'Fn');
        const table = new dynamodb.Table(stack, 'T', {
            partitionKey: {
                name: 'id',
                type: dynamodb.AttributeType.STRING
            },
            stream: dynamodb.StreamViewType.NEW_IMAGE
        });
        // THEN
        test.throws(() => fn.addEventSource(new sources.DynamoEventSource(table, {
            maxRecordAge: cdk.Duration.seconds(59),
            startingPosition: lambda.StartingPosition.LATEST
        })), /maxRecordAge must be between 60 seconds and 7 days inclusive/);
        test.done();
    },
    'fails if maxRecordAge > 7 days'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const fn = new test_function_1.TestFunction(stack, 'Fn');
        const table = new dynamodb.Table(stack, 'T', {
            partitionKey: {
                name: 'id',
                type: dynamodb.AttributeType.STRING
            },
            stream: dynamodb.StreamViewType.NEW_IMAGE
        });
        // THEN
        test.throws(() => fn.addEventSource(new sources.DynamoEventSource(table, {
            maxRecordAge: cdk.Duration.seconds(604801),
            startingPosition: lambda.StartingPosition.LATEST
        })), /maxRecordAge must be between 60 seconds and 7 days inclusive/);
        test.done();
    },
    'specific destinationConfig'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const fn = new test_function_1.TestFunction(stack, 'Fn');
        const queue = new sqs.Queue(stack, 'Queue');
        const table = new dynamodb.Table(stack, 'T', {
            partitionKey: {
                name: 'id',
                type: dynamodb.AttributeType.STRING
            },
            stream: dynamodb.StreamViewType.NEW_IMAGE
        });
        // WHEN
        fn.addEventSource(new sources.DynamoEventSource(table, {
            onFailure: new sources.SqsDlq(queue),
            startingPosition: lambda.StartingPosition.LATEST
        }));
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Lambda::EventSourceMapping', {
            "EventSourceArn": {
                "Fn::GetAtt": [
                    "TD925BC7E",
                    "StreamArn"
                ]
            },
            "FunctionName": {
                "Ref": "Fn9270CBC0"
            },
            "DestinationConfig": {
                "OnFailure": {
                    "Destination": {
                        "Fn::GetAtt": [
                            "Queue4A7E3555",
                            "Arn"
                        ]
                    }
                }
            },
            "StartingPosition": "LATEST"
        }));
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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