"use strict";
const assert_1 = require("@aws-cdk/assert");
const sqs = require("@aws-cdk/aws-sqs");
const cdk = require("@aws-cdk/core");
const sources = require("../lib");
const test_function_1 = require("./test-function");
module.exports = {
    'defaults'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const fn = new test_function_1.TestFunction(stack, 'Fn');
        const q = new sqs.Queue(stack, 'Q');
        // WHEN
        fn.addEventSource(new sources.SqsEventSource(q));
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
            "PolicyDocument": {
                "Statement": [
                    {
                        "Action": [
                            "sqs:ReceiveMessage",
                            "sqs:ChangeMessageVisibility",
                            "sqs:GetQueueUrl",
                            "sqs:DeleteMessage",
                            "sqs:GetQueueAttributes"
                        ],
                        "Effect": "Allow",
                        "Resource": {
                            "Fn::GetAtt": [
                                "Q63C6E3AB",
                                "Arn"
                            ]
                        }
                    }
                ],
                "Version": "2012-10-17"
            }
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Lambda::EventSourceMapping', {
            "EventSourceArn": {
                "Fn::GetAtt": [
                    "Q63C6E3AB",
                    "Arn"
                ]
            },
            "FunctionName": {
                "Ref": "Fn9270CBC0"
            }
        }));
        test.done();
    },
    'specific batch size'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const fn = new test_function_1.TestFunction(stack, 'Fn');
        const q = new sqs.Queue(stack, 'Q');
        // WHEN
        fn.addEventSource(new sources.SqsEventSource(q, {
            batchSize: 5
        }));
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Lambda::EventSourceMapping', {
            "EventSourceArn": {
                "Fn::GetAtt": [
                    "Q63C6E3AB",
                    "Arn"
                ]
            },
            "FunctionName": {
                "Ref": "Fn9270CBC0"
            },
            "BatchSize": 5
        }));
        test.done();
    },
    'fails if batch size is < 1'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const fn = new test_function_1.TestFunction(stack, 'Fn');
        const q = new sqs.Queue(stack, 'Q');
        // WHEN/THEN
        test.throws(() => fn.addEventSource(new sources.SqsEventSource(q, {
            batchSize: 0
        })), /Maximum batch size must be between 1 and 10 inclusive \(given 0\)/);
        test.done();
    },
    'fails if batch size is > 10'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const fn = new test_function_1.TestFunction(stack, 'Fn');
        const q = new sqs.Queue(stack, 'Q');
        // WHEN/THEN
        test.throws(() => fn.addEventSource(new sources.SqsEventSource(q, {
            batchSize: 11
        })), /Maximum batch size must be between 1 and 10 inclusive \(given 11\)/);
        test.done();
    },
    'contains eventSourceMappingId after lambda binding'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const fn = new test_function_1.TestFunction(stack, 'Fn');
        const q = new sqs.Queue(stack, 'Q');
        const eventSource = new sources.SqsEventSource(q);
        // WHEN
        fn.addEventSource(eventSource);
        // THEN
        test.ok(eventSource.eventSourceMappingId);
        test.done();
    },
    'eventSourceMappingId throws error before binding to lambda'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const q = new sqs.Queue(stack, 'Q');
        const eventSource = new sources.SqsEventSource(q);
        // WHEN/THEN
        test.throws(() => eventSource.eventSourceMappingId, /SqsEventSource is not yet bound to an event source mapping/);
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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