# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['hrepr']

package_data = \
{'': ['*'], 'hrepr': ['style/*']}

setup_kwargs = {
    'name': 'hrepr',
    'version': '0.2.4',
    'description': 'Extensible HTML representation for Python objects.',
    'long_description': 'hrepr\n=====\n\n``hrepr`` is a package to create an HTML representation for Python\nobjects. It can output to Jupyter Notebooks and it can also generate\nstandalone pages with the representation of choice objects.\n\n.. image:: https://github.com/breuleux/hrepr/raw/master/images/hrepr1.png\n\n.. image:: https://github.com/breuleux/hrepr/raw/master/images/hrepr2.png\n\nSee the ``examples.ipynb`` notebook for more examples, but keep in mind that\nGitHub doesn\'t display it properly.\n\nInstall\n-------\n\n``hrepr`` requires at least Python 3.6\n\n.. code:: bash\n\n    $ pip3 install hrepr\n\nUsage\n-----\n\n.. code:: python\n\n    from hrepr import hrepr\n    obj = {\'potatoes\': [1, 2, 3], \'bananas\': {\'cantaloups\': 8}}\n\n    # Print the HTML representation of obj\n    print(hrepr(obj))\n\n    # Wrap the representation in <html><body> tags and embed the default\n    # css style files to produce a standalone page.\n    print(hrepr(obj).as_page())\n\nIn a Jupyter Notebook, return ``hrepr(obj)`` directly.\n\nCustom representations\n----------------------\n\nA custom representation for an object can be defined using the\n``__hrepr__`` method on the object, and if necessary, an\n``__hrepr_resources__`` method on the class. No dependency on ``hrepr``\nis necessary:\n\n.. code:: python\n\n    class RedGreen:\n        def __init__(self, r, g):\n            self.r = r\n            self.g = g\n\n        @classmethod\n        def __hrepr_resources__(cls, H):\n            return H.style(\'\'\'\n            .red { background-color: red; }\n            .green { background-color: green; }\n            \'\'\')\n\n        def __hrepr__(self, H, hrepr):\n            return H.div(\n                H.div[\'red\'](hrepr(self.r)),\n                H.div[\'green\'](hrepr(self.g))\n            )\n\n``__hrepr__`` receives two arguments:\n\n-  ``H`` is the HTML builder, which has the simple interface:\n   ``H.tag[\'klass\', ...](child, ..., attr=value, ...)`` to create the\n   tag ``<tag class=klass attr=value>child</tag>``. You can add more\n   classes, attributes and children, i.e. it is legal to write\n   ``H.a[\'cls1\'](href=\'blah\')[\'cls2\'](\'hello\', attr=value)``\n\n   -  Use ``H.raw(string)`` to insert an unescaped string (e.g. literal\n      HTML)\n   -  Use ``H.inline(...)`` to concatenate the children without wrapping\n      them in a tag.\n\n-  ``hrepr`` is a function that can be called recursively to get the\n   representation of the object\'s fields.\n\n``__hrepr_resources__`` is optional, but if it is defined, it should\nreturn one or a list of tags that should be inserted in the ``<head>``\nsection of the page in order to properly format the representation.\nThese can be ``<style>`` tags, ``<script>`` tags, or ``<link>`` tags.\n\n``__hrepr_short__`` is also optional, and should return a representation\nthat has a constant (small) size, e.g. the value of a class\'s ``name``\nfield, the length of the array, or something or other.\n\nConfigure the representation\n----------------------------\n\n``hrepr`` can take an arbitrary number of keyword arguments. Some of them are\ntreated specially, whereas others are stashed in the ``hrepr`` object passed to\n``__hrepr__`` methods and may be used to implement custom display options on\ncustom elements. The following keys are special (examples of their use can\nbe found in the ``examples.ipynb`` notebook):\n\n- **max_depth** limits how many layers of nested objects will be displayed.\n  Past that depth, objects are represented with their ``__hrepr_short__``\n  method.\n\n- **type_handlers** maps one or more types to functions with signature\n  ``(obj, H, hrepr) -> Tag`` which are used to generate the HTML structure to\n  display. These handlers override ``__hrepr__``.\n\n- **type_handlers_short**: same, but for short representations.\n\n- **resources** is one or a list of functions with signature ``(H,) -> [Tag]``,\n  meaning that they take the ``H`` constructor and return one or more ``style``\n  or ``script`` or ``link`` tags that are globally needed.\n\n- **preprocess** is a function with signature ``(obj, hrepr) -> obj``. It\n  must return an alternative object to display instead of ``obj``.\n\n- **postprocess** is a function with signature ``(obj, tag, H, hrepr) -> Tag``.\n  It must return an alternative or modified ``Tag`` object to display. For\n  example, it could return ``tag[\'highlight\']`` which is the syntax to add\n  the ``\'highlight\'`` class to a ``Tag``, in which case the postprocessor\n  is ostensibly highlighting the corresponding object. If the postprocessor\n  returns ``None``, the object will be represented by ``tag``, as it would\n  be if there was no postprocessor.\n\nCustom hrepr\n------------\n\nYou can also customize the ``hrepr`` function by subclassing the\n``HRepr`` or ``StdHRepr`` classes (the difference between the two is\nthat the latter defines default representations for several Python types\nlike ``list`` or ``dict`` whereas the former does not).\n\nYour subclass can override the following functions and fields:\n\n-  ``global_resources(H)`` should return one or a list of tags to\n   insert in ``<head>``.\n-  ``__default_handlers__()`` should return a dict that associates types to\n   handlers with the signature ``handler(obj, H, hrepr)``. When given\n   an object of a certain type, hrepr will look for it there first.\n-  ``__call__(obj)`` is the main representation function, and will be\n   called recursively for every object to represent.\n\n.. code:: python\n\n    from hrepr import StdHRepr\n\n    class MyRepr(StdHRepr):\n        def __default_handlers__(self):\n            dh = super().__default__handlers__()\n            return {**dh, int: self.repr_int}\n\n        def global_resources(self, H):\n            return H.style(".my-integer { color: fuchsia; }")\n\n        def repr_int(self, n, H, hrepr):\n            return H.span[\'my-integer\'](\'The number \', str(n))\n\n    def myrepr(obj):\n        # Call hrepr_with_resources to attach the resources to the\n        # return value, otherwise .as_page() will not work as\n        # intended.\n        return MyRepr().hrepr_with_resources(obj)\n\n    print(myrepr(10)) # <span class="my-integer">The number 10</span>\n    print(myrepr(10).as_page()) # This will include the style\n',
    'author': 'Olivier Breuleux',
    'author_email': 'breuleux@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': None,
    'packages': packages,
    'package_data': package_data,
    'python_requires': '>=3.6,<4.0',
}


setup(**setup_kwargs)
