import * as cloudwatch from 'aws-cdk-lib/aws-cloudwatch';
import * as events from 'aws-cdk-lib/aws-events';
import * as iam from 'aws-cdk-lib/aws-iam';
import * as logs from 'aws-cdk-lib/aws-logs';
import * as s3 from 'aws-cdk-lib/aws-s3';
import * as cdk from 'aws-cdk-lib';
import * as constructs from 'constructs';
import { JobExecutable } from '.';
import { IConnection } from './connection';
import { ISecurityConfiguration } from './security-configuration';
/**
 * (experimental) The type of predefined worker that is allocated when a job runs.
 *
 * If you need to use a WorkerType that doesn't exist as a static member, you
 * can instantiate a `WorkerType` object, e.g: `WorkerType.of('other type')`.
 *
 * @experimental
 */
export declare class WorkerType {
    /**
     * (experimental) Each worker provides 4 vCPU, 16 GB of memory and a 50GB disk, and 2 executors per worker.
     *
     * @experimental
     */
    static readonly STANDARD: WorkerType;
    /**
     * (experimental) Each worker maps to 1 DPU (4 vCPU, 16 GB of memory, 64 GB disk), and provides 1 executor per worker.
     *
     * Suitable for memory-intensive jobs.
     *
     * @experimental
     */
    static readonly G_1X: WorkerType;
    /**
     * (experimental) Each worker maps to 2 DPU (8 vCPU, 32 GB of memory, 128 GB disk), and provides 1 executor per worker.
     *
     * Suitable for memory-intensive jobs.
     *
     * @experimental
     */
    static readonly G_2X: WorkerType;
    /**
     * (experimental) Custom worker type.
     *
     * @param workerType custom worker type.
     * @experimental
     */
    static of(workerType: string): WorkerType;
    /**
     * (experimental) The name of this WorkerType, as expected by Job resource.
     *
     * @experimental
     */
    readonly name: string;
    private constructor();
}
/**
 * (experimental) Job states emitted by Glue to CloudWatch Events.
 *
 * @see https://docs.aws.amazon.com/AmazonCloudWatch/latest/events/EventTypes.html#glue-event-types for more information.
 * @experimental
 */
export declare enum JobState {
    /**
     * (experimental) State indicating job run succeeded.
     *
     * @experimental
     */
    SUCCEEDED = "SUCCEEDED",
    /**
     * (experimental) State indicating job run failed.
     *
     * @experimental
     */
    FAILED = "FAILED",
    /**
     * (experimental) State indicating job run timed out.
     *
     * @experimental
     */
    TIMEOUT = "TIMEOUT",
    /**
     * (experimental) State indicating job is starting.
     *
     * @experimental
     */
    STARTING = "STARTING",
    /**
     * (experimental) State indicating job is running.
     *
     * @experimental
     */
    RUNNING = "RUNNING",
    /**
     * (experimental) State indicating job is stopping.
     *
     * @experimental
     */
    STOPPING = "STOPPING",
    /**
     * (experimental) State indicating job stopped.
     *
     * @experimental
     */
    STOPPED = "STOPPED"
}
/**
 * (experimental) The Glue CloudWatch metric type.
 *
 * @see https://docs.aws.amazon.com/glue/latest/dg/monitoring-awsglue-with-cloudwatch-metrics.html
 * @experimental
 */
export declare enum MetricType {
    /**
     * (experimental) A value at a point in time.
     *
     * @experimental
     */
    GAUGE = "gauge",
    /**
     * (experimental) An aggregate number.
     *
     * @experimental
     */
    COUNT = "count"
}
/**
 * (experimental) Interface representing a created or an imported {@link Job}.
 *
 * @experimental
 */
export interface IJob extends cdk.IResource, iam.IGrantable {
    /**
     * (experimental) The name of the job.
     *
     * @experimental
     * @attribute true
     */
    readonly jobName: string;
    /**
     * (experimental) The ARN of the job.
     *
     * @experimental
     * @attribute true
     */
    readonly jobArn: string;
    /**
     * (experimental) Defines a CloudWatch event rule triggered when something happens with this job.
     *
     * @see https://docs.aws.amazon.com/AmazonCloudWatch/latest/events/EventTypes.html#glue-event-types
     * @experimental
     */
    onEvent(id: string, options?: events.OnEventOptions): events.Rule;
    /**
     * (experimental) Defines a CloudWatch event rule triggered when this job moves to the input jobState.
     *
     * @see https://docs.aws.amazon.com/AmazonCloudWatch/latest/events/EventTypes.html#glue-event-types
     * @experimental
     */
    onStateChange(id: string, jobState: JobState, options?: events.OnEventOptions): events.Rule;
    /**
     * (experimental) Defines a CloudWatch event rule triggered when this job moves to the SUCCEEDED state.
     *
     * @see https://docs.aws.amazon.com/AmazonCloudWatch/latest/events/EventTypes.html#glue-event-types
     * @experimental
     */
    onSuccess(id: string, options?: events.OnEventOptions): events.Rule;
    /**
     * (experimental) Defines a CloudWatch event rule triggered when this job moves to the FAILED state.
     *
     * @see https://docs.aws.amazon.com/AmazonCloudWatch/latest/events/EventTypes.html#glue-event-types
     * @experimental
     */
    onFailure(id: string, options?: events.OnEventOptions): events.Rule;
    /**
     * (experimental) Defines a CloudWatch event rule triggered when this job moves to the TIMEOUT state.
     *
     * @see https://docs.aws.amazon.com/AmazonCloudWatch/latest/events/EventTypes.html#glue-event-types
     * @experimental
     */
    onTimeout(id: string, options?: events.OnEventOptions): events.Rule;
    /**
     * (experimental) Create a CloudWatch metric.
     *
     * @param metricName name of the metric typically prefixed with `glue.driver.`, `glue.<executorId>.` or `glue.ALL.`.
     * @param type the metric type.
     * @param props metric options.
     * @see https://docs.aws.amazon.com/glue/latest/dg/monitoring-awsglue-with-cloudwatch-metrics.html
     * @experimental
     */
    metric(metricName: string, type: MetricType, props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * (experimental) Create a CloudWatch Metric indicating job success.
     *
     * @experimental
     */
    metricSuccess(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * (experimental) Create a CloudWatch Metric indicating job failure.
     *
     * @experimental
     */
    metricFailure(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * (experimental) Create a CloudWatch Metric indicating job timeout.
     *
     * @experimental
     */
    metricTimeout(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
}
declare abstract class JobBase extends cdk.Resource implements IJob {
    abstract readonly jobArn: string;
    abstract readonly jobName: string;
    abstract readonly grantPrincipal: iam.IPrincipal;
    /**
     * (experimental) Create a CloudWatch Event Rule for this Glue Job when it's in a given state.
     *
     * @param id construct id.
     * @param options event options.
     * @see https://docs.aws.amazon.com/AmazonCloudWatch/latest/events/EventTypes.html#glue-event-types
     * @experimental
     */
    onEvent(id: string, options?: events.OnEventOptions): events.Rule;
    /**
     * (experimental) Create a CloudWatch Event Rule for the transition into the input jobState.
     *
     * @param id construct id.
     * @param jobState the job state.
     * @param options optional event options.
     * @experimental
     */
    onStateChange(id: string, jobState: JobState, options?: events.OnEventOptions): events.Rule;
    /**
     * (experimental) Create a CloudWatch Event Rule matching JobState.SUCCEEDED.
     *
     * @param id construct id.
     * @param options optional event options.
     * @experimental
     */
    onSuccess(id: string, options?: events.OnEventOptions): events.Rule;
    /**
     * (experimental) Return a CloudWatch Event Rule matching FAILED state.
     *
     * @param id construct id.
     * @param options optional event options.
     * @experimental
     */
    onFailure(id: string, options?: events.OnEventOptions): events.Rule;
    /**
     * (experimental) Return a CloudWatch Event Rule matching TIMEOUT state.
     *
     * @param id construct id.
     * @param options optional event options.
     * @experimental
     */
    onTimeout(id: string, options?: events.OnEventOptions): events.Rule;
    /**
     * (experimental) Create a CloudWatch metric.
     *
     * @param metricName name of the metric typically prefixed with `glue.driver.`, `glue.<executorId>.` or `glue.ALL.`.
     * @param type the metric type.
     * @param props metric options.
     * @see https://docs.aws.amazon.com/glue/latest/dg/monitoring-awsglue-with-cloudwatch-metrics.html
     * @experimental
     */
    metric(metricName: string, type: MetricType, props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * (experimental) Return a CloudWatch Metric indicating job success.
     *
     * This metric is based on the Rule returned by no-args onSuccess() call.
     *
     * @experimental
     */
    metricSuccess(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * (experimental) Return a CloudWatch Metric indicating job failure.
     *
     * This metric is based on the Rule returned by no-args onFailure() call.
     *
     * @experimental
     */
    metricFailure(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * (experimental) Return a CloudWatch Metric indicating job timeout.
     *
     * This metric is based on the Rule returned by no-args onTimeout() call.
     *
     * @experimental
     */
    metricTimeout(props?: cloudwatch.MetricOptions): cloudwatch.Metric;
    /**
     * Creates or retrieves a singleton event rule for the input job state for use with the metric JobState methods.
     *
     * @param id construct id.
     * @param jobState the job state.
     * @private
     */
    private metricJobStateRule;
}
/**
 * (experimental) Properties for enabling Spark UI monitoring feature for Spark-based Glue jobs.
 *
 * @see https://docs.aws.amazon.com/glue/latest/dg/aws-glue-programming-etl-glue-arguments.html
 * @experimental
 */
export interface SparkUIProps {
    /**
     * (experimental) Enable Spark UI.
     *
     * @experimental
     */
    readonly enabled: boolean;
    /**
     * (experimental) The bucket where the Glue job stores the logs.
     *
     * @default a new bucket will be created.
     * @experimental
     */
    readonly bucket?: s3.IBucket;
    /**
     * (experimental) The path inside the bucket (objects prefix) where the Glue job stores the logs.
     *
     * @default '/' - the logs will be written at the root of the bucket
     * @experimental
     */
    readonly prefix?: string;
}
/**
 * (experimental) The Spark UI logging location.
 *
 * @see https://docs.aws.amazon.com/glue/latest/dg/aws-glue-programming-etl-glue-arguments.html
 * @experimental
 */
export interface SparkUILoggingLocation {
    /**
     * (experimental) The bucket where the Glue job stores the logs.
     *
     * @experimental
     */
    readonly bucket: s3.IBucket;
    /**
     * (experimental) The path inside the bucket (objects prefix) where the Glue job stores the logs.
     *
     * @default '/' - the logs will be written at the root of the bucket
     * @experimental
     */
    readonly prefix?: string;
}
/**
 * (experimental) Properties for enabling Continuous Logging for Glue Jobs.
 *
 * @see https://docs.aws.amazon.com/glue/latest/dg/aws-glue-programming-etl-glue-arguments.html
 * @experimental
 */
export interface ContinuousLoggingProps {
    /**
     * (experimental) Enable continouous logging.
     *
     * @experimental
     */
    readonly enabled: boolean;
    /**
     * (experimental) Specify a custom CloudWatch log group name.
     *
     * @default - a log group is created with name `/aws-glue/jobs/logs-v2/`.
     * @experimental
     */
    readonly logGroup?: logs.ILogGroup;
    /**
     * (experimental) Specify a custom CloudWatch log stream prefix.
     *
     * @default - the job run ID.
     * @experimental
     */
    readonly logStreamPrefix?: string;
    /**
     * (experimental) Filter out non-useful Apache Spark driver/executor and Apache Hadoop YARN heartbeat log messages.
     *
     * @default true
     * @experimental
     */
    readonly quiet?: boolean;
    /**
     * (experimental) Apply the provided conversion pattern.
     *
     * This is a Log4j Conversion Pattern to customize driver and executor logs.
     *
     * @default `%d{yy/MM/dd HH:mm:ss} %p %c{1}: %m%n`
     * @experimental
     */
    readonly conversionPattern?: string;
}
/**
 * (experimental) Attributes for importing {@link Job}.
 *
 * @experimental
 */
export interface JobAttributes {
    /**
     * (experimental) The name of the job.
     *
     * @experimental
     */
    readonly jobName: string;
    /**
     * (experimental) The IAM role assumed by Glue to run this job.
     *
     * @default - undefined
     * @experimental
     */
    readonly role?: iam.IRole;
}
/**
 * (experimental) Construction properties for {@link Job}.
 *
 * @experimental
 */
export interface JobProps {
    /**
     * (experimental) The job's executable properties.
     *
     * @experimental
     */
    readonly executable: JobExecutable;
    /**
     * (experimental) The name of the job.
     *
     * @default - a name is automatically generated
     * @experimental
     */
    readonly jobName?: string;
    /**
     * (experimental) The description of the job.
     *
     * @default - no value
     * @experimental
     */
    readonly description?: string;
    /**
     * (experimental) The number of AWS Glue data processing units (DPUs) that can be allocated when this job runs.
     *
     * Cannot be used for Glue version 2.0 and later - workerType and workerCount should be used instead.
     *
     * @default - 10 when job type is Apache Spark ETL or streaming, 0.0625 when job type is Python shell
     * @experimental
     */
    readonly maxCapacity?: number;
    /**
     * (experimental) The maximum number of times to retry this job after a job run fails.
     *
     * @default 0
     * @experimental
     */
    readonly maxRetries?: number;
    /**
     * (experimental) The maximum number of concurrent runs allowed for the job.
     *
     * An error is returned when this threshold is reached. The maximum value you can specify is controlled by a service limit.
     *
     * @default 1
     * @experimental
     */
    readonly maxConcurrentRuns?: number;
    /**
     * (experimental) The number of minutes to wait after a job run starts, before sending a job run delay notification.
     *
     * @default - no delay notifications
     * @experimental
     */
    readonly notifyDelayAfter?: cdk.Duration;
    /**
     * (experimental) The maximum time that a job run can consume resources before it is terminated and enters TIMEOUT status.
     *
     * @default cdk.Duration.hours(48)
     * @experimental
     */
    readonly timeout?: cdk.Duration;
    /**
     * (experimental) The type of predefined worker that is allocated when a job runs.
     *
     * @default - differs based on specific Glue version
     * @experimental
     */
    readonly workerType?: WorkerType;
    /**
     * (experimental) The number of workers of a defined {@link WorkerType} that are allocated when a job runs.
     *
     * @default - differs based on specific Glue version/worker type
     * @experimental
     */
    readonly workerCount?: number;
    /**
     * (experimental) The {@link Connection}s used for this job.
     *
     * Connections are used to connect to other AWS Service or resources within a VPC.
     *
     * @default [] - no connections are added to the job
     * @experimental
     */
    readonly connections?: IConnection[];
    /**
     * (experimental) The {@link SecurityConfiguration} to use for this job.
     *
     * @default - no security configuration.
     * @experimental
     */
    readonly securityConfiguration?: ISecurityConfiguration;
    /**
     * (experimental) The default arguments for this job, specified as name-value pairs.
     *
     * @default - no arguments
     * @see https://docs.aws.amazon.com/glue/latest/dg/aws-glue-programming-etl-glue-arguments.html for a list of reserved parameters
     * @experimental
     */
    readonly defaultArguments?: {
        [key: string]: string;
    };
    /**
     * (experimental) The tags to add to the resources on which the job runs.
     *
     * @default {} - no tags
     * @experimental
     */
    readonly tags?: {
        [key: string]: string;
    };
    /**
     * (experimental) The IAM role assumed by Glue to run this job.
     *
     * If providing a custom role, it needs to trust the Glue service principal (glue.amazonaws.com) and be granted sufficient permissions.
     *
     * @default - a role is automatically generated
     * @see https://docs.aws.amazon.com/glue/latest/dg/getting-started-access.html
     * @experimental
     */
    readonly role?: iam.IRole;
    /**
     * (experimental) Enables the collection of metrics for job profiling.
     *
     * @default - no profiling metrics emitted.
     * @see `--enable-metrics` at https://docs.aws.amazon.com/glue/latest/dg/aws-glue-programming-etl-glue-arguments.html
     * @experimental
     */
    readonly enableProfilingMetrics?: boolean;
    /**
     * (experimental) Enables the Spark UI debugging and monitoring with the specified props.
     *
     * @default - Spark UI debugging and monitoring is disabled.
     * @see https://docs.aws.amazon.com/glue/latest/dg/aws-glue-programming-etl-glue-arguments.html
     * @experimental
     */
    readonly sparkUI?: SparkUIProps;
    /**
     * (experimental) Enables continuous logging with the specified props.
     *
     * @default - continuous logging is disabled.
     * @see https://docs.aws.amazon.com/glue/latest/dg/aws-glue-programming-etl-glue-arguments.html
     * @experimental
     */
    readonly continuousLogging?: ContinuousLoggingProps;
}
/**
 * (experimental) A Glue Job.
 *
 * @experimental
 */
export declare class Job extends JobBase {
    /**
     * (experimental) Creates a Glue Job.
     *
     * @param scope The scope creating construct (usually `this`).
     * @param id The construct's id.
     * @param attrs Import attributes.
     * @experimental
     */
    static fromJobAttributes(scope: constructs.Construct, id: string, attrs: JobAttributes): IJob;
    /**
     * (experimental) The ARN of the job.
     *
     * @experimental
     */
    readonly jobArn: string;
    /**
     * (experimental) The name of the job.
     *
     * @experimental
     */
    readonly jobName: string;
    /**
     * (experimental) The IAM role Glue assumes to run this job.
     *
     * @experimental
     */
    readonly role: iam.IRole;
    /**
     * (experimental) The principal this Glue Job is running as.
     *
     * @experimental
     */
    readonly grantPrincipal: iam.IPrincipal;
    /**
     * (experimental) The Spark UI logs location if Spark UI monitoring and debugging is enabled.
     *
     * @see https://docs.aws.amazon.com/glue/latest/dg/aws-glue-programming-etl-glue-arguments.html
     * @experimental
     */
    readonly sparkUILoggingLocation?: SparkUILoggingLocation;
    /**
     * @experimental
     */
    constructor(scope: constructs.Construct, id: string, props: JobProps);
    /**
     * Check no usage of reserved arguments.
     *
     * @see https://docs.aws.amazon.com/glue/latest/dg/aws-glue-programming-etl-glue-arguments.html
     */
    private checkNoReservedArgs;
    private executableArguments;
    private setupSparkUI;
    private setupContinuousLogging;
    private codeS3ObjectUrl;
}
export {};
