"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Workload = void 0;
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const cdk = require("@aws-cdk/core");
const analytics_1 = require("./analytics");
const iam_roles_1 = require("./iam-roles");
const network_1 = require("./network");
const scheduler_1 = require("./scheduler");
const storage_1 = require("./storage");
/**
 * @experimental
 */
class Workload extends cdk.Construct {
    /**
     * @experimental
     */
    constructor(scope, id, props = {}) {
        var _a, _b, _c, _d;
        super(scope, id);
        const stack = cdk.Stack.of(this);
        const s3InstallBucket = (_a = props.s3InstallBucket) !== null && _a !== void 0 ? _a : 'solutions-reference';
        const s3InstallFolder = (_b = props.s3InstallFolder) !== null && _b !== void 0 ? _b : 'scale-out-computing-on-aws/v2.5.0';
        // network construct
        const network = new network_1.Network(stack, 'SocaNetwork');
        // assets construct
        // const assets = new Assets(stack, 'S3Assets');
        // iam roles
        new iam_roles_1.IamRoles(stack, 'IamRoles', {
            s3InstallBucketName: s3InstallBucket,
            network,
        });
        // compute node security group
        const computeNodeSecurityGroup = new aws_ec2_1.SecurityGroup(this, 'ComputeNodeSecurityGroup', {
            vpc: network.vpc,
        });
        /**
         * Allow all traffic internally
         */
        computeNodeSecurityGroup.connections.allowInternally(aws_ec2_1.Port.allTraffic());
        const schedulerSecurityGroup = new aws_ec2_1.SecurityGroup(this, 'SchedulerSecurityGroup', {
            vpc: network.vpc,
        });
        /**
         * SchedulerInboundRule
         * Allow all traffic from computeNodeSecurityGroup to schedulerSecurityGroup
         */
        schedulerSecurityGroup.connections.allowFrom(computeNodeSecurityGroup, aws_ec2_1.Port.allTraffic());
        /**
         * SchedulerInboundRuleAllowClientIP
         * SchedulerInboundRuleAllowClientIPHTTPS
         * SchedulerInboundRuleAllowClientIPHTTP
         * 1. Allow SSH traffic from client IP to master host
         * 2. Allow HTTP traffic from client IP to ELB
         * 3. Allow HTTPS traffic from client IP to ELB
         */
        if (props.clientIpCidr) {
            schedulerSecurityGroup.connections.allowFrom(aws_ec2_1.Peer.ipv4(props.clientIpCidr), aws_ec2_1.Port.tcp(22));
            schedulerSecurityGroup.connections.allowFrom(aws_ec2_1.Peer.ipv4(props.clientIpCidr), aws_ec2_1.Port.tcp(80));
            schedulerSecurityGroup.connections.allowFrom(aws_ec2_1.Peer.ipv4(props.clientIpCidr), aws_ec2_1.Port.tcp(443));
        }
        /**
         * Allow traffic between Master agent and compute nodes
         */
        computeNodeSecurityGroup.connections.allowFrom(schedulerSecurityGroup, aws_ec2_1.Port.allTraffic());
        /**
         * Allow ELB healtcheck to communicate with web ui on master host
         */
        schedulerSecurityGroup.connections.allowInternally(aws_ec2_1.Port.tcp(8443));
        const storage = new storage_1.EfsStorage(stack, 'EfsStorage', {
            clusterId: network.clusterId,
            vpc: network.vpc,
            schedulerSecurityGroup,
            computeNodeSecurityGroup,
        });
        // add elasticsearch stack
        new analytics_1.Analytics(stack, 'Analytics', {
            vpc: network.vpc,
            schedulerSecurityGroup: schedulerSecurityGroup,
            clusterId: network.clusterId,
        });
        new scheduler_1.Scheduler(stack, 'Scheduler', {
            s3InstallBucket,
            s3InstallFolder,
            schedulerSecurityGroup,
            network,
            storage,
            instanceType: new aws_ec2_1.InstanceType('m5.large'),
            ldapUserName: (_c = props.ldapUserName) !== null && _c !== void 0 ? _c : 'ldapUserName',
            ldapUserPassword: (_d = props.ldapUserPassword) !== null && _d !== void 0 ? _d : 'ldapUserPassword!123',
        });
    }
}
exports.Workload = Workload;
//# sourceMappingURL=data:application/json;base64,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