"""
Asymptote extension for Markdown (e.g. for mkdocs) :
Renders the output inline, eliminating the need to configure an output
directory.

Supports outputs types of SVG and PNG. The output will be taken from the
filename specified in the tag, if given, or. Example:

in SVG:

```asy
import math;

size(6cm,0);

add(grid(6,5,lightgray));

pair pA=(1,1), pB=(5,1), pC=(4,4);
 
draw(pA--pB--pC--cycle);
dot("$A$",pA,dir(pC--pA,pB--pA));  
dot("$B$",pB,dir(pC--pB,pA--pB));  
dot("$C$",pC,dir(pA--pC,pB--pC));
```

```asy svg
import math;

size(6cm,0);

add(grid(6,5,lightgray));

pair pA=(1,1), pB=(5,1), pC=(4,4);
 
draw(pA--pB--pC--cycle);
dot("$A$",pA,dir(pC--pA,pB--pA));  
dot("$B$",pB,dir(pC--pB,pA--pB));  
dot("$C$",pC,dir(pA--pC,pB--pC));
```

in PNG:

```asy png
import math;

size(6cm,0);

add(grid(6,5,lightgray));

pair pA=(1,1), pB=(5,1), pC=(4,4);
 
draw(pA--pB--pC--cycle);
dot("$A$",pA,dir(pC--pA,pB--pA));  
dot("$B$",pB,dir(pC--pB,pA--pB));  
dot("$C$",pC,dir(pA--pC,pB--pC));
```

Requires the Asymptote library (https://asymptote.sourceforge.io/) and python 3

[Rodrigo SCHWENCKE > mkdocs-graphviz](https://gitlab.com/rodrigo.schwencke/mkdocs-graphviz)
"""

import re
import markdown
import subprocess
import base64

# Global vars
# BLOCK_RE_GRAVE_ACCENT = re.compile(
#         r'^[\s]*```graphviz[\s]+(?P<command>\w+)\s+(?P<filename>[^\s]+)\s*\n(?P<content>.*?)```\n$',
#     re.MULTILINE | re.DOTALL)

BLOCK_RE_GRAVE_ACCENT_ASY = re.compile(
        r'^[ 	]*```asy\n(?P<content>.*?)```\s*$',
    re.MULTILINE | re.DOTALL)

BLOCK_RE_GRAVE_ACCENT_ASY_SVG = re.compile(
        r'^[ 	]*```asy[ 	]+svg\n(?P<content>.*?)```\s*$',
    re.MULTILINE | re.DOTALL)

BLOCK_RE_GRAVE_ACCENT_ASY_PNG = re.compile(
        r'^[ 	]*```asy[ 	]+png\n(?P<content>.*?)```\s*$',
    re.MULTILINE | re.DOTALL)

ASY_COMMAND = 0

# Command whitelist
# SUPPORTED_COMMAMDS = ['dot', 'neato', 'fdp', 'sfdp', 'twopi', 'circo']


class MkdocsAsyExtension(markdown.Extension):

    def extendMarkdown(self, md, md_globals):
        """ Add MkdocsAsyPreprocessor to the Markdown instance. """
        md.registerExtension(self)

        md.preprocessors.add('asy_block',
                             MkdocsAsyPreprocessor(md),
                             "_begin")


class MkdocsAsyPreprocessor(markdown.preprocessors.Preprocessor):

    def __init__(self, md):
        super(MkdocsAsyPreprocessor, self).__init__(md)

#     def repair_broken_svg_in(self, output):
#         """Returns a repaired svg output. Indeed:
#         The Original svg ouput is broken in two places:
#         - in the DOCTYPE, after "\\EN". Does not break the code, but still
#         - in the svg tag, after the 'height' attribute; THIS BREAKS THE CODE AND HAD TO BE REPAIRED
#         """
#         encoding='utf-8'
#         output = output.decode(encoding)
#         lines = output.split("\n")
#         newLines = []
#         searchText = "Generated by graphviz"
#         for i in range(len(lines)):
#             if i+3 <= len(lines)-1 and ( (searchText in lines[i+1]) or (searchText in lines[i+2]) or (searchText in lines[i+3]) ) :
#                 continue
#             if i>=3 and ("<svg" in lines[i-1] and searchText in lines[i-4]):
#                 continue
#             if i>=3 and ("<svg" in lines[i] and searchText in lines[i-3]):
#                 newLines.append(lines[i]+lines[i+1])
#             else:
#                 newLines.append(lines[i])
#         newOutput = "\n".join(newLines)
#         xmlHeaders = """<?xml version="1.0" encoding="UTF-8" standalone="no"?>\n
# <!DOCTYPE svg PUBLIC "-//W3C//DTD SVG 1.1//EN" "http://www.w3.org/Graphics/SVG/1.1/DTD/svg11.dtd">\n"""
#         newOutput = xmlHeaders + newOutput
#         return newOutput

    def read_block(self, text:str)->(str, int) or (None, -1):
        """Returns a tuple:
        - the asy block, if exists, and
        - a code integer to caracterize the command : 
            0 for 'asy' command,
            1 for 'asy svg' command,
            2 for 'asy png' command)
        or (None, -1), if not a recognised asy command block"""
        blocks = [BLOCK_RE_GRAVE_ACCENT_ASY.search(text),
                  BLOCK_RE_GRAVE_ACCENT_ASY_PNG.search(text)]
        for i in range(len(blocks)):
            if blocks[i] is not None:
                return blocks[i], i
        return None, -1

    def get_decalage(self, command:str, text:str)->int:
        """Renvoie le décalage (nombre d'espaces) où commencent les ``` dans la ligne ```command ...
        Cela suppose que le 'text' réellement la commande, ce qui est censé être le cas lros de l'utilisation de cette fonction
        """
        # command = 'dot' or 'graphviz dot' or 'graphviz neato' or etc..
        i_command = text.find("```"+command)
        i_previous_linefeed = text[:i_command].rfind("\n")
        decalage = i_command - i_previous_linefeed-1
        return decalage

    def run(self, lines):
        """ Match and generate asy code blocks."""

        text = "\n".join(lines)
        print("TEXT=", text)
        while 1:
            m, block_type = self.read_block(text)
            print("m=", m)
            if not m:
                break
            else:
                # if block_type == ASY_COMMAND: # General Graphviz command
                #     command = m.group('command')
                #      # Whitelist command, prevent command injection.
                #     # if command not in SUPPORTED_COMMAMDS:
                #     #     raise Exception('Command not supported: %s' % command)
                #     filename = m.group('filename')
                #     decalage = self.get_decalage("graphviz "+command, text)
                # else: # DOT command
                    # if block_type == 0: # asy command
                    #     print("ASY COMMAND DETECTED")
                    #     filename = "noname.svg"
                    #     command = "asy"
                    #     decalage = self.get_decalage(command, text)

                print("ASY COMMAND DETECTED")
                filename = "othername.svg"
                command = "asy"
                decalage = self.get_decalage(command, text)

                filetype = filename[filename.rfind('.')+1:]
                args = [command, '-'+filetype+'emulation --tex latex']
                content = m.group('content')
                print("CONTENT=", content)

                try:
                    proc = subprocess.Popen(
                        args,
                        stdin=subprocess.PIPE,
                        stderr=subprocess.PIPE,
                        stdout=subprocess.PIPE)
                    proc.stdin.write(content.encode('utf-8'))
                    output, err = proc.communicate()

                    if filetype == 'svg':
                        # output = self.repair_broken_svg_in(output)
                        encoding = 'base64'
                        output = output.encode('utf-8')
                        output = base64.b64encode(output).decode('utf-8')
                        data_url_filetype = 'svg+xml'
                        data_path = "data:image/%s;%s,%s" % (
                            data_url_filetype,
                            encoding,
                            output)
                        img = " "*decalage+"![" + filename + "](" + data_path + ")"
                    
                    if filetype == 'png':
                        data_url_filetype = 'png'
                        encoding = 'base64'
                        output = base64.b64encode(output).decode('utf-8')
                        data_path = "data:image/%s;%s,%s" % (
                            data_url_filetype,
                            encoding,
                            output)
                        img = " "*decalage+"![" + filename + "](" + data_path + ")"

                    text = '%s\n%s\n%s' % (
                        text[:m.start()], img, text[m.end():])

                except Exception as e:
                        err = str(e) + ' : ' + str(args)
                        return (
                            '<pre>Error : ' + err + '</pre>'
                            '<pre>' + content + '</pre>').split('\n')

        return text.split("\n")

def makeExtension(*args, **kwargs):
    return MkdocsAsyExtension(*args, **kwargs)
