# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['universi', 'universi.structure']

package_data = \
{'': ['*']}

install_requires = \
['fastapi>=0.96.1', 'pydantic>=1.10.0,<2.0.0', 'typing-extensions']

setup_kwargs = {
    'name': 'universi',
    'version': '1.5.0',
    'description': 'Modern Stripe-like API versioning in FastAPI',
    'long_description': '# Universi\n\nModern [Stripe-like](https://stripe.com/blog/api-versioning) API versioning in FastAPI\n\n---\n\n<p align="center">\n<a href="https://github.com/ovsyanka83/universi/actions?query=workflow%3ATests+event%3Apush+branch%3Amain" target="_blank">\n    <img src="https://github.com/Ovsyanka83/universi/actions/workflows/test.yaml/badge.svg?branch=main&event=push" alt="Test">\n</a>\n<a href="https://codecov.io/gh/ovsyanka83/universi" target="_blank">\n    <img src="https://img.shields.io/codecov/c/github/ovsyanka83/universi?color=%2334D058" alt="Coverage">\n</a>\n<a href="https://pypi.org/project/universi/" target="_blank">\n    <img alt="PyPI" src="https://img.shields.io/pypi/v/universi?color=%2334D058&label=pypi%20package" alt="Package version">\n</a>\n<a href="https://pypi.org/project/universi/" target="_blank">\n    <img src="https://img.shields.io/pypi/pyversions/universi?color=%2334D058" alt="Supported Python versions">\n</a>\n</p>\n\n## Installation\n\n```bash\npip install universi\n```\n\n<!---\n# TODO: Note that we don\'t handle "from .schemas import Schema as OtherSchema" case\n# TODO: Need to validate that the user doesn\'t use versioned schemas instead of the latest ones\n-->\n\n## Who is this for?\n\nUniversi allows you to support a single version of your code, auto-generating the code/routes for older versions. You keep versioning encapsulated in small and independent "version change" modules while your business logic knows nothing about versioning.\n\nIts approach will be useful if you want to:\n\n1. Support many API versions for a long time\n2. Effortlessly backport features and bugfixes to all of your versions\n\nOtherwise, more conventional methods of API versioning may be preferable.\n\n## Tutorial\n\nThis guide provides a step-by-step tutorial for setting up automatic API versioning using Universi library. I will illustrate this with an example of a User API, where we will be implementing changes to a User\'s address.\n\n### A dummy setup\n\nHere is an initial API setup where the User has a single address. We will be implementing two routes - one for creating a user and another for retrieving user details. We\'ll be using "int" for ID for simplicity.\n\nThe first API you come up with usually doesn\'t require more than one address -- why bother?\n\nSo we create our file with schemas:\n\n```python\nfrom pydantic import BaseModel\n\n\nclass UserCreateRequest(BaseModel):\n    address: str\n\nclass UserResource(BaseModel):\n    id: int\n    address: str\n```\n\nAnd we create our file with routes:\n\n```python\nfrom versions.latest.users import UserCreateRequest, UserResource\nfrom universi import VersionedAPIRouter\n\nrouter = VersionedAPIRouter()\n\n@router.post("/users", response_model=UserResource)\nasync def create_user(payload: UserCreateRequest):\n    return {\n        "id": 83,\n        "address": payload.address,\n    }\n\n@router.get("/users/{user_id}", response_model=UserResource)\nasync def get_user(user_id: int):\n    return {\n        "id": user_id,\n        "address": "123 Example St",\n    }\n```\n\n### Turning address into a list\n\nDuring our development, we have realized that the initial API design was wrong and that addresses should have always been a list because the user wants to have multiple addresses to choose from so now we have to change the type of the "address" field to the list of strings.\n\n```python\nfrom pydantic import BaseModel\nfrom pydantic import Field\n\n\nclass UserCreateRequest(BaseModel):\n    addresses: list[str] = Field(min_items=1)\n\nclass UserResource(BaseModel):\n    id: int\n    addresses: list[str]\n```\n\n```python\n@router.post("/users", response_model=UserResource)\nasync def create_user(payload: UserCreateRequest):\n    return {\n        "id": 83,\n        "addresses": payload.addresses,\n    }\n\n@router.get("/users/{user_id}", response_model=UserResource)\nasync def get_user(user_id: int):\n    return {\n        "id": user_id,\n        "addresses": ["123 Example St", "456 Main St"],\n    }\n\n```\n\nBut every user of ours will now have their API integration broken. To prevent that, we have to introduce API versioning. There aren\'t many methods of doing that. Most of them force you to either duplicate your schemas, your endpoints, or your entire app instance. And it makes sense, really: duplication is the only way to make sure that you will not break old versions with your new versions; the bigger the piece you duplicating -- the safer. Of course, the safest being duplicating the entire app instance and even having a separate database. But that is expensive and makes it either impossible to make breaking changes often or to support many versions. As a result, either you need infinite resources, very long development cycles, or your users will need to often migrate from version to version.\n\nStripe has come up [with a solution](https://stripe.com/blog/api-versioning): let\'s have one latest app version whose responses get migrated to older versions and let\'s describe changes between these versions using migrations. This approach allows them to keep versions for **years** without dropping them. Obviously, each breaking change is still bad and each version still makes our system more complex and expensive, but their approach gives us a chance to minimize that. Additionally, it allows us backport features and bugfixes to older versions. However, you will also be backporting bugs, which is a sad consequence of eliminating duplication.\n\nUniversi is heavily inspired by this approach so let\'s continue our tutorial and now try to combine the two versions we created using versioning.\n\n### Creating the Migration\n\nWe need to create a migration to handle changes between these versions. For every endpoint whose `response_model` is `UserResource`, this migration will convert the list of addresses back to a single address when migrating to the previous version. Yes, migrating **back**: you might be used to database migrations where we write upgrade migration and downgrade migration but here our goal is to have an app of latest version and to describe what older versions looked like in comparison to it. That way the old versions are frozen in migrations and you can **almost** safely forget about them.\n\n```python\nfrom pydantic import Field\nfrom universi.structure import (\n    schema,\n    VersionChange,\n    convert_response_to_previous_version_for,\n)\n\nclass ChangeAddressToList(VersionChange):\n    description = (\n        "Change user address to a list of strings to "\n        "allow the user to specify multiple addresses"\n    )\n    instructions_to_migrate_to_previous_version = (\n        # You should use schema inheritance if you don\'t want to repeat yourself in such cases\n        schema(UserCreateRequest).field("addresses").didnt_exist,\n        schema(UserCreateRequest).field("address").existed_with(type=str, info=Field()),\n        schema(UserResource).field("addresses").didnt_exist,\n        schema(UserResource).field("address").existed_with(type=str, info=Field()),\n    )\n\n    @convert_response_to_previous_version_for(UserResource)\n    def change_addresses_to_single_item(cls, data: dict[str, Any]) -> None:\n        data["address"] = data.pop("addresses")[0]\n    \n    @schema(UserCreateRequest).had_property("addresses")\n    def addresses_property(parsed_schema):\n        return [parsed_schema.address]\n\n```\n\nSee how we are popping the first address from the list? This is only guaranteed to be possible because we specified earlier that `min_items` for `addresses` must be `1`. If we didn\'t, then the user would be able to create a user in a newer version that would be impossible to represent in the older version. I.e. If anyone tried to get that user from the older version, they would get a `ResponseValidationError` because the user wouldn\'t have data for a mandatory `address` field. You need to always keep in mind tht API versioning is only for versioning your **API**, your interface. Your versions must still be completely compatible in terms of data. If they are not, then you are versioning your data and you should really go with a separate app instance. Otherwise, your users will have a hard time migrating back and forth between API versions and so many unexpected errors.\n\nSee how we added the `addresses` property? This simple instruction will allow us to use `addresses` even from the old schema, which means that our api route will not need to know anything about versioning. The main goal of universi is to shift the logic of versioning away from your business logic and api endpoints which makes your project easier to navigate and which makes deleting versions a breeze.\n\n### Grouping Version Changes\n\nFinally, we group the version changes in the `VersionBundle` class. This represents the different versions of your API and the changes between them. You can add any "version changes" to any version. For simplicity, let\'s use versions 2002 and 2001 which means that we had a single address in API in 2001 and added addresses as a list in 2002\'s version.\n\n```python\nfrom universi.structure import Version, VersionBundle\nfrom datetime import date\n\nversions = VersionBundle(\n    Version(date(2002, 1, 1), ChangeAddressToList),\n    Version(date(2001, 1, 1)),\n)\n```\n\nThat\'s it. You\'re done with describing things. Now you just gotta ask universi to do the rest for you. We\'ll need the VersionedAPIRouter we used previously, our API versions, and the module representing the latest versions of our schemas.\n\n```python\nfrom versions import latest\nfrom universi import regenerate_dir_to_all_versions, api_version_var\n\nregenerate_dir_to_all_versions(latest, versions)\nrouter_versions = router.create_versioned_copies(\n    versions,\n    latest_schemas_module=latest,\n)\napi_version_var.set(date(2002, 1, 1))\nuvicorn.run(router_versions[date(2002, 1, 1)])\n```\n\nUniversi has generated multiple things in this code:\n\n* Three versions of our schemas: one for each API version and one that includes definitions of unions of all versions for each schema which will be useful when you want to type check that you are using requests of different versions correctly. For example, we\'ll have `UserCreateRequest` defined there which is a `TypeAlias` pointing to the union of 2002 version and 2001 version of `UserCreateRequest`.\n* Two versions of our API router: one for each API version\n\nYou can now just pick a router by its version and run it separately or use a parent router/app to specify the logic by which you\'d like to pick a version. I recommend using a header-based router with version dates as headers. And yes, that\'s how Stripe does it.\n\nNote that universi migrates your response data based on the api_version_var context variable so you must set it with each request. `universi.header` has a dependency that does that for you.\n\nObviously, this was just a simple example and universi has a lot more features so if you\'re interested -- take a look at the reference.\n\n### Examples\n\nPlease, see [tutorial examples](https://github.com/Ovsyanka83/universi/tree/main/tests/test_tutorial) for the fully working version of the project above.\n\n## Important warnings\n\n1. The goal of Universi is to **minimize** the impact of versioning on your business logic. It provides all necessary tools to prevent you from **ever** checking for a concrete version in your code. So please, if you are tempted to check something like `api_version_var.get() >= date(2022, 11, 11)` -- please, take another look into [reference](#version-changes-with-side-effects) section. I am confident that you will find a better solution there.\n2. Universi does not include a header-based router like FastAPI. We hope that soon a framework for header-based routing will surface which will allow universi to be a full versioning solution.\n3. I ask you to be very detailed in your descriptions for version changes. Spending these 5 extra minutes will potentially save you tens of hours in the future when everybody forgets when, how, and why the version change was made.\n4. We migrate responses backwards in versions from the latest version using data migration functions and requests forward in versions until the latest version using properties on pydantic models.\n\n## Reference\n\n### Endpoints\n\n#### Defining endpoints that didn\'t exist in new versions\n\nIf you had an endpoint in old version but do not have it in a new one, you must still define it but mark it as deleted.\n\n```python\n@router.only_exists_in_older_versions\n@router.get("/my_old_endpoint")\nasync def my_old_endpoint():\n    ...\n```\n\nand then define it as existing in one of the older versions:\n\n```python\nfrom universi.structure import VersionChange, endpoint\n\nclass MyChange(VersionChange):\n    description = "..."\n    instructions_to_migrate_to_previous_version = (\n        endpoint("/my_old_endpoint").existed,\n    )\n\n```\n\n#### Defining endpoints that didn\'t exist in old versions\n\nIf you have an endpoint in your new version that must not exist in older versions, you define it as usual and then mark it as "non-existing" in old versions:\n\n```python\nfrom universi.structure import VersionChange, endpoint\n\nclass MyChange(VersionChange):\n    description = "..."\n    instructions_to_migrate_to_previous_version = (\n        endpoint("/my_new_endpoint").didnt_exist,\n    )\n\n```\n\n#### Changing endpoint attributes\n\nIf you want to change any attribute of your endpoint in a new version, you can return the attribute\'s value in all older versions like so:\n\n```python\nfrom universi.structure import VersionChange, endpoint\n\nclass MyChange(VersionChange):\n    description = "..."\n    instructions_to_migrate_to_previous_version = (\n        endpoint("/my_endpoint").had(description="My old description"),\n    )\n\n```\n\n### Enums\n\n#### Adding enum members\n\nNote that adding enum members **can** be a breaking change unlike adding optional fields to a schema. For example, if I return a list of entities, each of which has some type, and I add a new type -- then my client\'s code is likely to break.\n\nSo I suggest adding enum members in new versions as well.\n\n```python\nfrom universi.structure import VersionChange, enum\nfrom enum import auto\n\nclass MyChange(VersionChange):\n    description = "..."\n    instructions_to_migrate_to_previous_version = (\n        enum(my_enum).had(foo="baz", bar=auto()),\n    )\n\n```\n\n#### Removing enum members\n\n```python\nfrom universi.structure import VersionChange, enum\n\nclass MyChange(VersionChange):\n    description = "..."\n    instructions_to_migrate_to_previous_version = (\n        enum(my_endpoint).didnt_have("foo", "bar"),\n    )\n\n```\n\n### Schemas\n\n#### Add a field\n\n```python\nfrom pydantic import Field\nfrom universi.structure import VersionChange, schema\n\nclass MyChange(VersionChange):\n    description = "..."\n    instructions_to_migrate_to_previous_version = (\n        schema(MySchema).field("foo").existed_with(type=list[str], info=Field(description="Foo")),\n    )\n\n```\n\n#### Remove a field\n\n```python\nfrom universi.structure import VersionChange, schema\n\nclass MyChange(VersionChange):\n    description = "..."\n    instructions_to_migrate_to_previous_version = (\n        schema(MySchema).field("foo").didnt_exist,\n    )\n\n```\n\n#### Change a field\n\n```python\nfrom universi.structure import VersionChange, schema\n\nclass MyChange(VersionChange):\n    description = "..."\n    instructions_to_migrate_to_previous_version = (\n        schema(MySchema).field("foo").had(description="Foo"),\n    )\n\n```\n\n#### Add a property\n\n```python\nfrom universi.structure import VersionChange, schema\n\nclass MyChange(VersionChange):\n    description = "..."\n    instructions_to_migrate_to_previous_version = ()\n\n    @schema(MySchema).had_property("foo")\n    def any_name_here(parsed_schema):\n        # Anything can be returned from here\n        return parsed_schema.some_other_field\n\n```\n\n#### Remove a property\n\n```python\nfrom universi.structure import VersionChange, schema\n\nclass MyChange(VersionChange):\n    description = "..."\n    instructions_to_migrate_to_previous_version = (\n        schema(MySchema).property("foo").didnt_exist,\n    )\n\n```\n\n### Unions\n\nAs you probably realize, when you have many versions with different request schemas and your business logic receives one of them -- you\'re in trouble. You could handle them all separately by checking the version of each schema and then using the correct logic for it but universi tries to offer something better.\n\nInstead, we take a union of all of our request schemas and write our business logic as if it receives that union. For example, if version 2000 had field "foo" of type `str` and then version 2001 changed that field to type `int`, then a union of these schemas will have foo as `str | int` so your type checker will protect you against incorrect usage. Same goes for added/deleted fields. Obviously, manually importing all your schemas and then taking a union of them is tough, especially if you have many versions, which is why Universi not only generates a directory for each of your versions, but it also generates a "unions" directory that contains unions of all your schemas and enums.\n\nFor example, if we had a schema named `MySchema` and two versions of it: 2000 and 2001, then the union definition will look like the following:\n\n```python\nfrom ..latest import my_schema_module as latest_my_schema_module\nfrom ..v2000_01_01 import my_schema_module as v2000_01_01_my_schema_module\nfrom ..v2001_01_01 import my_schema_module as v2001_01_01_my_schema_module\n\nMySchema = (\n    latest_my_schema_module.MySchema |\n    v2000_01_01_my_schema_module.MySchema |\n    v2001_01_01_my_schema_module.MySchema\n)\n```\n\nand you would be able to use it like so:\n\n```python\nfrom src.schemas.unions.my_schema_module import MySchema\n\nasync def the_entrypoint_of_my_business_logic(request_payload: MySchema):\n    ...\n\n```\n\nNote that this feature only affects type checking and does not affect your functionality.\n\n### Data conversion\n\nAs described in the tutorial, universi can convert your response data into older versions. It does so by running your "migration" functions whenever it encounters a version change:\n\n```python\nfrom universi.structure import VersionChange, convert_response_to_previous_version_for\nfrom typing import Any\n\nclass ChangeAddressToList(VersionChange):\n    description = "..."\n\n    @convert_response_to_previous_version_for(MyEndpointResponseModel)\n    def change_addresses_to_single_item(cls, data: dict[str, Any]) -> None:\n        data["address"] = data.pop("addresses")[0]\n\n```\n\nIt is done by applying `universi.VersionBundle.versioned(...)` decorator to each endpoint with the given `response_model` which automatically detects the API version by getting it from the [contextvar](#api-version-header-and-context-variables) and applying all version changes until the selected version in reverse. Note that if the version is not set, then no changes will be applied.\n\nIf you want to convert a specific response to a specific version, you can use `universi.VersionBundle.data_to_version(...)`.\n\n### Version changes with side effects\n\nSometimes you will use API versioning to handle a breaking change in your **business logic**, not in the schemas themselves. In such cases, it is tempting to add a version check and just follow the new business logic such as:\n\n```python\nif api_version_var.get() >= date(2022, 11, 11):\n    # do new logic here\n```\n\nIn universi, this approach is highly discouraged. It is recommended that you avoid side effects like this at any cost because each one makes your core logic harder to understand. But if you cannot, then I urge you to at least abstract away versions and versioning from your business logic which will make your code much easier to read.\n\nTo simplify this, universi has a special `VersionChangeWithSideEffects` class. It makes finding dangerous versions that have side effects much easier and provides a nice abstraction for checking whether we are on a version where these side effects have been applied.\n\nAs an example, let\'s use the tutorial section\'s case with the user and their address. Let\'s say that we use an external service to check whether user\'s address is listed in it and return 400 response if it is not. Let\'s also say that we only added this check in the newest version.\n\n```python\nfrom universi.structure import VersionChangeWithSideEffects\n\nclass UserAddressIsCheckedInExternalService(VersionChangeWithSideEffects):\n    description = (\n        "User\'s address is now checked for existense in an external service. "\n        "If it doesn\'t exist there, a 400 code is returned."\n    )\n\n```\n\nThen we will have the following check in our business logic:\n\n```python\nfrom src.versions import versions, UserAddressIsCheckedInExternalService\n\n\nasync def create_user(payload):\n    if UserAddressIsCheckedInExternalService.is_applied:\n        check_user_address_exists_in_an_external_service(payload.address)\n    ...\n```\n\nSo this change can be contained in any version -- your business logic doesn\'t know which version it has and shouldn\'t.\n\n### API Version header and context variables\n\n**Note that this behavior is deprecated. Only use a custom contextvar that you made yourself**\n\nUniversi automatically converts your data to a correct version and has "version checks" when dealing with side effects as described in [the section above](#version-changes-with-side-effects). It can only do so using a special [context variable](https://docs.python.org/3/library/contextvars.html) that stores the current API version.\n\nUniversi has such default variable defined as `universi.api_version_var`. You can also use `universi.get_universi_dependency` to get a `fastapi.Depends` that automatically sets this contextvar based on a header name that you pick.\n\nYou can also set the variable yourself or even pass a different compatible contextvar to your `universi.VersionBundle` constructor.\n\n## Similar projects\n\nThe following projects are trying to accomplish similar results with a lot more simplistic functionality.\n\n* <https://github.com/sjkaliski/pinned>\n* <https://github.com/lukepolo/laravel-api-migrations>\n* <https://github.com/tomschlick/request-migrations>\n',
    'author': 'Stanislav Zmiev',
    'author_email': 'zmievsa@gmail.com',
    'maintainer': 'None',
    'maintainer_email': 'None',
    'url': 'https://github.com/ovsyanka83/universi',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.10,<4.0',
}


setup(**setup_kwargs)
