"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.PolicyStatementWithPrincipal = exports.PrincipalType = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const _5_effect_1 = require("./5-effect");
var PrincipalType;
(function (PrincipalType) {
    PrincipalType["AWS"] = "AWS";
    PrincipalType["FEDERATED"] = "Federated";
    PrincipalType["CANONICAL_USER"] = "CanonicalUser";
    PrincipalType["SERVICE"] = "Service";
})(PrincipalType = exports.PrincipalType || (exports.PrincipalType = {}));
/**
 * Adds "principal" functionality to the Policy Statement
 */
class PolicyStatementWithPrincipal extends _5_effect_1.PolicyStatementWithEffect {
    constructor() {
        super(...arguments);
        this.useNotPrincipal = false;
        this.myPrincipals = {};
    }
    /**
     * Injects principals into the statement.
     *
     * Only relevant for the main package. In CDK mode this only calls super.
     */
    toJSON() {
        // @ts-ignore only available after swapping 1-base
        if (typeof this.addResources == 'function') {
            this.cdkApplyPrincipals();
            return super.toJSON();
        }
        const mode = this.useNotPrincipal ? 'NotPrincipal' : 'Principal';
        const statement = super.toJSON();
        if (this.hasPrincipals()) {
            statement[mode] = this.myPrincipals;
        }
        return statement;
    }
    toStatementJson() {
        this.cdkApplyPrincipals();
        // @ts-ignore only available after swapping 1-base
        return super.toStatementJson();
    }
    freeze() {
        // @ts-ignore only available after swapping 1-base
        if (!this.frozen) {
            this.cdkApplyPrincipals();
        }
        return super.freeze();
    }
    cdkApplyPrincipals() { }
    /**
     * Switches the statement to use [`notPrincipal`](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_notprincipal.html).
     */
    notPrincipal() {
        this.useNotPrincipal = true;
        return this;
    }
    /**
     * Checks weather a principal was applied to the policy
     */
    hasPrincipals() {
        return Object.keys(this.myPrincipals).length > 0;
    }
    /**
     * Adds a principal to the statement
     *
     * @param prefix One of **AWS**, **Federated**, **CanonicalUser** or **Service**
     * @param principal The principal string
     */
    addPrincipal(prefix, principal) {
        this.skipAutoResource = true;
        if (!(prefix in this.myPrincipals)) {
            this.myPrincipals[prefix] = [];
        }
        this.myPrincipals[prefix].push(principal);
        return this;
    }
    /**
     * Adds any principal to the statement
     *
     * @param arn The ARN of the principal
     * @param prefix One of **AWS**, **Federated**, **CanonicalUser** or **Service** - Default: **AWS**
     */
    for(arn, prefix) {
        return this.addPrincipal(prefix || PrincipalType.AWS, arn);
    }
    /**
     * Adds one or more account principals to the statement
     *
     * @param accounts ID of the AWS account
     */
    forAccount(...accounts) {
        accounts.forEach((account) => this.addPrincipal(PrincipalType.AWS, `arn:${PolicyStatementWithPrincipal.defaultPartition}:iam::${account}:root`));
        return this;
    }
    /**
     * Adds one or more [federated](https://aws.amazon.com/identity/federation/) (web identity) principals to the statement
     *
     * @param providers ID of the AWS account
     */
    forFederated(...providers) {
        providers.forEach((provider) => this.addPrincipal(PrincipalType.FEDERATED, provider));
        return this;
    }
    /**
     * Adds a federated [AWS Cognito](https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_providers_oidc_cognito.html) principal to the statement
     */
    forFederatedCognito() {
        return this.forFederated('cognito-identity.amazonaws.com');
    }
    /**
     * Adds a federated [Amazon](https://login.amazon.com/) principal to the statement
     */
    forFederatedAmazon() {
        return this.forFederated('www.amazon.com');
    }
    /**
     * Adds a federated [Facebook](https://developers.facebook.com/docs/facebook-login) principal to the statement
     */
    forFederatedFacebook() {
        return this.forFederated('graph.facebook.com');
    }
    /**
     * Adds a federated [Google](https://developers.google.com/identity/protocols/oauth2/openid-connect) principal to the statement
     */
    forFederatedGoogle() {
        return this.forFederated('accounts.google.com');
    }
    /**
     * Adds one or more canonical user principals to the statement
     *
     * @param userIDs The user ID
     *
     * You can [find the canonical user ID](https://docs.aws.amazon.com/general/latest/gr/acct-identifiers.html#FindingCanonicalId) for your AWS account in the AWS Management Console. The canonical user ID for an AWS account is specific to the account. You can retrieve the canonical user ID for your AWS account as either the root user or an IAM user.
     */
    forCanonicalUser(...userIDs) {
        userIDs.forEach((userID) => this.addPrincipal(PrincipalType.CANONICAL_USER, userID));
        return this;
    }
    /**
     * Adds one or more federated SAML principals to the statement
     *
     * @param account ID of the AWS account
     * @param providerNames Name of the SAML provider
     */
    forSaml(account, ...providerNames) {
        providerNames.forEach((providerName) => this.forFederated(`arn:${PolicyStatementWithPrincipal.defaultPartition}:iam::${account}:saml-provider/${providerName}`));
        return this;
    }
    /**
     * Adds one or more IAM user principals to the statement
     *
     * @param account ID of the AWS account
     * @param users Name of the IAM user
     */
    forUser(account, ...users) {
        users.forEach((user) => this.addPrincipal(PrincipalType.AWS, `arn:${PolicyStatementWithPrincipal.defaultPartition}:iam::${account}:user/${user}`));
        return this;
    }
    /**
     * Adds one or more IAM role principals to the statement
     *
     * @param account ID of the AWS account
     * @param roles Name of the IAM role
     */
    forRole(account, ...roles) {
        roles.forEach((role) => this.addPrincipal(PrincipalType.AWS, `arn:${PolicyStatementWithPrincipal.defaultPartition}:iam::${account}:role/${role}`));
        return this;
    }
    /**
     * Adds one or more specific assumed role session principals to the statement
     *
     * @param account ID of the AWS account
     * @param roleName Name of the IAM role
     * @param sessionNames Name of the session. You cannot use a wildcard (`*`) to mean *all sessions*. Principals must always name a specific session
     */
    forAssumedRoleSession(account, roleName, ...sessionNames) {
        sessionNames.forEach((sessionName) => {
            this.addPrincipal(PrincipalType.AWS, `arn:${PolicyStatementWithPrincipal.defaultPartition}:sts::${account}:assumed-role/${roleName}/${sessionName}`);
        });
        return this;
    }
    /**
     * Adds one or more service principals to the statement
     *
     * @param services Long version of the service name. Usually in the format: `long_service-name.amazonaws.com`
     *
     * The service principal is defined by the service. To learn the service principal for a service, see the documentation for that service. For some services, see [AWS Services That Work with IAM](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_aws-services-that-work-with-iam.html) and look for the services that have **Yes** in the **Service-Linked Role** column. Choose a **Yes** with a link to view the service-linked role documentation for that service. View the **Service-Linked Role Permissions** section for that service to view the service principal.
     */
    forService(...services) {
        services.forEach((service) => this.addPrincipal(PrincipalType.SERVICE, service));
        return this;
    }
    /**
     * Grants public asses
     *
     * **EVERYONE IN THE WORLD HAS ACCESS**
     *
     * We strongly recommend that you do not use a wildcard in the Principal element in a role's trust policy unless you otherwise restrict access through a Condition element in the policy. Otherwise, any IAM user in any account in your partition can access the role.
     */
    forPublic() {
        return this.addPrincipal(PrincipalType.AWS, '*');
    }
}
exports.PolicyStatementWithPrincipal = PolicyStatementWithPrincipal;
_a = JSII_RTTI_SYMBOL_1;
PolicyStatementWithPrincipal[_a] = { fqn: "iam-floyd.PolicyStatementWithPrincipal", version: "0.504.0" };
//# sourceMappingURL=data:application/json;base64,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