"""Check if a host is in the Google Chrome HSTS Preload list"""

import functools
import os
import typing

__version__ = "2020.9.25"
__checksum__ = "df90ce69df19dbac74cef76a5d60c08ce4414d31940a36b34b51aba78054dcd7"
__all__ = ["in_hsts_preload"]

# fmt: off
_GTLD_INCLUDE_SUBDOMAINS = {b'android', b'app', b'bank', b'chrome', b'dev', b'foo', b'gle', b'gmail', b'google', b'hangout', b'insurance', b'meet', b'new', b'page', b'play', b'search', b'youtube'}  # noqa: E501
_JUMPTABLE = [[(0, 11), (11, 5), None, (16, 57), (73, 26), (99, 12), None, (111, 19), (130, 22), (152, 7), (159, 20), (179, 18), None, (197, 22), (219, 45), (264, 7), (271, 9), (280, 36), (316, 10), (326, 10), (336, 21), None, (357, 50), (407, 8), (415, 9), (424, 19), (443, 13), (456, 14), (470, 14), None, None, (484, 29), (513, 16), (529, 35), (564, 14), (578, 24), (602, 9), None, (611, 25), (636, 20), (656, 8), (664, 13), (677, 10), None, (687, 17), (704, 6), (710, 26), (736, 5), (741, 5), (746, 10), (756, 10), (766, 11), (777, 12), (789, 27), None, (816, 11), (827, 11), (838, 7), (845, 29), (874, 18), (892, 27), (919, 46), (965, 25), (990, 16), (1006, 8), (1014, 5), (1019, 22), (1041, 18), None, (1059, 36), (1095, 15), (1110, 8), (1118, 11), None, (1129, 5), (1134, 16), (1150, 14), (1164, 18), None, (1182, 14), (1196, 18), (1214, 48), (1262, 19), (1281, 5), (1286, 46), (1332, 14), (1346, 14), (1360, 20), None, (1380, 10), (1390, 13), (1403, 10), (1413, 19), None, (1432, 13), (1445, 19), (1464, 5), (1469, 4), (1473, 22), (1495, 10), (1505, 7), (1512, 14), (1526, 21), (1547, 11), (1558, 10), (1568, 12), (1580, 32), None, (1612, 10), (1622, 14), (1636, 12), (1648, 45), (1693, 15), None, (1708, 11), (1719, 23), (1742, 21), (1763, 26), (1789, 6), (1795, 6), (1801, 7), (1808, 5), (1813, 20), (1833, 23), (1856, 24), (1880, 13), (1893, 15), (1908, 19), (1927, 6), (1933, 61), (1994, 44), (2038, 12), (2050, 23), (2073, 16), (2089, 38), (2127, 6), (2133, 12), (2145, 44), (2189, 6), (2195, 41), (2236, 13), (2249, 23), (2272, 30), (2302, 16), (2318, 8), (2326, 15), (2341, 12), (2353, 19), (2372, 21), (2393, 15), None, (2408, 35), (2443, 21), (2464, 17), (2481, 19), (2500, 26), (2526, 5), (2531, 37), (2568, 30), (2598, 16), (2614, 10), (2624, 17), (2641, 23), (2664, 14), (2678, 17), (2695, 8), (2703, 4), (2707, 7), (2714, 29), (2743, 6), (2749, 18), (2767, 27), (2794, 20), (2814, 17), (2831, 19), (2850, 12), (2862, 40), (2902, 40), (2942, 12), (2954, 48), (3002, 25), (3027, 12), None, (3039, 8), (3047, 20), (3067, 19), (3086, 6), (3092, 23), None, (3115, 30), (3145, 33), (3178, 14), (3192, 12), (3204, 27), None, (3231, 26), (3257, 41), (3298, 50), (3348, 15), (3363, 20), (3383, 15), (3398, 21), (3419, 32), (3451, 24), (3475, 20), (3495, 13), (3508, 60), (3568, 19), (3587, 9), (3596, 12), (3608, 12), (3620, 11), (3631, 10), (3641, 48), (3689, 32), None, (3721, 25), (3746, 12), None, (3758, 8), (3766, 8), (3774, 7), None, (3781, 25), (3806, 17), None, (3823, 21), (3844, 35), (3879, 12), (3891, 10), (3901, 36), (3937, 20), (3957, 22), (3979, 23), (4002, 19), (4021, 12), (4033, 5), (4038, 30), (4068, 24), (4092, 14), (4106, 14), (4120, 47), (4167, 46), None, None, (4213, 51), (4264, 42), None, (4306, 14), None, (4320, 15), (4335, 8), (4343, 21), (4364, 6), (4370, 16), (4386, 17)], [(4403, 6359), (10762, 6807), (17569, 7188), (24757, 6145), (30902, 6411), (37313, 6210), (43523, 7049), (50572, 6407), (56979, 6943), (63922, 6167), (70089, 7230), (77319, 6610), (83929, 6862), (90791, 7275), (98066, 6662), (104728, 6870), (111598, 7219), (118817, 6010), (124827, 6375), (131202, 6646), (137848, 7037), (144885, 6668), (151553, 7019), (158572, 6218), (164790, 6517), (171307, 6737), (178044, 6733), (184777, 7057), (191834, 6365), (198199, 6855), (205054, 6938), (211992, 6603), (218595, 6610), (225205, 7113), (232318, 6293), (238611, 6929), (245540, 6296), (251836, 7192), (259028, 6877), (265905, 7005), (272910, 7676), (280586, 6484), (287070, 6428), (293498, 6264), (299762, 6410), (306172, 6285), (312457, 6477), (318934, 7151), (326085, 6544), (332629, 5999), (338628, 6505), (345133, 6747), (351880, 6712), (358592, 6873), (365465, 6983), (372448, 6921), (379369, 6947), (386316, 6099), (392415, 7047), (399462, 5945), (405407, 6745), (412152, 6549), (418701, 6434), (425135, 6966), (432101, 6739), (438840, 6700), (445540, 6365), (451905, 7227), (459132, 6903), (466035, 6873), (472908, 6593), (479501, 6629), (486130, 5783), (491913, 7156), (499069, 7193), (506262, 7188), (513450, 6180), (519630, 7337), (526967, 7127), (534094, 6146), (540240, 6877), (547117, 5847), (552964, 6510), (559474, 6770), (566244, 6540), (572784, 6483), (579267, 6740), (586007, 6718), (592725, 6923), (599648, 6807), (606455, 7384), (613839, 6108), (619947, 6386), (626333, 6646), (632979, 6586), (639565, 7217), (646782, 7106), (653888, 6520), (660408, 6303), (666711, 6257), (672968, 6360), (679328, 6834), (686162, 6332), (692494, 6608), (699102, 6307), (705409, 6945), (712354, 6794), (719148, 7178), (726326, 8119), (734445, 7193), (741638, 7110), (748748, 6632), (755380, 6448), (761828, 6786), (768614, 7006), (775620, 6818), (782438, 6338), (788776, 6489), (795265, 6432), (801697, 7188), (808885, 6977), (815862, 7074), (822936, 7095), (830031, 7074), (837105, 7925), (845030, 6516), (851546, 5945), (857491, 7048), (864539, 6679), (871218, 8194), (879412, 7194), (886606, 6194), (892800, 6957), (899757, 6965), (906722, 6396), (913118, 6856), (919974, 6273), (926247, 6845), (933092, 6571), (939663, 6617), (946280, 6666), (952946, 7383), (960329, 6395), (966724, 6354), (973078, 6697), (979775, 6634), (986409, 6671), (993080, 6915), (999995, 6422), (1006417, 7282), (1013699, 6724), (1020423, 6866), (1027289, 7017), (1034306, 6541), (1040847, 6760), (1047607, 6718), (1054325, 6529), (1060854, 6621), (1067475, 6394), (1073869, 6136), (1080005, 6389), (1086394, 6766), (1093160, 7339), (1100499, 6209), (1106708, 6701), (1113409, 7028), (1120437, 6449), (1126886, 6350), (1133236, 7070), (1140306, 6640), (1146946, 6051), (1152997, 6571), (1159568, 7824), (1167392, 6248), (1173640, 6419), (1180059, 6924), (1186983, 6496), (1193479, 6819), (1200298, 6527), (1206825, 6185), (1213010, 7583), (1220593, 6894), (1227487, 6608), (1234095, 7114), (1241209, 7523), (1248732, 7434), (1256166, 6265), (1262431, 7127), (1269558, 6436), (1275994, 6680), (1282674, 6909), (1289583, 6334), (1295917, 7016), (1302933, 7195), (1310128, 6692), (1316820, 6791), (1323611, 6581), (1330192, 6601), (1336793, 6845), (1343638, 6399), (1350037, 6828), (1356865, 6171), (1363036, 7148), (1370184, 6961), (1377145, 6688), (1383833, 7120), (1390953, 5912), (1396865, 6801), (1403666, 6611), (1410277, 6925), (1417202, 6923), (1424125, 7161), (1431286, 6772), (1438058, 6988), (1445046, 6961), (1452007, 6505), (1458512, 6680), (1465192, 6686), (1471878, 6685), (1478563, 6559), (1485122, 6622), (1491744, 6102), (1497846, 7617), (1505463, 6774), (1512237, 6381), (1518618, 6695), (1525313, 6859), (1532172, 6031), (1538203, 6909), (1545112, 6666), (1551778, 7561), (1559339, 6587), (1565926, 6112), (1572038, 7117), (1579155, 6496), (1585651, 7303), (1592954, 6366), (1599320, 6383), (1605703, 5896), (1611599, 6782), (1618381, 6630), (1625011, 6942), (1631953, 6457), (1638410, 6652), (1645062, 6595), (1651657, 7203), (1658860, 6514), (1665374, 5966), (1671340, 6678), (1678018, 6394), (1684412, 6845), (1691257, 7027), (1698284, 7174), (1705458, 6356), (1711814, 6369), (1718183, 6806)], [(1724989, 736), (1725725, 625), (1726350, 665), (1727015, 757), (1727772, 553), (1728325, 670), (1728995, 671), (1729666, 836), (1730502, 640), (1731142, 667), (1731809, 536), (1732345, 574), (1732919, 792), (1733711, 853), (1734564, 1020), (1735584, 850), (1736434, 1242), (1737676, 653), (1738329, 875), (1739204, 696), (1739900, 757), (1740657, 746), (1741403, 853), (1742256, 731), (1742987, 703), (1743690, 715), (1744405, 955), (1745360, 1131), (1746491, 807), (1747298, 734), (1748032, 922), (1748954, 787), (1749741, 586), (1750327, 746), (1751073, 748), (1751821, 800), (1752621, 659), (1753280, 746), (1754026, 724), (1754750, 1119), (1755869, 695), (1756564, 812), (1757376, 727), (1758103, 719), (1758822, 728), (1759550, 396), (1759946, 942), (1760888, 870), (1761758, 735), (1762493, 592), (1763085, 834), (1763919, 692), (1764611, 780), (1765391, 1012), (1766403, 944), (1767347, 558), (1767905, 661), (1768566, 550), (1769116, 578), (1769694, 771), (1770465, 772), (1771237, 776), (1772013, 1054), (1773067, 915), (1773982, 725), (1774707, 719), (1775426, 767), (1776193, 494), (1776687, 595), (1777282, 592), (1777874, 692), (1778566, 877), (1779443, 596), (1780039, 725), (1780764, 650), (1781414, 717), (1782131, 573), (1782704, 711), (1783415, 787), (1784202, 496), (1784698, 789), (1785487, 629), (1786116, 828), (1786944, 646), (1787590, 603), (1788193, 425), (1788618, 597), (1789215, 746), (1789961, 804), (1790765, 780), (1791545, 853), (1792398, 1092), (1793490, 839), (1794329, 856), (1795185, 725), (1795910, 436), (1796346, 955), (1797301, 857), (1798158, 580), (1798738, 671), (1799409, 728), (1800137, 885), (1801022, 887), (1801909, 571), (1802480, 632), (1803112, 772), (1803884, 419), (1804303, 466), (1804769, 924), (1805693, 940), (1806633, 831), (1807464, 793), (1808257, 654), (1808911, 792), (1809703, 672), (1810375, 699), (1811074, 709), (1811783, 485), (1812268, 722), (1812990, 710), (1813700, 914), (1814614, 679), (1815293, 804), (1816097, 483), (1816580, 691), (1817271, 780), (1818051, 835), (1818886, 926), (1819812, 781), (1820593, 942), (1821535, 802), (1822337, 524), (1822861, 815), (1823676, 630), (1824306, 803), (1825109, 779), (1825888, 717), (1826605, 681), (1827286, 635), (1827921, 673), (1828594, 621), (1829215, 674), (1829889, 694), (1830583, 648), (1831231, 494), (1831725, 603), (1832328, 661), (1832989, 597), (1833586, 717), (1834303, 594), (1834897, 773), (1835670, 514), (1836184, 523), (1836707, 701), (1837408, 628), (1838036, 657), (1838693, 639), (1839332, 870), (1840202, 610), (1840812, 622), (1841434, 903), (1842337, 870), (1843207, 583), (1843790, 695), (1844485, 873), (1845358, 659), (1846017, 692), (1846709, 455), (1847164, 609), (1847773, 684), (1848457, 788), (1849245, 598), (1849843, 932), (1850775, 760), (1851535, 807), (1852342, 721), (1853063, 668), (1853731, 609), (1854340, 664), (1855004, 706), (1855710, 1394), (1857104, 575), (1857679, 662), (1858341, 650), (1858991, 1027), (1860018, 800), (1860818, 798), (1861616, 546), (1862162, 609), (1862771, 823), (1863594, 601), (1864195, 589), (1864784, 828), (1865612, 678), (1866290, 911), (1867201, 810), (1868011, 764), (1868775, 710), (1869485, 868), (1870353, 637), (1870990, 909), (1871899, 662), (1872561, 819), (1873380, 630), (1874010, 742), (1874752, 483), (1875235, 828), (1876063, 823), (1876886, 665), (1877551, 916), (1878467, 787), (1879254, 868), (1880122, 954), (1881076, 1077), (1882153, 887), (1883040, 669), (1883709, 895), (1884604, 731), (1885335, 533), (1885868, 443), (1886311, 794), (1887105, 823), (1887928, 455), (1888383, 1007), (1889390, 513), (1889903, 778), (1890681, 881), (1891562, 792), (1892354, 806), (1893160, 696), (1893856, 806), (1894662, 751), (1895413, 784), (1896197, 607), (1896804, 579), (1897383, 422), (1897805, 666), (1898471, 484), (1898955, 816), (1899771, 880), (1900651, 764), (1901415, 718), (1902133, 658), (1902791, 604), (1903395, 848), (1904243, 495), (1904738, 606), (1905344, 780), (1906124, 494), (1906618, 870), (1907488, 2090), (1909578, 548), (1910126, 707), (1910833, 890), (1911723, 912), (1912635, 522)], [(1913157, 48), None, (1913205, 35), (1913240, 42), None, None, None, None, None, None, None, None, None, None, None, None, None, (1913282, 42), None, (1913324, 25), (1913349, 44), (1913393, 22), (1913415, 18), None, None, None, None, (1913433, 26), None, None, None, None, (1913459, 21), (1913480, 25), None, None, (1913505, 26), None, None, None, None, (1913531, 44), (1913575, 21), (1913596, 23), None, None, None, None, (1913619, 48), None, None, None, None, None, (1913667, 31), None, None, None, None, (1913698, 42), None, (1913740, 22), None, (1913762, 21), None, (1913783, 26), (1913809, 42), None, None, (1913851, 77), None, None, None, None, None, (1913928, 21), (1913949, 21), None, None, (1913970, 34), (1914004, 42), None, None, None, (1914046, 25), None, None, (1914071, 21), None, None, None, None, None, (1914092, 24), (1914116, 21), None, None, (1914137, 26), None, (1914163, 18), None, (1914181, 54), None, None, None, None, None, None, (1914235, 26), None, (1914261, 19), None, (1914280, 20), None, None, (1914300, 42), (1914342, 42), (1914384, 17), None, (1914401, 26), None, (1914427, 26), None, None, None, (1914453, 26), (1914479, 20), (1914499, 26), None, (1914525, 42), (1914567, 63), None, None, None, (1914630, 40), (1914670, 48), None, None, None, (1914718, 47), None, None, None, None, None, None, None, (1914765, 42), None, (1914807, 55), None, (1914862, 9), None, (1914871, 21), (1914892, 42), None, None, (1914934, 42), (1914976, 82), None, None, (1915058, 42), None, None, None, None, None, None, None, None, None, (1915100, 42), (1915142, 21), (1915163, 21), None, (1915184, 42), (1915226, 25), None, None, (1915251, 21), (1915272, 42), None, None, (1915314, 21), (1915335, 19), (1915354, 26), None, None, None, (1915380, 21), None, None, (1915401, 38), None, (1915439, 22), (1915461, 21), (1915482, 21), None, None, (1915503, 63), None, (1915566, 21), (1915587, 42), None, (1915629, 17), None, None, None, None, (1915646, 21), (1915667, 21), None, None, (1915688, 21), None, None, (1915709, 21), None, (1915730, 26), None, (1915756, 50), None, None, None, (1915806, 50), (1915856, 26), (1915882, 21), (1915903, 21), (1915924, 19), None, (1915943, 35), (1915978, 26), (1916004, 23), (1916027, 21), (1916048, 42), None, None, None, None, None, None, (1916090, 21), None, None, None, (1916111, 21), None, None, (1916132, 90), None, (1916222, 239), (1916461, 38), None, None, None, None]]  # noqa: E501
_CRC8_TABLE = [
    0x00, 0x07, 0x0e, 0x09, 0x1c, 0x1b, 0x12, 0x15,
    0x38, 0x3f, 0x36, 0x31, 0x24, 0x23, 0x2a, 0x2d,
    0x70, 0x77, 0x7e, 0x79, 0x6c, 0x6b, 0x62, 0x65,
    0x48, 0x4f, 0x46, 0x41, 0x54, 0x53, 0x5a, 0x5d,
    0xe0, 0xe7, 0xee, 0xe9, 0xfc, 0xfb, 0xf2, 0xf5,
    0xd8, 0xdf, 0xd6, 0xd1, 0xc4, 0xc3, 0xca, 0xcd,
    0x90, 0x97, 0x9e, 0x99, 0x8c, 0x8b, 0x82, 0x85,
    0xa8, 0xaf, 0xa6, 0xa1, 0xb4, 0xb3, 0xba, 0xbd,
    0xc7, 0xc0, 0xc9, 0xce, 0xdb, 0xdc, 0xd5, 0xd2,
    0xff, 0xf8, 0xf1, 0xf6, 0xe3, 0xe4, 0xed, 0xea,
    0xb7, 0xb0, 0xb9, 0xbe, 0xab, 0xac, 0xa5, 0xa2,
    0x8f, 0x88, 0x81, 0x86, 0x93, 0x94, 0x9d, 0x9a,
    0x27, 0x20, 0x29, 0x2e, 0x3b, 0x3c, 0x35, 0x32,
    0x1f, 0x18, 0x11, 0x16, 0x03, 0x04, 0x0d, 0x0a,
    0x57, 0x50, 0x59, 0x5e, 0x4b, 0x4c, 0x45, 0x42,
    0x6f, 0x68, 0x61, 0x66, 0x73, 0x74, 0x7d, 0x7a,
    0x89, 0x8e, 0x87, 0x80, 0x95, 0x92, 0x9b, 0x9c,
    0xb1, 0xb6, 0xbf, 0xb8, 0xad, 0xaa, 0xa3, 0xa4,
    0xf9, 0xfe, 0xf7, 0xf0, 0xe5, 0xe2, 0xeb, 0xec,
    0xc1, 0xc6, 0xcf, 0xc8, 0xdd, 0xda, 0xd3, 0xd4,
    0x69, 0x6e, 0x67, 0x60, 0x75, 0x72, 0x7b, 0x7c,
    0x51, 0x56, 0x5f, 0x58, 0x4d, 0x4a, 0x43, 0x44,
    0x19, 0x1e, 0x17, 0x10, 0x05, 0x02, 0x0b, 0x0c,
    0x21, 0x26, 0x2f, 0x28, 0x3d, 0x3a, 0x33, 0x34,
    0x4e, 0x49, 0x40, 0x47, 0x52, 0x55, 0x5c, 0x5b,
    0x76, 0x71, 0x78, 0x7f, 0x6a, 0x6d, 0x64, 0x63,
    0x3e, 0x39, 0x30, 0x37, 0x22, 0x25, 0x2c, 0x2b,
    0x06, 0x01, 0x08, 0x0f, 0x1a, 0x1d, 0x14, 0x13,
    0xae, 0xa9, 0xa0, 0xa7, 0xb2, 0xb5, 0xbc, 0xbb,
    0x96, 0x91, 0x98, 0x9f, 0x8a, 0x8d, 0x84, 0x83,
    0xde, 0xd9, 0xd0, 0xd7, 0xc2, 0xc5, 0xcc, 0xcb,
    0xe6, 0xe1, 0xe8, 0xef, 0xfa, 0xfd, 0xf4, 0xf3
]
# fmt: on

_IS_LEAF = 0x80
_INCLUDE_SUBDOMAINS = 0x40


try:
    from importlib.resources import open_binary

    def open_pkg_binary(path: str) -> typing.BinaryIO:
        return open_binary("hstspreload", path)


except ImportError:

    def open_pkg_binary(path: str) -> typing.BinaryIO:
        return open(
            os.path.join(os.path.dirname(os.path.abspath(__file__)), path),
            "rb",
        )


@functools.lru_cache(maxsize=1024)
def in_hsts_preload(host: typing.AnyStr) -> bool:
    """Determines if an IDNA-encoded host is on the HSTS preload list"""

    if isinstance(host, str):
        host = host.encode("ascii")
    labels = host.lower().split(b".")

    # Fast-branch for gTLDs that are registered to preload all sub-domains.
    if labels[-1] in _GTLD_INCLUDE_SUBDOMAINS:
        return True

    with open_pkg_binary("hstspreload.bin") as f:
        for layer, label in enumerate(labels[::-1]):
            # None of our layers are greater than 4 deep.
            if layer > 3:
                return False

            # Read the jump table for the layer and label
            jump_info = _JUMPTABLE[layer][_crc8(label)]
            if jump_info is None:
                # No entry: host is not preloaded
                return False

            # Read the set of entries for that layer and label
            f.seek(jump_info[0])
            data = bytearray(jump_info[1])
            f.readinto(data)

            for is_leaf, include_subdomains, ent_label in _iter_entries(data):
                # We found a potential leaf
                if is_leaf:
                    if ent_label == host:
                        return True
                    if include_subdomains and host.endswith(b"." + ent_label):
                        return True

                # Continue traversing as we're not at a leaf.
                elif label == ent_label:
                    break
            else:
                return False
    return False


def _iter_entries(data: bytes) -> typing.Iterable[typing.Tuple[int, int, bytes]]:
    while data:
        flags = data[0]
        size = data[1]
        label = bytes(data[2 : 2 + size])
        yield (flags & _IS_LEAF, flags & _INCLUDE_SUBDOMAINS, label)
        data = data[2 + size :]


def _crc8(value: bytes) -> int:
    # CRC8 reference implementation: https://github.com/niccokunzmann/crc8
    checksum = 0x00
    for byte in value:
        checksum = _CRC8_TABLE[checksum ^ byte]
    return checksum
