import { ICertificate } from "aws-cdk-lib/aws-certificatemanager";
import { AddBehaviorOptions, Distribution, ErrorResponse } from "aws-cdk-lib/aws-cloudfront";
import { Bucket, IBucket } from "aws-cdk-lib/aws-s3";
import { BucketDeployment } from "aws-cdk-lib/aws-s3-deployment";
import { Construct } from "constructs";
/**
 * Options for configuring the default origin behavior.
 */
export interface OriginBehaviourOptions extends AddBehaviorOptions {
}
/**
 * Configuration related to using custom domain names/certificates in Cloudfront.
 */
export interface CloudfrontDomainOptions {
    /**
     * A certificate to associate with the distribution. The certificate must be located in N. Virginia (us-east-1).
     *
     * @default - the CloudFront wildcard certificate (*.cloudfront.net) will be used.
     */
    readonly certificate: ICertificate;
    /**
     * Alternative domain names for this distribution.
     *
     * If you want to use your own domain name, such as www.example.com, instead of the cloudfront.net domain name,
     * you can add an alternate domain name to your distribution. If you attach a certificate to the distribution,
     * you must add (at least one of) the domain names of the certificate to this list.
     *
     * @default - The distribution will only support the default generated name (e.g., d111111abcdef8.cloudfront.net)
     */
    readonly domainNames: string[];
}
/**
 * Configuration related to Cloudfront Logging
 */
export interface CloudfrontLoggingOptions {
    /**
     * Enable access logging for the distribution.
     *
     * @default - false, unless `logBucket` is specified.
     */
    readonly enableLogging?: boolean;
    /**
     * The Amazon S3 bucket to store the access logs in.
     *
     * @default - A bucket is created if `enableLogging` is true
     */
    readonly logBucket?: IBucket;
    /**
     * Specifies whether you want CloudFront to include cookies in access logs
     *
     * @default false
     */
    readonly logIncludesCookies?: boolean;
    /**
     * An optional string that you want CloudFront to prefix to the access log filenames for this distribution.
     *
     * @default - no prefix
     */
    readonly logFilePrefix?: string;
}
/**
 * Dynamic configuration which gets resolved only during deployment.
 *
 * @example
 *
 * // Will store a JSON file called runtime-config.json in the root of the StaticWebsite S3 bucket containing any
 * // and all resolved values.
 * const runtimeConfig = {jsonPayload: {bucketArn: s3Bucket.bucketArn}};
 * new StaticWebsite(scope, 'StaticWebsite', {websiteContentPath: 'path/to/website', runtimeConfig});
 */
export interface RuntimeOptions {
    /**
     * File name to store runtime configuration (jsonPayload).
     *
     * Must follow pattern: '*.json'
     *
     * @default "runtime-config.json"
     */
    readonly jsonFileName?: string;
    /**
     * Arbitrary JSON payload containing runtime values to deploy. Typically this contains resourceArns, etc which
     * are only known at deploy time.
     *
     * @example { userPoolId: some.userPool.userPoolId, someResourceArn: some.resource.Arn }
     */
    readonly jsonPayload: any;
}
/**
 * Properties for configuring the StaticWebsite.
 */
export interface StaticWebsiteProps {
    /**
     * Path to the directory containing the static website files and assets. This directory must contain an index.html file.
     */
    readonly websiteContentPath: string;
    /**
     * How CloudFront should handle requests that are not successful (e.g., PageNotFound).
     *
     * @default - [{httpStatus: 404,responseHttpStatus: 200,responsePagePath: '/index.html'}]
     */
    readonly errorResponses?: ErrorResponse[];
    /**
     * Dynamic configuration which gets resolved only during deployment.
     */
    readonly runtimeOptions?: RuntimeOptions;
    /**
     * Configuration related to using custom domain names/certificates.
     */
    readonly domainOptions?: CloudfrontDomainOptions;
    /**
     * Configuration related to Cloudfront Logging.
     */
    readonly loggingOptions?: CloudfrontLoggingOptions;
    /**
     * Options for configuring the default origin behavior.
     */
    readonly originBehaviourOptions?: OriginBehaviourOptions;
}
/**
 * Deploys a Static Website using a private S3 bucket as an origin and Cloudfront as the entrypoint.
 *
 * This construct configures a webAcl containing rules that are generally applicable to web applications. This
 * provides protection against exploitation of a wide range of vulnerabilities, including some of the high risk
 * and commonly occurring vulnerabilities described in OWASP publications such as OWASP Top 10.
 *
 */
export declare class StaticWebsite extends Construct {
    readonly websiteBucket: Bucket;
    readonly cloudFrontDistribution: Distribution;
    readonly bucketDeployment: BucketDeployment;
    constructor(scope: Construct, id: string, props: StaticWebsiteProps);
    private validateProps;
    private validateRuntimeConfig;
}
//# sourceMappingURL=static-website.d.ts.map