"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.StaticWebsite = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_cloudfront_1 = require("aws-cdk-lib/aws-cloudfront");
const aws_cloudfront_origins_1 = require("aws-cdk-lib/aws-cloudfront-origins");
const aws_s3_1 = require("aws-cdk-lib/aws-s3");
const aws_s3_deployment_1 = require("aws-cdk-lib/aws-s3-deployment");
const constructs_1 = require("constructs");
const cloudfront_web_acl_1 = require("./cloudfront-web-acl");
const DEFAULT_RUNTIME_CONFIG_FILENAME = "runtime-config.json";
/**
 * Deploys a Static Website using a private S3 bucket as an origin and Cloudfront as the entrypoint.
 *
 * This construct configures a webAcl containing rules that are generally applicable to web applications. This
 * provides protection against exploitation of a wide range of vulnerabilities, including some of the high risk
 * and commonly occurring vulnerabilities described in OWASP publications such as OWASP Top 10.
 *
 */
class StaticWebsite extends constructs_1.Construct {
    constructor(scope, id, props) {
        var _b, _c, _d, _e, _f, _g, _h, _j;
        super(scope, id);
        this.validateProps = (props) => {
            props.runtimeOptions && this.validateRuntimeConfig(props.runtimeOptions);
        };
        this.validateRuntimeConfig = (config) => {
            if (!config) {
                throw new Error(`validateRuntimeConfig onlt accepts non-null RuntimeOptions.`);
            }
            if (config.jsonFileName && !config.jsonFileName.endsWith(".json")) {
                throw new Error(`RuntimeOptions.jsonFileName must be a json file.`);
            }
            try {
                JSON.parse(config.jsonPayload);
            }
            catch (e) {
                throw new Error(`Invalid jsonPayload ${e}`);
            }
        };
        this.validateProps(props);
        // S3 Bucket to hold website files
        this.websiteBucket = new aws_s3_1.Bucket(this, "WebsiteBucket", {
            websiteIndexDocument: "index.html",
            versioned: true,
            enforceSSL: true,
            autoDeleteObjects: true,
            removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
        });
        // Web ACL
        const webAcl = new cloudfront_web_acl_1.CloudfrontWebAcl(this, "WebsiteAcl", {
            managedRules: [{ vendor: "AWS", name: "AWSManagedRulesCommonRuleSet" }],
        });
        // Cloudfront Distribution
        this.cloudFrontDistribution = new aws_cloudfront_1.Distribution(this, "CloudfrontDistribution", {
            webAclId: webAcl.webAclArn,
            defaultBehavior: {
                ...props.originBehaviourOptions,
                origin: new aws_cloudfront_origins_1.S3Origin(this.websiteBucket),
                viewerProtocolPolicy: aws_cloudfront_1.ViewerProtocolPolicy.REDIRECT_TO_HTTPS,
            },
            certificate: (_b = props.domainOptions) === null || _b === void 0 ? void 0 : _b.certificate,
            domainNames: (_c = props.domainOptions) === null || _c === void 0 ? void 0 : _c.domainNames,
            errorResponses: [
                {
                    httpStatus: 404,
                    responseHttpStatus: 200,
                    responsePagePath: "/index.html",
                },
            ],
            enableLogging: (_d = props.loggingOptions) === null || _d === void 0 ? void 0 : _d.enableLogging,
            logBucket: (_e = props.loggingOptions) === null || _e === void 0 ? void 0 : _e.logBucket,
            logIncludesCookies: (_f = props.loggingOptions) === null || _f === void 0 ? void 0 : _f.logIncludesCookies,
            logFilePrefix: (_g = props.loggingOptions) === null || _g === void 0 ? void 0 : _g.logFilePrefix,
        });
        // Deploy Website
        this.bucketDeployment = new aws_s3_deployment_1.BucketDeployment(this, "WebsiteDeployment", {
            sources: [
                aws_s3_deployment_1.Source.asset(props.websiteContentPath),
                ...(props.runtimeOptions
                    ? [
                        aws_s3_deployment_1.Source.jsonData(((_h = props.runtimeOptions) === null || _h === void 0 ? void 0 : _h.jsonFileName) ||
                            DEFAULT_RUNTIME_CONFIG_FILENAME, (_j = props.runtimeOptions) === null || _j === void 0 ? void 0 : _j.jsonPayload),
                    ]
                    : []),
            ],
            destinationBucket: this.websiteBucket,
            // Files in the distribution's edge caches will be invalidated after files are uploaded to the destination bucket.
            distribution: this.cloudFrontDistribution,
            serverSideEncryption: aws_s3_deployment_1.ServerSideEncryption.AES_256,
        });
    }
}
exports.StaticWebsite = StaticWebsite;
_a = JSII_RTTI_SYMBOL_1;
StaticWebsite[_a] = { fqn: "@aws-prototyping-sdk/static-website.StaticWebsite", version: "0.0.73" };
//# sourceMappingURL=data:application/json;base64,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