"""
A PyDev is the "one ring to rule them all"; it's represented by a `pyproject.toml` file that links most/all of the
python projects in a repository.

It has some additional features and tricks that do not apply to other projects. This is centralized here.
"""

from pathlib import Path
from typing import Generator, Tuple, Set

from coveo_functools.casing import flexfactory
from coveo_itertools.lookups import dict_lookup
from coveo_stew.utils import load_toml_from_path
from coveo_systools.filesystem import find_repo_root, safe_text_write
import tomlkit
from tomlkit.items import item as toml_item, Item as TomlItem

from .exceptions import PythonProjectException
from .stew import PythonProject, StewPackage


class NotPyDevProject(PythonProjectException):
    ...


class PyDev(PythonProject):
    """Represents a python development project that ties all other projects together."""

    def __init__(self, path: Path = None, *, verbose: bool = False) -> None:
        super().__init__(path or find_repo_root(default="."), verbose=verbose)
        if not self.options.pydev:
            raise NotPyDevProject(f"{self.project_path}: Not a PyDev project.")

    def pull_dev_requirements(self, *, dry_run: bool = False) -> bool:
        """Pulls the dev requirement from dependencies into pydev's dev requirements."""
        # prepare a toml container for our data
        toml = tomlkit.loads(self.toml_path.read_text())
        all_dev_dependencies = tomlkit.table()

        # extract the dev requirements from the local dependencies
        for item in sorted(self._dev_dependencies_of_dependencies()):
            all_dev_dependencies.add(*item)

        # the py dev environment package has no code, no tests, no entrypoints, no nothin'!
        # as such, dev dependencies are irrelevant; we reserve the section for the current purpose.
        all_dev_dependencies.comment(
            "pydev projects' dev-dependencies are autogenerated; do not edit manually!"
        )
        toml["tool"]["poetry"]["dev-dependencies"] = all_dev_dependencies

        if safe_text_write(
            self.toml_path,
            "\n".join(self._format_toml(tomlkit.dumps(toml))),
            only_if_changed=True,
            dry_run=dry_run,
        ):
            if not dry_run and self.lock_if_needed():
                self.install()
            return True
        return False

    @staticmethod
    def _format_toml(toml_content: str) -> Generator[str, None, None]:
        """tomlkit sometimes forgets to add empty lines before sections."""
        first = True
        for line in toml_content.split("\n"):
            if not first and line.strip().startswith("["):
                yield "\n"  # extra empty line between each section means 2 empty lines between sections
            if line.strip():
                yield line
            first = False

    def _dev_dependencies_of_dependencies(self) -> Generator[Tuple[str, TomlItem], None, None]:
        """Yields the dev dependencies of this project's dependencies."""
        # we mark our direct dependencies as seen, so that we don't duplicate them in the dev section.
        seen: Set[str] = set(self.package.dependencies)
        # we only care about local non-dev dependencies from the project.
        for dependency in filter(lambda _: _.is_local, self.package.dependencies.values()):
            assert not dependency.path.is_absolute()
            project = PythonProject(self.project_path / dependency.path, verbose=self.verbose)
            new = set(project.package.dev_dependencies).difference(seen)
            seen.update(new)
            for dev_dependency in (project.package.dev_dependencies[_] for _ in new):
                if dev_dependency.is_local:
                    value = tomlkit.inline_table()
                    value.append("path", str(dev_dependency.path.relative_to(self.project_path)))
                else:
                    value = dev_dependency.version
                yield dev_dependency.name, toml_item(value)

    @classmethod
    def find_pyproject_paths(cls, path: Path) -> Generator[Path, None, None]:
        """Only return the paths that contain PyDev projects."""

        def _is_pydev(_path: Path) -> bool:
            tool_stew_config = dict_lookup(load_toml_from_path(_path), "tool", "stew", default={})
            return flexfactory(StewPackage, **tool_stew_config).pydev

        yield from filter(_is_pydev, super().find_pyproject_paths(path))
