
from PySide2.QtGui import QTextDocument
from PySide2.QtWidgets import QPlainTextDocumentLayout

from angr.analyses.decompiler.structured_codegen import CConstant, CVariable, CFunctionCall, StructuredCodeGenerator, \
    CStructField

from ...config import Conf


class QCodeDocument(QTextDocument):
    """
    QTextDocument class for pseudocode generated by StructuredCodeGenerator analysis in angr.
    """
    def __init__(self, codegen):
        super().__init__()

        self._codegen = codegen  # type: StructuredCodeGenerator

        # default font
        self.setDefaultFont(Conf.code_font)

        self.setPlainText(self._codegen.text)
        self.setDocumentLayout(QPlainTextDocumentLayout(self))

    @property
    def posmap(self):
        """

        :return:
        :rtype:     Optional[PositionMapping]
        """
        if self._codegen is None:
            return None
        return self._codegen.posmap

    @property
    def nodemap(self):
        if self._codegen is None:
            return None
        return self._codegen.nodemap

    def get_node_at_position(self, pos):

        if self._codegen is not None and self._codegen.posmap is not None:
            n = self._codegen.posmap.get_node(pos)
            if n is None:
                n = self._codegen.posmap.get_node(pos - 1)
            return n

        return None

    def find_related_text_chunks(self, node):

        if self._codegen is None or self._codegen.nodemap is None:
            return None

        if isinstance(node, CConstant):
            starts = self._codegen.nodemap.get(node.value, None)
            if starts is None:
                return [ ]

        elif isinstance(node, CVariable):
            starts = self._codegen.nodemap.get(node.variable, None)
            if starts is None:
                return [ ]

        elif isinstance(node, CFunctionCall):
            starts = self._codegen.nodemap.get(node.callee_func if node.callee_func is not None else node.callee_target,
                                               None)
            if starts is None:
                return [ ]

        elif isinstance(node, CStructField):
            key = (node.struct_type, node.offset)
            starts = self._codegen.nodemap.get(key, None)

            if starts is None:
                return [ ]

        else:
            # Unsupported
            return [ ]

        chunks = [ ]
        for start in starts:
            elem = self._codegen.posmap.get_element(start)
            if elem is None:
                continue
            chunks.append((elem.start, elem.length + elem.start))
        return chunks
